/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.builder;

import static org.mule.metadata.api.model.MetadataFormat.JAVA;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED;
import static org.mule.runtime.ast.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.AllureConstants.ArtifactAst.AstTraversal.PARAMETER_AST;

import static java.util.Collections.singletonList;
import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.internal.DefaultComponentParameterAst;
import org.mule.runtime.extension.api.declaration.type.ExtensionsTypeLoaderFactory;
import org.mule.runtime.extension.api.declaration.type.annotation.LiteralTypeAnnotation;

import java.util.Arrays;

import org.junit.Test;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(PARAMETER_AST)
public class ComponentAstParameterTestCase {

  private final ClassTypeLoader TYPE_LOADER = ExtensionsTypeLoaderFactory.getDefault().createTypeLoader();

  private static final String RAW_VALUE = "rawValue";
  private static final String RAW_VALUE_EXPRESSION = "#[payload]";

  @Description("A component parameter with a different raw-value than default model parameter isn't a default parameter")
  @Test
  public void componentParameterWithRawValueIsNonADefaultParameter() {
    ParameterModel model = mock(ParameterModel.class);
    ComponentParameterAst parameterAst =
        new DefaultComponentParameterAst(RAW_VALUE, model, null, mock(ComponentGenerationInformation.class),
                                         new PropertiesResolver());
    assertThat(parameterAst.isDefaultValue(), is(false));
  }

  @Description("A component parameter without a raw-value is a default parameter")
  @Test
  public void componentParameterWithoutRawValueIsADefaultParameter() {
    ParameterModel model = mock(ParameterModel.class);
    ComponentParameterAst parameterAst =
        new DefaultComponentParameterAst((String) null, model, null, mock(ComponentGenerationInformation.class),
                                         new PropertiesResolver());
    assertThat(parameterAst.isDefaultValue(), is(true));
  }

  @Description("A component parameter wit same raw-value as a default model parameter is a default parameter")
  @Test
  public void componentParameterWithSameRawValueAsModelDefaultIsADefaultParameter() {
    ParameterModel model = mock(ParameterModel.class);
    when(model.getDefaultValue()).thenReturn(RAW_VALUE);
    when(model.getType()).thenReturn(TYPE_LOADER.load(String.class));
    ComponentParameterAst parameterAst =
        new DefaultComponentParameterAst(RAW_VALUE, model, null, mock(ComponentGenerationInformation.class),
                                         new PropertiesResolver());
    assertThat(parameterAst.isDefaultValue(), is(true));
  }

  @Description("A component parameter wit same raw-value expression as a default model parameter is a default parameter")
  @Test
  public void componentParameterWithSameRawValueExpressionAsModelDefaultIsADefaultParameter() {
    ParameterModel model = mock(ParameterModel.class);
    when(model.getDefaultValue()).thenReturn(RAW_VALUE_EXPRESSION);
    when(model.getType()).thenReturn(TYPE_LOADER.load(String.class));
    ComponentParameterAst parameterAst =
        new DefaultComponentParameterAst(RAW_VALUE_EXPRESSION, model, null, mock(ComponentGenerationInformation.class),
                                         new PropertiesResolver());
    assertThat(parameterAst.isDefaultValue(), is(true));
    assertThat(parameterAst.getValue().getLeft(),
               is(((DefaultComponentParameterAst) parameterAst).extractExpression(RAW_VALUE_EXPRESSION).get()));
  }

  @Description("A component parameter with complex value is a default parameter")
  @Test
  public void componentParameterWithComplexValueIsADefaultParameter() {
    ParameterModel model = mock(ParameterModel.class);
    DefaultComponentAstBuilder child = mock(DefaultComponentAstBuilder.class);
    ComponentParameterAst parameterAst =
        new DefaultComponentParameterAst(child, model, null, null, mock(ComponentGenerationInformation.class),
                                         new PropertiesResolver());
    assertThat(parameterAst.isDefaultValue(), is(true));
  }

  @Test
  public void referencesAsFixedValue() {
    ParameterModel model = mock(ParameterModel.class);
    when(model.getAllowedStereotypes()).thenReturn(singletonList(mock(StereotypeModel.class)));
    when(model.getType()).thenReturn(TYPE_LOADER.load(String.class));
    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst("aReference", model, null, mock(ComponentGenerationInformation.class),
                                         new PropertiesResolver());

    assertThat(parameterAst.getValue().isRight(), is(true));
    assertThat(parameterAst.getValue().getRight(), is("aReference"));
  }

  @Test
  public void notSupportedExpressionsAsFixedValue() {
    ParameterModel model = mock(ParameterModel.class);
    when(model.getExpressionSupport()).thenReturn(NOT_SUPPORTED);
    when(model.getType()).thenReturn(TYPE_LOADER.load(String.class));
    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst("#[justMarkers]", model, null, mock(ComponentGenerationInformation.class),
                                         new PropertiesResolver());

    assertThat(parameterAst.getValue().isRight(), is(true));
    assertThat(parameterAst.getValue().getRight(), is("#[justMarkers]"));
  }

  @Test
  public void literalAsFixedValue() {
    final MetadataType paramType = spy(TYPE_LOADER.load(String.class));
    when(paramType.getAnnotation(LiteralTypeAnnotation.class)).thenReturn(of(new LiteralTypeAnnotation()));

    ParameterModel model = mock(ParameterModel.class);
    when(model.getExpressionSupport()).thenReturn(SUPPORTED);
    when(model.getType()).thenReturn(paramType);
    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst("#['hello']", model, null, mock(ComponentGenerationInformation.class),
                                         new PropertiesResolver());

    assertThat(parameterAst.getValue().isRight(), is(true));
    assertThat(parameterAst.getValue().getRight(), is("#['hello']"));
  }

  @Test
  public void fixedValue() {
    ParameterModel model = mock(ParameterModel.class);
    when(model.getExpressionSupport()).thenReturn(SUPPORTED);
    when(model.getType()).thenReturn(TYPE_LOADER.load(String.class));
    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst("hello", model, null, mock(ComponentGenerationInformation.class),
                                         new PropertiesResolver());

    assertThat(parameterAst.getValue().isRight(), is(true));
    assertThat(parameterAst.getValue().getRight(), is("hello"));
  }

  @Test
  public void expression() {
    ParameterModel model = mock(ParameterModel.class);
    when(model.getExpressionSupport()).thenReturn(SUPPORTED);
    when(model.getType()).thenReturn(TYPE_LOADER.load(String.class));
    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst("#['hello']", model, null, mock(ComponentGenerationInformation.class),
                                         new PropertiesResolver());

    assertThat(parameterAst.getValue().isLeft(), is(true));
    assertThat(parameterAst.getValue().getLeft(), is("'hello'"));
  }

  @Test
  public void emptyPropertyToNumberWithDefaultValue() {
    ParameterModel model = mock(ParameterModel.class);
    when(model.getExpressionSupport()).thenReturn(NOT_SUPPORTED);
    when(model.getDefaultValue()).thenReturn("10");
    when(model.getType()).thenReturn(TYPE_LOADER.load(Integer.class));

    final PropertiesResolver propertiesResolver = new PropertiesResolver();
    propertiesResolver.setMappingFunction(k -> null);

    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst("${emptyProp}", model, null, mock(ComponentGenerationInformation.class),
                                         propertiesResolver);

    assertThat(parameterAst.getValue().isRight(), is(true));
    assertThat(parameterAst.getValue().getRight(), is(10));
  }

  @Test
  @Issue("W-11068928")
  public void emptyPropertyToNumberWithoutDefaultValue() {
    ParameterModel model = mock(ParameterModel.class);
    when(model.getExpressionSupport()).thenReturn(NOT_SUPPORTED);
    when(model.getType()).thenReturn(TYPE_LOADER.load(Integer.class));

    final PropertiesResolver propertiesResolver = new PropertiesResolver();
    propertiesResolver.setMappingFunction(k -> k);

    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst("${emptyProp}", model, null, mock(ComponentGenerationInformation.class),
                                         propertiesResolver);

    assertThat(parameterAst.getValue().isRight(), is(true));
    assertThat(parameterAst.getValue().getRight(), is("${emptyProp}"));
  }

  @Test
  @Issue("MULE-19074")
  public void primitiveToIntegerWithDefaultValue() {
    assertPrimitiveWithDefaultValue(Integer.toString(Integer.MAX_VALUE), Integer.MAX_VALUE, int.class);
  }

  @Test
  @Issue("MULE-19074")
  public void primitiveToFloatWithDefaultValue() {
    assertPrimitiveWithDefaultValue("99.99", 99.99f, float.class);
  }

  @Test
  @Issue("MULE-19074")
  public void primitiveToLongWithDefaultValue() {
    assertPrimitiveWithDefaultValue(Long.toString(Long.MAX_VALUE), Long.MAX_VALUE, long.class);
  }

  @Test
  @Issue("MULE-19871")
  public void primitiveToNumberWithDefaultValue() {
    assertPrimitiveWithDefaultValue(Long.toString(Long.MAX_VALUE), Long.MAX_VALUE, Number.class);
  }

  @Test
  @Issue("MULE-19871")
  public void primitiveToSmallNumberWithDefaultValue() {
    assertPrimitiveWithDefaultValue(Integer.toString(Integer.MAX_VALUE), Integer.MAX_VALUE, Number.class);
  }

  @Test
  @Issue("MULE-19871")
  public void primitiveToDecimalNumberWithDefaultValue() {
    assertPrimitiveWithDefaultValue("2.53", 2.53, Number.class);
  }

  @Test
  @Issue("MULE-19074")
  public void primitiveToByteWithDefaultValue() {
    assertPrimitiveWithDefaultValue("1", (byte) 1, byte.class);
  }

  @Test
  @Issue("MULE-19074")
  public void primitiveToShortWithDefaultValue() {
    assertPrimitiveWithDefaultValue("123", (short) 123, short.class);
  }

  @Test
  @Issue("MULE-19074")
  public void primitiveToDoubleWithDefaultValue() {
    assertPrimitiveWithDefaultValue("99.99", 99.99, double.class);
  }

  @Test
  public void nullPropertyToNumberWithDefaultValue() {
    ParameterModel model = mock(ParameterModel.class);
    when(model.getExpressionSupport()).thenReturn(NOT_SUPPORTED);
    when(model.getDefaultValue()).thenReturn(10);
    when(model.getType()).thenReturn(TYPE_LOADER.load(Integer.class));

    final PropertiesResolver propertiesResolver = new PropertiesResolver();
    propertiesResolver.setMappingFunction(k -> null);

    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst("${emptyProp}", model, null, mock(ComponentGenerationInformation.class),
                                         propertiesResolver);

    assertThat(parameterAst.getValue().isRight(), is(true));
    assertThat(parameterAst.getValue().getRight(), is(10));
  }

  @Test
  public void nullPropertyToNumberWithExpressionDefaultValue() {
    ParameterModel model = mock(ParameterModel.class);
    when(model.getExpressionSupport()).thenReturn(SUPPORTED);
    when(model.getDefaultValue()).thenReturn("#[10]");
    when(model.getType()).thenReturn(TYPE_LOADER.load(Integer.class));

    final PropertiesResolver propertiesResolver = new PropertiesResolver();
    propertiesResolver.setMappingFunction(k -> null);

    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst("${emptyProp}", model, null, mock(ComponentGenerationInformation.class),
                                         propertiesResolver);

    assertThat(parameterAst.getValue().isLeft(), is(true));
    assertThat(parameterAst.getValue().getLeft(), is("10"));
  }

  @Test
  public void nullValueToNumberWithDefaultValue() {
    ParameterModel model = mock(ParameterModel.class);
    when(model.getExpressionSupport()).thenReturn(NOT_SUPPORTED);
    when(model.getType()).thenReturn(TYPE_LOADER.load(Integer.class));

    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst((String) null, model, null, mock(ComponentGenerationInformation.class),
                                         new PropertiesResolver());

    assertThat(parameterAst.getValue().isLeft(), is(false));
    assertThat(parameterAst.getValue().isRight(), is(false));
  }

  @Test
  public void emptyStringNullDefaultNotNull() {
    ParameterModel model = mock(ParameterModel.class);
    when(model.getExpressionSupport()).thenReturn(SUPPORTED);
    when(model.getType()).thenReturn(TYPE_LOADER.load(String.class));
    when(model.getDefaultValue()).thenReturn(null);

    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst("", model, null, mock(ComponentGenerationInformation.class), new PropertiesResolver());

    assertThat(parameterAst.getValue().isRight(), is(true));
    assertThat(parameterAst.getValue().getRight(), is(""));
  }

  @Test
  @Issue("MULE-19183")
  @Description("A string-type parameter set to \"\" shouldn't be overridden by its default value")
  public void emptyStringOverridesNotNullDefaultValue() {
    ParameterModel model = mock(ParameterModel.class);
    when(model.getExpressionSupport()).thenReturn(SUPPORTED);
    when(model.getType()).thenReturn(TYPE_LOADER.load(String.class));
    when(model.getDefaultValue()).thenReturn(RAW_VALUE);

    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst("", model, null, mock(ComponentGenerationInformation.class), new PropertiesResolver());

    assertThat(parameterAst.getValue().isRight(), is(true));
    assertThat(parameterAst.getValue().getRight(), is(""));
  }

  @Test
  @Issue("MULE-19117")
  public void enumValue() {
    MetadataType testEnumType = BaseTypeBuilder.create(JAVA).stringType()
        .with(new EnumAnnotation<>(Arrays.stream(TestEnum.values()).map(TestEnum::name).toArray(String[]::new)))
        .with(new ClassInformationAnnotation(TestEnum.class))
        .build();

    ParameterModel model = mock(ParameterModel.class);
    when(model.getExpressionSupport()).thenReturn(NOT_SUPPORTED);
    when(model.getType()).thenReturn(testEnumType);

    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst(TestEnum.VALUE_B.name(), model, null, mock(ComponentGenerationInformation.class),
                                         new PropertiesResolver());

    assertThat(parameterAst.getValue().isRight(), is(true));
    assertThat(parameterAst.getValue().getRight(), is(TestEnum.VALUE_B.name()));
  }

  @Test
  @Issue("MULE-19542")
  public void listExpressionWithStereotype() {
    final ArrayType arrayType =
        BaseTypeBuilder.create(JAVA).arrayType().of(BaseTypeBuilder.create(JAVA).objectType().build()).build();

    ParameterModel model = mock(ParameterModel.class);
    when(model.getExpressionSupport()).thenReturn(SUPPORTED);
    when(model.getType()).thenReturn(arrayType);
    when(model.getAllowedStereotypes()).thenReturn(singletonList(mock(StereotypeModel.class)));

    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst("#[[{}, {}]]", model, null, mock(ComponentGenerationInformation.class),
                                         new PropertiesResolver());

    assertThat(parameterAst.getValue().isLeft(), is(true));
    assertThat(parameterAst.getValue().getLeft(), is("[{}, {}]"));
  }

  @Test
  @Issue("MULE-19542")
  public void complexExpressionWithStereotype() {
    final ObjectType objectType = BaseTypeBuilder.create(JAVA).objectType().build();

    ParameterModel model = mock(ParameterModel.class);
    when(model.getExpressionSupport()).thenReturn(SUPPORTED);
    when(model.getType()).thenReturn(objectType);
    when(model.getAllowedStereotypes()).thenReturn(singletonList(mock(StereotypeModel.class)));

    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst("#[{}]", model, null, mock(ComponentGenerationInformation.class),
                                         new PropertiesResolver());

    assertThat(parameterAst.getValue().isLeft(), is(true));
    assertThat(parameterAst.getValue().getLeft(), is("{}"));
  }

  @Test
  @Issue("MULE-20045")
  public void propertiesPlaceholdersRefreshed() {
    ParameterModel model = mock(ParameterModel.class);
    when(model.getExpressionSupport()).thenReturn(NOT_SUPPORTED);
    when(model.getType()).thenReturn(TYPE_LOADER.load(Integer.class));

    final PropertiesResolver propertiesResolver = new PropertiesResolver();
    propertiesResolver.setMappingFunction(k -> "10");

    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst("${someNumber}", model, null, mock(ComponentGenerationInformation.class),
                                         propertiesResolver);

    assertThat(parameterAst.getResolvedRawValue(), is("10"));
    assertThat(parameterAst.getValue().getRight(), is(10));

    propertiesResolver.setMappingFunction(k -> "20");

    assertThat(parameterAst.getResolvedRawValue(), is("20"));
    assertThat(parameterAst.getValue().getRight(), is(20));
  }

  public enum TestEnum {
    VALUE_A, VALUE_B;
  }

  private void assertPrimitiveWithDefaultValue(String value, Number expected, Class<? extends Number> type) {
    ParameterModel model = mock(ParameterModel.class);
    when(model.getExpressionSupport()).thenReturn(NOT_SUPPORTED);
    when(model.getDefaultValue()).thenReturn(value);
    when(model.getType()).thenReturn(TYPE_LOADER.load(type));

    final PropertiesResolver propertiesResolver = new PropertiesResolver();
    propertiesResolver.setMappingFunction(k -> null);

    final DefaultComponentParameterAst parameterAst =
        new DefaultComponentParameterAst("${emptyProp}", model, null, mock(ComponentGenerationInformation.class),
                                         propertiesResolver);

    assertThat(parameterAst.getValue().isRight(), is(true));
    assertThat(parameterAst.getValue().getRight(), is(equalTo(expected)));
  }
}
