/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal;

import static java.util.Collections.emptyList;
import static java.util.Collections.unmodifiableList;
import static java.util.Collections.unmodifiableMap;

import org.mule.runtime.ast.api.AstParserAttribute;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.builder.ComponentMetadataAstBuilder;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;

import java.net.URI;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.stream.Collectors;

public class DefaultComponentMetadataAst implements ComponentMetadataAst {

  public static Builder builder() {
    return new Builder(new PropertiesResolver());
  }

  public static Builder builder(PropertiesResolver propertiesResolver) {
    return new Builder(propertiesResolver);
  }

  private final Optional<String> fileName;
  private final Optional<URI> fileUri;
  private final List<ImportedResource> importChain;

  private final OptionalInt startLine;
  private final OptionalInt startColumn;
  private final OptionalInt endLine;
  private final OptionalInt endColumn;

  private final Optional<String> sourceCode;

  private final Map<String, String> docAttributes;

  /**
   * Cached resolution of document attributes. Even though the docAttributes are immutable, the property resolver mapping function
   * is not, so we need to invalidate if it changes.
   */
  private transient Map<String, String> resolvedDocAttributes = null;

  private final Map<String, Object> syntheticAttributes;

  private final PropertiesResolver propertiesResolver;

  private DefaultComponentMetadataAst(Optional<String> fileName,
                                      Optional<URI> fileUri,
                                      List<ImportedResource> importChain,
                                      OptionalInt startLine,
                                      OptionalInt startColumn,
                                      OptionalInt endLine,
                                      OptionalInt endColumn,
                                      Optional<String> sourceCode,
                                      Map<String, String> docAttributes,
                                      Map<String, Object> syntheticAttributes,
                                      PropertiesResolver propertiesResolver) {
    this.fileName = fileName;
    this.fileUri = fileUri;
    this.importChain = importChain;
    this.startLine = startLine;
    this.startColumn = startColumn;
    this.endLine = endLine;
    this.endColumn = endColumn;
    this.sourceCode = sourceCode;
    this.docAttributes = docAttributes;
    this.syntheticAttributes = syntheticAttributes;
    this.propertiesResolver = propertiesResolver;

    // If necessary, registers a callback for invalidating the resolved document attributes if the mapping function changes.
    if (docAttributes.values().stream().anyMatch(attr -> attr.contains("${"))) {
      propertiesResolver.onMappingFunctionChanged(this::invalidateResolvedDocAttributes);
    }
  }

  @Override
  public Optional<String> getFileName() {
    return fileName;
  }

  @Override
  public Optional<URI> getFileUri() {
    return fileUri;
  }

  @Override
  public List<ImportedResource> getImportChain() {
    return importChain;
  }

  @Override
  public OptionalInt getStartLine() {
    return startLine;
  }

  @Override
  public OptionalInt getStartColumn() {
    return startColumn;
  }

  @Override
  public OptionalInt getEndLine() {
    return endLine;
  }

  @Override
  public OptionalInt getEndColumn() {
    return endColumn;
  }

  @Override
  public Optional<String> getSourceCode() {
    return sourceCode;
  }

  @Override
  public Map<String, String> getDocAttributes() {
    if (resolvedDocAttributes == null) {
      resolvedDocAttributes = computeResolvedDocAttributes();
    }
    return resolvedDocAttributes;
  }

  @Override
  public Map<String, Object> getParserAttributes() {
    return syntheticAttributes;
  }

  private Map<String, String> computeResolvedDocAttributes() {
    return docAttributes.entrySet()
        .stream()
        .collect(Collectors.toMap(Map.Entry::getKey,
                                  entry -> propertiesResolver.apply(entry.getValue())));
  }

  private void invalidateResolvedDocAttributes() {
    resolvedDocAttributes = null;
  }

  public static class Builder implements ComponentMetadataAstBuilder {

    private Optional<String> fileName = Optional.empty();
    private Optional<URI> fileUri = Optional.empty();
    private List<ImportedResource> importChain = emptyList();

    private OptionalInt startLine = OptionalInt.empty();
    private OptionalInt startColumn = OptionalInt.empty();
    private OptionalInt endLine = OptionalInt.empty();
    private OptionalInt endColumn = OptionalInt.empty();

    private Optional<String> sourceCode = Optional.empty();

    private final Map<String, String> docAttributes = new HashMap<>();

    private final Map<String, Object> parserAttributes = new HashMap<>();

    private final PropertiesResolver propertiesResolver;

    public Builder(PropertiesResolver propertiesResolver) {
      this.propertiesResolver = propertiesResolver;
    }

    @Override
    public Builder setFileName(String fileName) {
      this.fileName = Optional.of(fileName);
      return this;
    }

    @Override
    public Builder setFileUri(URI fileUri) {
      this.fileUri = Optional.of(fileUri);
      return this;
    }

    @Override
    public Builder setImportChain(List<ImportedResource> importChain) {
      this.importChain = unmodifiableList(importChain);
      return this;
    }

    @Override
    public Builder setStartLine(int startLine) {
      this.startLine = OptionalInt.of(startLine);
      return this;
    }

    @Override
    public Builder setStartColumn(int startColumn) {
      this.startColumn = OptionalInt.of(startColumn);
      return this;
    }

    @Override
    public Builder setEndLine(int endLine) {
      this.endLine = OptionalInt.of(endLine);
      return this;
    }

    @Override
    public Builder setEndColumn(int endColumn) {
      this.endColumn = OptionalInt.of(endColumn);
      return this;
    }

    @Override
    public Builder setSourceCode(String sourceCode) {
      this.sourceCode = Optional.of(sourceCode);
      return this;
    }

    @Override
    public Builder putDocAttribute(String key, String value) {
      this.docAttributes.put(key, value);
      return this;
    }

    @Override
    public Builder putParserAttribute(String key, Object value) {
      this.parserAttributes.put(key, value);
      return this;
    }

    @Override
    public <T> Builder putParserAttribute(AstParserAttribute<T> attribute, T value) {
      this.parserAttributes.put(attribute.getKey(), value);
      return this;
    }

    @Override
    public ComponentMetadataAst build() {
      return new DefaultComponentMetadataAst(fileName, fileUri, importChain,
                                             startLine, startColumn, endLine, endColumn,
                                             sourceCode,
                                             unmodifiableMap(docAttributes), unmodifiableMap(parserAttributes),
                                             propertiesResolver);
    }
  }
}
