/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.validation;

import static java.lang.Thread.currentThread;
import static java.util.ServiceLoader.load;
import static java.util.stream.StreamSupport.stream;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.ast.api.util.MuleAstUtils;
import org.mule.runtime.ast.api.validation.ArtifactValidation;
import org.mule.runtime.ast.api.validation.Validation;
import org.mule.runtime.ast.api.validation.ValidationsProvider;
import org.mule.runtime.ast.api.validation.ArtifactAstValidator;
import org.mule.runtime.ast.api.validation.ArtifactAstValidatorBuilder;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Predicate;

import org.slf4j.Logger;

public class DefaultValidatorBuilder implements ArtifactAstValidatorBuilder {

  private static final Logger LOGGER = getLogger(DefaultValidatorBuilder.class);

  private Consumer<ValidationsProvider> validationEnricher = p -> {
  };
  private Predicate<Validation> validationsFilter = v -> true;
  private ClassLoader artifactRegionClassLoader = currentThread().getContextClassLoader();

  @Override
  public DefaultValidatorBuilder withValidationEnricher(Consumer<ValidationsProvider> validationEnricher) {
    this.validationEnricher = validationEnricher;
    return this;
  }

  @Override
  public DefaultValidatorBuilder withValidationsFilter(Predicate<Validation> validationsFilter) {
    this.validationsFilter = validationsFilter;
    return this;
  }

  @Override
  public DefaultValidatorBuilder withArtifactRegionClassLoader(ClassLoader artifactRegionClassLoader) {
    this.artifactRegionClassLoader = artifactRegionClassLoader;
    return this;
  }

  @Override
  public ArtifactAstValidator build() {
    List<Validation> componentValidations = new ArrayList<>();
    List<ArtifactValidation> artifactValidations = new ArrayList<>();

    Map<Class<? extends ValidationsProvider>, ValidationsProvider> providers = new LinkedHashMap<>();

    // Get the core/runtime validations
    stream(load(ValidationsProvider.class, MuleAstUtils.class.getClassLoader()).spliterator(), false)
        .forEach(vp -> providers.putIfAbsent(vp.getClass(), vp));
    // Get the validations from the extensions
    stream(load(ValidationsProvider.class).spliterator(), false)
        .forEach(vp -> providers.putIfAbsent(vp.getClass(), vp));

    providers.values()
        .forEach(vp -> {
          vp.setArtifactRegionClassLoader(artifactRegionClassLoader);
          validationEnricher.accept(vp);

          vp.get()
              .stream()
              .filter(validationsFilter)
              .forEach(componentValidations::add);
          vp.getArtifactValidations()
              .stream()
              .filter(validationsFilter)
              .forEach(artifactValidations::add);
        });

    LOGGER.debug("Discovered component validations: '{}'", componentValidations);
    LOGGER.debug("Discovered artifact validations: '{}'", artifactValidations);

    return new DefaultValidator(artifactValidations, componentValidations);
  }
}
