/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api;

import java.util.Optional;

import static java.util.Optional.ofNullable;

/**
 * Parsing attribute that might be DSL-specific.
 *
 * For example, in case of XML-based DSLs, an attribute could be the position of the opening tag in the source file.
 *
 * @param <T> The type of the attribute value. Please try to stick to basic Java types like Integer, Double, BigDecimal, String,
 *            Boolean, etc. to avoid class loading issues when serializing/deserializing these.
 *
 * @since 1.1
 */
public class AstParserAttribute<T> {

  private final String key;

  protected AstParserAttribute(String key) {
    this.key = key;
  }

  /**
   * @return the key for this attribute in the attributes map of {@link ComponentMetadataAst}.
   */
  public String getKey() {
    return key;
  }

  /**
   * Convenience for extracting the value from the attributes map of the given {@link ComponentMetadataAst} with type conversion.
   *
   * @param componentMetadataAst A {@link ComponentMetadataAst} to get the attribute value from.
   * @return the attribute value with the right type.
   */
  public Optional<T> get(ComponentMetadataAst componentMetadataAst) {
    return ofNullable((T) componentMetadataAst.getParserAttributes().get(key));
  }

  /**
   * Gets the raw value from the attributes map of the given {@link ComponentMetadataAst} without trying to perform type
   * conversion.
   *
   * {@link #get(ComponentMetadataAst)} should be preferred for type safety.
   *
   * @param componentMetadataAst A {@link ComponentMetadataAst} to get the attribute value from.
   * @return the attribute value without any type conversion.
   */
  public Optional<Object> getRaw(ComponentMetadataAst componentMetadataAst) {
    return ofNullable(componentMetadataAst.getParserAttributes().get(key));
  }
}
