/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.util;

import static org.mule.runtime.ast.api.ArtifactType.APPLICATION;
import static org.mule.runtime.ast.api.error.ErrorTypeRepositoryProvider.getCoreErrorTypeRepo;
import static org.mule.runtime.ast.api.util.AstTraversalDirection.TOP_DOWN;
import static org.mule.runtime.extension.api.ExtensionConstants.ERROR_MAPPINGS_PARAMETER_NAME;
import static org.mule.runtime.extension.api.component.ComponentParameterization.builder;
import static org.mule.runtime.extension.api.util.ExtensionMetadataTypeUtils.isMap;
import static org.mule.runtime.extension.api.util.ExtensionModelUtils.getGroupAndParametersPairs;
import static org.mule.runtime.internal.dsl.DslConstants.CORE_PREFIX;

import static java.lang.Thread.currentThread;
import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ArtifactType;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.validation.ArtifactAstValidator;
import org.mule.runtime.ast.api.validation.ArtifactAstValidatorBuilder;
import org.mule.runtime.ast.api.validation.Validation;
import org.mule.runtime.ast.api.validation.ValidationResult;
import org.mule.runtime.ast.api.validation.ValidationsProvider;
import org.mule.runtime.ast.internal.builder.LightComponentAstBuilder;
import org.mule.runtime.ast.internal.validation.DefaultValidatorBuilder;
import org.mule.runtime.extension.api.component.ComponentParameterization;

import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.UnaryOperator;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Set of utility methods for working with {@link ArtifactAst}s and {@link ComponentAst}s.
 *
 * @since 1.0
 */
public final class MuleAstUtils {

  private static final Logger LOGGER = LoggerFactory.getLogger(MuleAstUtils.class);

  private static final ExtensionModel CORE_EXT_MODEL;

  static {
    ExtensionModel coreExtModel = null;
    try {
      Class coreExtModelProivdelClass = Class.forName("org.mule.runtime.core.api.extension.MuleExtensionModelProvider");
      coreExtModel = (ExtensionModel) coreExtModelProivdelClass.getMethod("getExtensionModel").invoke(null);
    } catch (ClassNotFoundException | IllegalAccessException | IllegalArgumentException | InvocationTargetException
        | NoSuchMethodException | SecurityException e) {
      LOGGER.warn(e.getClass().getName() + ": " + e.getMessage());
    }
    CORE_EXT_MODEL = coreExtModel;
  }

  private static final ArtifactAst EMPTY_ARTIFACT = new BaseArtifactAst() {

    @Override
    public String getArtifactName() {
      return "<empty>";
    }

    @Override
    public ArtifactType getArtifactType() {
      return APPLICATION;
    }

    @Override
    public Set<ExtensionModel> dependencies() {
      if (CORE_EXT_MODEL == null) {
        return Collections.<ExtensionModel>emptySet();
      } else {
        return singleton(CORE_EXT_MODEL);
      }
    }

    @Override
    public Optional<ArtifactAst> getParent() {
      return empty();
    }

    @Override
    public List<ComponentAst> topLevelComponents() {
      return emptyList();
    }

    @Override
    public void updatePropertiesResolver(UnaryOperator<String> newPropertiesResolver) {
      // Nothing to do
    }

    @Override
    public ErrorTypeRepository getErrorTypeRepository() {
      return getCoreErrorTypeRepo();
    }

    @Override
    public Collection<ImportedResource> getImportedResources() {
      return emptyList();
    }
  };

  private MuleAstUtils() {
    // Nothing to do
  }

  /**
   * Given an {@link ArtifactAst}, return all those components within it that are not accessible from a top-level component.
   *
   * @param appModel the model to determine orphan components from.
   * @return components within the provided {@code appModel} that are not accessible from a top-level component.
   */
  public static Set<ComponentAst> resolveOrphanComponents(ArtifactAst appModel) {
    final Set<ComponentAst> notOrphanComponents = appModel.topLevelComponentsStream()
        .filter(componentAst -> componentAst.getModel(ParameterizedModel.class).isPresent())
        .flatMap(ComponentAst::recursiveStream)
        .collect(toSet());
    notOrphanComponents.addAll(appModel
        .filteredComponents(cm -> !notOrphanComponents.contains(cm))
        .flatMap(ComponentAst::directChildrenStream)
        .collect(toSet()));

    return appModel
        .filteredComponents(cm -> !notOrphanComponents.contains(cm))
        .collect(toSet());
  }

  /**
   * Provides a singleton empty {@link ArtifactAst} instance.
   *
   * @return an {@link ArtifactAst} without components.
   */
  public static ArtifactAst emptyArtifact() {
    return EMPTY_ARTIFACT;
  }

  /**
   * Given a {@link ComponentAst}, lookup for a parameter that references an object with a certain stereotype.
   *
   * @param referent            the {@link ComponentAst} that will be checked for the reference.
   * @param referenceStereotype the {@link StereotypeModel} to check for.
   * @return any {@link ComponentParameterAst} representing a parameter of {@code referent} that references an object assignable
   *         to {@code referenceStereotype}, if it exists.
   */
  public static Optional<ComponentParameterAst> parameterOfType(ComponentAst referent, StereotypeModel referenceStereotype) {
    return referent.getModel(ParameterizedModel.class)
        .flatMap(parameterized -> getGroupAndParametersPairs(parameterized)
            .filter(pair -> pair.getSecond().getAllowedStereotypes()
                .stream()
                .anyMatch(allowed -> allowed.isAssignableTo(referenceStereotype)))
            .map(pair -> referent.getParameter(pair.getFirst().getName(), pair.getSecond().getName()))
            .findAny());
  }

  /**
   * Given a {@link ComponentAst}, lookup for all parameters that reference an object with a certain stereotype.
   *
   * @param referent            the {@link ComponentAst} that will be checked for the reference.
   * @param referenceStereotype the {@link StereotypeModel} to check for.
   * @return all {@link ComponentParameterAst}s representing parameters of {@code referent} that reference an object assignable to
   *         {@code referenceStereotype}.
   */
  public static List<ComponentParameterAst> parametersOfType(ComponentAst referent, StereotypeModel referenceStereotype) {
    return referent.getModel(ParameterizedModel.class)
        .map(parameterized -> getGroupAndParametersPairs(parameterized)
            .filter(pair -> pair.getSecond().getAllowedStereotypes()
                .stream()
                .anyMatch(allowed -> allowed.isAssignableTo(referenceStereotype)))
            .map(pair -> referent.getParameter(pair.getFirst().getName(), pair.getSecond().getName()))
            .collect(toList()))
        .orElse(emptyList());
  }

  /**
   * Performs a given action on the {@link ComponentAst}s that form the value of the provided parameter.
   *
   * @param param  the parameter on which complex parameter values the {@code action} will be called.
   * @param action the callback to perform on each found parameter item.
   */
  public static void doOnParamComponents(ComponentParameterAst param, Consumer<ComponentAst> action) {
    if (param.getModel().getName().equals(ERROR_MAPPINGS_PARAMETER_NAME)) {
      return;
    }
    param.getModel().getType().accept(new MetadataTypeVisitor() {

      @Override
      public void visitArrayType(ArrayType arrayType) {
        final Object value = param.getValue().getRight();
        if (value != null && value instanceof Collection) {
          ((Iterable<ComponentAst>) value).forEach(this::doAction);
        }
      }

      @Override
      public void visitObject(ObjectType objectType) {
        final Object value = param.getValue().getRight();
        if (isMap(objectType)) {
          if (value != null && value instanceof Collection) {
            ((Iterable<ComponentAst>) value).forEach(this::doAction);
          }
        } else {
          if (value instanceof ComponentAst) {
            doAction(value);
          }
        }
      }

      private void doAction(Object component) {
        if (component != null && component instanceof ComponentAst) {
          action.accept((ComponentAst) component);
        }
      }
    });
  }

  /**
   * Similar to {@link ArtifactAst#recursiveStream()}, but it enhances the {@link Stream} by adding hierarchy context to each
   * item. The {@link ArtifactAst} is traversed in {@link TOP_DOWN} direction.
   * <p>
   * The elements of the {@link Stream} are {@link Pair}s:
   * <ul>
   * <li>the first element is the same component that would be obtained by calling {@link ArtifactAst#recursiveStream()}.</li>
   * <li>the second element is the hierarchy that was traversed to get to the first element, the first element being a root and
   * the last being the direct parent of the first element of the pair.</li>
   * </ul>
   *
   * @param artifact the {@link ArtifactAst} to generate a {@link Stream} for.
   * @return an enhanced {@link Stream} with hierarchy context.
   */
  public static Stream<Pair<ComponentAst, List<ComponentAst>>> recursiveStreamWithHierarchy(ArtifactAst artifact) {
    return recursiveStreamWithHierarchy(artifact, TOP_DOWN, false);
  }

  /**
   * Similar to {@link ArtifactAst#recursiveStream()}, but it enhances the {@link Stream} by adding hierarchy context to each
   * item. The {@link ArtifactAst} is traversed in {@link AstTraversalDirection} direction
   * <p>
   * The elements of the {@link Stream} are {@link Pair}s:
   * <ul>
   * <li>the first element is the same component that would be obtained by calling {@link ArtifactAst#recursiveStream()}.</li>
   * <li>the second element is the hierarchy that was traversed to get to the first element, the first element being a root and
   * the last being the direct parent of the first element of the pair.</li>
   * </ul>
   *
   * @param artifact                     the {@link ArtifactAst} to generate a {@link Stream} for.
   * @param direction                    the {@link AstTraversalDirection} used to navigate the {@code artifact}.
   * @param ignoreParameterComplexValues whether the returned {@link Stream} will contain any complex values in component
   *                                     parameters.
   * @return an enhanced {@link Stream} with hierarchy context.
   */
  public static Stream<Pair<ComponentAst, List<ComponentAst>>> recursiveStreamWithHierarchy(ArtifactAst artifact,
                                                                                            AstTraversalDirection direction) {
    return recursiveStreamWithHierarchy(artifact, direction, false);
  }

  /**
   * Similar to {@link ArtifactAst#recursiveStream()}, but it enhances the {@link Stream} by adding hierarchy context to each
   * item. The {@link ArtifactAst} is traversed in {@link AstTraversalDirection} direction
   * <p>
   * The elements of the {@link Stream} are {@link Pair}s:
   * <ul>
   * <li>the first element is the same component that would be obtained by calling {@link ArtifactAst#recursiveStream()}.</li>
   * <li>the second element is the hierarchy that was traversed to get to the first element, the first element being a root and
   * the last being the direct parent of the first element of the pair.</li>
   * </ul>
   *
   * @param artifact                     the {@link ArtifactAst} to generate a {@link Stream} for.
   * @param direction                    the {@link AstTraversalDirection} used to navigate the {@code artifact}.
   * @param ignoreParameterComplexValues whether the returned {@link Stream} will contain any complex values in component
   *                                     parameters.
   * @return an enhanced {@link Stream} with hierarchy context.
   */
  public static Stream<Pair<ComponentAst, List<ComponentAst>>> recursiveStreamWithHierarchy(ArtifactAst artifact,
                                                                                            AstTraversalDirection direction,
                                                                                            boolean ignoreParameterComplexValues) {

    final Set<ComponentAst> orphanComponents = resolveOrphanComponents(artifact);

    if (orphanComponents.isEmpty()) {
      return direction.recursiveStreamWithHierarchy(artifact.topLevelComponentsStream(), ignoreParameterComplexValues);
    } else {
      // Do not break code that checks if a component is top-level by checking for an empty hierarchy for orphans.
      final ComponentAst orphanPhantomContainer = new LightComponentAstBuilder()
          .withIdentifier(ComponentIdentifier.builder()
              .namespace(CORE_PREFIX)
              .name("orphan-container")
              .build())
          .build();

      return Stream.concat(direction.recursiveStreamWithHierarchy(artifact.topLevelComponentsStream()),
                           direction.recursiveStreamWithHierarchy(orphanComponents.stream(),
                                                                  singletonList(orphanPhantomContainer)));
    }
  }

  /**
   * Provides a way to execute validations on an {@link ArtifactAst}.
   * 
   * @return a builder for an {@link ArtifactAstValidator}.
   * @since 1.1
   */
  public static ArtifactAstValidatorBuilder validatorBuilder() {
    return new DefaultValidatorBuilder();
  }

  /**
   * Provides a way to create a {@link ComponentParameterization} from a {@link ComponentAst}.
   *
   * @param componentAst the {@link ComponentAst} to be transformed.
   * @return a {@link ComponentParameterization}.
   * @throws IllegalArgumentException if the {@link ComponentAst} model is not a {@link ParameterizedModel}
   * @since 1.1
   */
  // TODO W-11214395: support complex types
  public static ComponentParameterization<ParameterizedModel> createComponentParameterizationFromComponentAst(ComponentAst componentAst) {
    return componentAst.getModel(ParameterizedModel.class).map(model -> {
      ComponentParameterization.Builder<ParameterizedModel> builder = builder(model);
      componentAst.getParameters().forEach(parameter -> parameter.getValue().getValue()
          .ifPresent(value -> builder.withParameter(parameter.getGroupModel(), parameter.getModel(), value)));
      builder.withComponentIdentifier(componentAst.getIdentifier());
      return builder.build();
    }).orElseThrow(() -> new IllegalArgumentException("ComponentAst model must be a ParameterizedModel"));
  }

  /**
   * Runs all validations available from the current {@link Thread#getContextClassLoader()}.
   * <p>
   * All validations will be run even if failures are found in some. The result will contain all failures.
   *
   * @param artifact the artifact to validate
   * @return the result of the validation
   * 
   * @deprecated since 1.1, use {@link #validatorBuilder()} instead.
   */
  @Deprecated
  public static ValidationResult validate(ArtifactAst artifact) {
    return validate(artifact, v -> {
    }, v -> true, currentThread().getContextClassLoader());
  }

  /**
   * Runs all validations available from the current {@link Thread#getContextClassLoader()}.
   * <p>
   * All validations will be run even if failures are found in some. The result will contain all failures.
   *
   * @param artifact           the artifact to validate
   * @param validationEnricher a postprocessor to apply on every discovered {@link ValidationsProvider}.
   * @return the result of the validation
   * 
   * @deprecated since 1.1, use {@link #validatorBuilder()} instead.
   */
  @Deprecated
  public static ValidationResult validate(ArtifactAst artifact, Consumer<ValidationsProvider> validationEnricher) {
    return validate(artifact, validationEnricher, v -> true, currentThread().getContextClassLoader());
  }

  /**
   * Runs all validations available from the current {@link Thread#getContextClassLoader()}.
   * <p>
   * All validations will be run even if failures are found in some. The result will contain all failures.
   *
   * @param artifact          the artifact to validate
   * @param validationsFilter additional filter to apply on the discovered validations before evaluating them.
   * @return the result of the validation
   * 
   * @deprecated since 1.1, use {@link #validatorBuilder()} instead.
   */
  @Deprecated
  public static ValidationResult validate(ArtifactAst artifact, Predicate<Validation> validationsFilter) {
    return validate(artifact, v -> {
    }, validationsFilter, currentThread().getContextClassLoader());
  }

  /**
   * Runs all validations available from the current {@link Thread#getContextClassLoader()}.
   * <p>
   * All validations will be run even if failures are found in some. The result will contain all failures.
   *
   * @param artifact                  the artifact to validate
   * @param validationEnricher        a postprocessor to apply on every discovered {@link ValidationsProvider}.
   * @param validationsFilter         additional filter to apply on the discovered validations before evaluating them.
   * @param artifactRegionClassLoader the classLoader to use to load any resources referenced by the artifact
   * @return the result of the validation
   * 
   * @since 1.1
   * 
   * @deprecated since 1.1, use {@link #validatorBuilder()} instead.
   */
  @Deprecated
  public static ValidationResult validate(ArtifactAst artifact, Consumer<ValidationsProvider> validationEnricher,
                                          Predicate<Validation> validationsFilter,
                                          ClassLoader artifactRegionClassLoader) {
    return validatorBuilder()
        .withValidationEnricher(validationEnricher)
        .withValidationsFilter(validationsFilter)
        .withArtifactRegionClassLoader(artifactRegionClassLoader)
        .build()
        .validate(artifact);
  }
}
