/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.validation;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.Collections.unmodifiableMap;

import org.mule.api.annotation.NoImplement;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.internal.validation.DefaultValidationResultItem;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Relates a failing validation to the component where that validation failed.
 *
 * @since 1.0
 */
@NoImplement
public interface ValidationResultItem {

  /**
   * @param component  the component this validation result applies to.
   * @param validation the validation that failed.
   * @param message    the message explaining the reason of the validation failure.
   * @return an object representing a validation failure.
   */
  public static ValidationResultItem create(ComponentAst component, Validation validation, String message) {
    return create(component, emptyList(), validation, message);
  }

  /**
   * @param component  the component this validation result applies to.
   * @param parameter  the parameter from the component this validation result applies to.
   * @param validation the validation that failed.
   * @param message    the message explaining the reason of the validation failure.
   * @return an object representing a validation failure.
   */
  public static ValidationResultItem create(ComponentAst component, ComponentParameterAst parameter, Validation validation,
                                            String message) {
    return create(component, singletonList(parameter), validation, message);
  }

  /**
   * @param component  the component this validation result applies to.
   * @param parameters the parameters from the component this validation result applies to.
   * @param validation the validation that failed.
   * @param message    the message explaining the reason of the validation failure.
   * @return an object representing a validation failure.
   */
  public static ValidationResultItem create(ComponentAst component, List<ComponentParameterAst> parameters, Validation validation,
                                            String message) {
    return create(component, parameters, validation, message, emptyMap());
  }

  /**
   * @param component      the component this validation result applies to.
   * @param parameter      the parameter from the component this validation result applies to.
   * @param validation     the validation that failed.
   * @param message        the message explaining the reason of the validation failure.
   * @param additionalData additional data provided by the actual validation.
   * @return an object representing a validation failure.
   */
  public static ValidationResultItem create(ComponentAst component, ComponentParameterAst parameter, Validation validation,
                                            String message, Map<String, String> additionalData) {
    return create(singletonList(component), singletonList(parameter), validation, message, additionalData);
  }

  /**
   * @param component      the component this validation result applies to.
   * @param parameters     the parameters from the component this validation result applies to.
   * @param validation     the validation that failed.
   * @param message        the message explaining the reason of the validation failure.
   * @param additionalData additional data provided by the actual validation.
   * @return an object representing a validation failure.
   */
  public static ValidationResultItem create(ComponentAst component, List<ComponentParameterAst> parameters, Validation validation,
                                            String message, Map<String, String> additionalData) {
    return new DefaultValidationResultItem(singletonList(component), parameters, validation, message,
                                           unmodifiableMap(new HashMap<>(additionalData)));
  }

  /**
   * @param components the components this validation result applies to.
   * @param validation the validation that failed.
   * @param message    the message explaining the reason of the validation failure.
   * @return an object representing a validation failure.
   */
  public static ValidationResultItem create(List<ComponentAst> components, Validation validation, String message) {
    return create(components, emptyList(), validation, message);
  }

  /**
   * @param components the components this validation result applies to.
   * @param parameters the parameters from the components this validation result applies to.
   * @param validation the validation that failed.
   * @param message    the message explaining the reason of the validation failure.
   * @return an object representing a validation failure.
   */
  public static ValidationResultItem create(List<ComponentAst> components, List<ComponentParameterAst> parameters,
                                            Validation validation, String message) {
    return create(components, parameters, validation, message, emptyMap());
  }

  /**
   * @param components     the components this validation result applies to.
   * @param parameters     the parameters from the components this validation result applies to.
   * @param validation     the validation that failed.
   * @param message        the message explaining the reason of the validation failure.
   * @param additionalData additional data provided by the actual validation.
   * @return an object representing a validation failure.
   */
  public static ValidationResultItem create(List<ComponentAst> components, List<ComponentParameterAst> parameters,
                                            Validation validation, String message, Map<String, String> additionalData) {
    return new DefaultValidationResultItem(components, parameters, validation, message,
                                           unmodifiableMap(new HashMap<>(additionalData)));
  }

  /**
   *
   * @return the components this validation result applies to.
   */
  List<ComponentAst> getComponents();

  /**
   *
   * @return the parameters from the components this validation result applies to.
   */
  List<ComponentParameterAst> getParameters();

  /**
   *
   * @return the validation that failed.
   */
  Validation getValidation();

  /**
   *
   * @return the message explaining the reason of the validation failure.
   */
  String getMessage();

  /**
   * Any additional data provided by the actual validation, providing context for this item (for instance, in the case of
   * validation on parameter values, it may contain an entry indication in which character of the value the validation error was
   * found).
   * 
   * @since 1.1
   */
  Map<String, String> getAdditionalData();
}
