/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.util;

import static org.mule.runtime.api.functional.Either.right;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.ast.AllureConstants.DietToolingFeature.DIET_TOOLING;
import static org.mule.runtime.ast.AllureConstants.DietToolingFeature.Stories.METADATA_TYPES_RESOLUTION;
import static org.mule.runtime.ast.AllureConstants.DietToolingFeature.Stories.SAMPLE_DATA;
import static org.mule.runtime.ast.AllureConstants.DietToolingFeature.Stories.TEST_CONNECTIVITY;
import static org.mule.runtime.ast.AllureConstants.DietToolingFeature.Stories.VALUE_PROVIDER;
import static org.mule.runtime.ast.AllureConstants.OperationExecutionFeature.OPERATION_EXECUTION_FEATURE;
import static org.mule.runtime.ast.AllureConstants.OperationExecutionFeature.Stories.EXECUTE_OPERATION;
import static org.mule.runtime.ast.api.util.AstTraversalDirection.BOTTOM_UP;
import static org.mule.runtime.ast.api.util.AstTraversalDirection.TOP_DOWN;
import static org.mule.runtime.ast.api.util.MuleAstUtils.createComponentParameterizationFromComponentAst;
import static org.mule.runtime.ast.api.util.MuleAstUtils.parameterOfType;
import static org.mule.runtime.ast.api.util.MuleAstUtils.parametersOfType;
import static org.mule.runtime.ast.api.util.MuleAstUtils.resolveOrphanComponents;
import static org.mule.runtime.ast.api.util.TestUtils.createMockParameterGroup;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;

import static org.hamcrest.CoreMatchers.instanceOf;
import static org.hamcrest.CoreMatchers.sameInstance;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.aMapWithSize;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.collection.IsEmptyCollection.empty;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsCollectionContaining.hasItem;
import static org.hamcrest.core.IsCollectionContaining.hasItems;
import static org.hamcrest.core.IsNot.not;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.stereotype.ImmutableStereotypeModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.extension.api.component.ComponentParameterization;

import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import io.qameta.allure.Feature;
import io.qameta.allure.Features;
import io.qameta.allure.Stories;
import io.qameta.allure.Story;
import org.junit.Test;

public class MuleAstUtilsTestCase {

  @Test
  public void orphanComponentsTopLevelWithChildren() {
    final ArtifactAst artifact = mock(BaseArtifactAst.class);

    final ComponentAst processor = mockBaseComponentAst();
    when(processor.directChildrenStream()).thenAnswer(inv -> Stream.empty());
    when(processor.recursiveStream()).thenAnswer(inv -> Stream.of(processor));

    final ComponentAst topLevelComponent = mockBaseComponentAst();
    when(topLevelComponent.directChildrenStream()).thenAnswer(inv -> Stream.of(processor));
    when(topLevelComponent.recursiveStream()).thenAnswer(inv -> Stream.of(topLevelComponent, processor));

    when(artifact.topLevelComponentsStream()).thenAnswer(inv -> Stream.of(topLevelComponent));
    when(artifact.recursiveStream()).thenAnswer(inv -> Stream.of(topLevelComponent, processor));
    when(artifact.filteredComponents(any())).thenCallRealMethod();

    final Set<ComponentAst> orphanComponents = resolveOrphanComponents(artifact);

    assertThat(orphanComponents, not(hasItem(topLevelComponent)));
    assertThat(orphanComponents, not(hasItem(processor)));
  }

  private ComponentAst mockBaseComponentAst() {
    ComponentAst theMock = mock(BaseComponentAst.class);
    when(theMock.getModel(ParameterizedModel.class)).thenReturn(of(mock(ParameterizedModel.class)));
    return theMock;
  }

  @Test
  public void orphanComponentsSingleProcessor() {
    final ArtifactAst artifact = mock(BaseArtifactAst.class);

    final ComponentAst orphanProcessor = mockBaseComponentAst();
    when(orphanProcessor.directChildrenStream()).thenAnswer(inv -> Stream.empty());
    when(orphanProcessor.recursiveStream()).thenAnswer(inv -> Stream.of(orphanProcessor));

    when(artifact.topLevelComponentsStream()).thenAnswer(inv -> Stream.empty());
    when(artifact.recursiveStream()).thenAnswer(inv -> Stream.of(orphanProcessor));
    when(artifact.filteredComponents(any())).thenCallRealMethod();

    final Set<ComponentAst> orphanComponents = resolveOrphanComponents(artifact);

    assertThat(orphanComponents, hasItem(orphanProcessor));
  }

  @Test
  public void orphanComponentsNestedProcessor() {
    final ArtifactAst artifact = mock(BaseArtifactAst.class);

    final ComponentAst nestedProcessor = mockBaseComponentAst();
    when(nestedProcessor.directChildrenStream()).thenAnswer(inv -> Stream.empty());
    when(nestedProcessor.recursiveStream()).thenAnswer(inv -> Stream.of(nestedProcessor));

    final ComponentAst orphanRouter = mockBaseComponentAst();
    when(orphanRouter.directChildrenStream()).thenAnswer(inv -> Stream.of(nestedProcessor));
    when(orphanRouter.recursiveStream()).thenAnswer(inv -> Stream.of(orphanRouter, nestedProcessor));

    when(artifact.topLevelComponentsStream()).thenAnswer(inv -> Stream.empty());
    when(artifact.recursiveStream()).thenAnswer(inv -> Stream.of(nestedProcessor, orphanRouter));
    when(artifact.filteredComponents(any())).thenCallRealMethod();

    final Set<ComponentAst> orphanComponents = resolveOrphanComponents(artifact);

    assertThat(orphanComponents, hasItem(orphanRouter));
    assertThat(orphanComponents, not(hasItem(nestedProcessor)));
  }

  @Test
  public void parameterOfTypeSameStereotype() {
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_OWN", "MINE", null);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    ParameterGroupModel groupModel = createMockParameterGroup(DEFAULT_GROUP_NAME, parameterModel);

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);

    final ComponentAst comp = mockBaseComponentAst();
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter(anyString(), eq("myOwnParam"))).thenReturn(parameter);

    assertThat(parameterOfType(comp, myOwnStereotype).get(), sameInstance(parameter));
  }

  @Test
  public void parameterOfTypeParentStereotype() {
    final ImmutableStereotypeModel myOwnParentStereotype = new ImmutableStereotypeModel("MY_OWN", "MINE", null);
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_PRECIOUS", "MINE", myOwnParentStereotype);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    ParameterGroupModel groupModel = createMockParameterGroup(DEFAULT_GROUP_NAME, parameterModel);

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);

    final ComponentAst comp = mockBaseComponentAst();
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter(anyString(), eq("myOwnParam"))).thenReturn(parameter);

    assertThat(parameterOfType(comp, myOwnParentStereotype).get(), sameInstance(parameter));
  }

  @Test
  public void parameterOfTypeNotMatchingStereotype() {
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_PRECIOUS", "MINE", null);
    final ImmutableStereotypeModel myOtherStereotype = new ImmutableStereotypeModel("A_SWORD", "MINE", null);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getAllParameterModels()).thenReturn(singletonList(parameterModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);

    final ComponentAst comp = mockBaseComponentAst();
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter(anyString(), eq("myOwnParam"))).thenReturn(parameter);

    assertThat(parameterOfType(comp, myOtherStereotype).isPresent(), is(false));
  }

  @Test
  public void parametersOfTypeSameStereotype() {
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_OWN", "MINE", null);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    ParameterGroupModel groupModel = createMockParameterGroup(DEFAULT_GROUP_NAME, parameterModel);

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);

    final ComponentAst comp = mockBaseComponentAst();
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter(anyString(), eq("myOwnParam"))).thenReturn(parameter);

    assertThat(parametersOfType(comp, myOwnStereotype), hasItems(sameInstance(parameter)));
  }

  @Test
  public void parametersOfTypeSameStereotypeMultiple() {
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_OWN", "MINE", null);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    final ParameterModel parameter2Model = mock(ParameterModel.class);
    when(parameter2Model.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameter2Model.getName()).thenReturn("myOwnParam2");

    ParameterGroupModel groupModel = createMockParameterGroup(DEFAULT_GROUP_NAME, parameterModel, parameter2Model);

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);
    final ComponentParameterAst parameter2 = mock(ComponentParameterAst.class);

    final ComponentAst comp = mockBaseComponentAst();
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter(anyString(), eq("myOwnParam"))).thenReturn(parameter);
    when(comp.getParameter(anyString(), eq("myOwnParam2"))).thenReturn(parameter2);

    assertThat(parametersOfType(comp, myOwnStereotype), hasItems(sameInstance(parameter), sameInstance(parameter2)));
  }

  @Test
  public void parametersOfTypeParentStereotype() {
    final ImmutableStereotypeModel myOwnParentStereotype = new ImmutableStereotypeModel("MY_OWN", "MINE", null);
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_PRECIOUS", "MINE", myOwnParentStereotype);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    final ParameterGroupModel groupModel = createMockParameterGroup(DEFAULT_GROUP_NAME, parameterModel);

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);

    final ComponentAst comp = mockBaseComponentAst();
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter(anyString(), eq("myOwnParam"))).thenReturn(parameter);

    assertThat(parametersOfType(comp, myOwnParentStereotype), hasItems(sameInstance(parameter)));
  }

  @Test
  public void parametersOfTypeNotMatchingStereotype() {
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_PRECIOUS", "MINE", null);
    final ImmutableStereotypeModel myOtherStereotype = new ImmutableStereotypeModel("A_SWORD", "MINE", null);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getAllParameterModels()).thenReturn(singletonList(parameterModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);

    final ComponentAst comp = mockBaseComponentAst();
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter(anyString(), eq("myOwnParam"))).thenReturn(parameter);

    assertThat(parametersOfType(comp, myOtherStereotype), hasSize(0));
  }

  private ComponentAst createComponentAst(ComponentAst... innerComponents) {
    ComponentAst component = mockBaseComponentAst();
    when(component.directChildrenStream()).thenAnswer(inv -> Stream.of(innerComponents));
    when(component.recursiveStream()).thenCallRealMethod();
    when(component.recursiveStream(any(AstTraversalDirection.class))).thenAnswer(inv -> {
      AstTraversalDirection direction = inv.getArgument(0);
      return StreamSupport.stream(component.recursiveSpliterator(direction), false);
    });
    when(component.recursiveSpliterator()).thenCallRealMethod();
    when(component.recursiveSpliterator(any(AstTraversalDirection.class))).thenAnswer(inv -> {
      AstTraversalDirection direction = inv.getArgument(0);
      return direction.recursiveSpliterator(component);
    });
    return component;
  }

  @Test
  public void recursiveStreamWithHierarchy() {
    ComponentAst child1110 = createComponentAst();

    ComponentAst child1111 = createComponentAst();

    ComponentAst child1112 = createComponentAst();

    ComponentAst child111 = createComponentAst(child1110, child1111, child1112);

    ComponentAst child110 = createComponentAst();

    ComponentAst child10 = createComponentAst();

    ComponentAst child11 = createComponentAst(child110, child111);

    ComponentAst child1 = createComponentAst(child10, child11);

    ComponentAst child22 = createComponentAst();

    ComponentAst child2 = createComponentAst(child22);

    ComponentAst child3 = createComponentAst();

    ArtifactAst artifact = mock(BaseArtifactAst.class);

    when(artifact.recursiveStream())
        .thenAnswer(inv -> Stream.concat(child1.recursiveStream(), child2.recursiveStream()));
    when(artifact.topLevelComponentsStream()).thenAnswer(inv -> Stream.of(child1, child2, child3));

    List<Pair<ComponentAst, List<ComponentAst>>> withHierarchy =
        MuleAstUtils.recursiveStreamWithHierarchy(artifact).collect(toList());

    assertThat(withHierarchy, hasSize(11));
    assertThat(withHierarchy.get(0).getFirst(), sameInstance(child1));
    assertThat(withHierarchy.get(0).getSecond(), empty());
    assertThat(withHierarchy.get(1).getFirst(), sameInstance(child10));
    assertThat(withHierarchy.get(1).getSecond(), hasItem(child1));
    assertThat(withHierarchy.get(2).getFirst(), sameInstance(child11));
    assertThat(withHierarchy.get(2).getSecond(), hasItem(child1));
    assertThat(withHierarchy.get(3).getFirst(), sameInstance(child110));
    assertThat(withHierarchy.get(3).getSecond(), hasItem(child11));
    assertThat(withHierarchy.get(4).getFirst(), sameInstance(child111));
    assertThat(withHierarchy.get(4).getSecond(), hasItem(child11));
    assertThat(withHierarchy.get(5).getFirst(), sameInstance(child1110));
    assertThat(withHierarchy.get(5).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(6).getFirst(), sameInstance(child1111));
    assertThat(withHierarchy.get(6).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(7).getFirst(), sameInstance(child1112));
    assertThat(withHierarchy.get(7).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(8).getFirst(), sameInstance(child2));
    assertThat(withHierarchy.get(8).getSecond(), empty());
    assertThat(withHierarchy.get(9).getFirst(), sameInstance(child22));
    assertThat(withHierarchy.get(9).getSecond(), hasItem(child2));
    assertThat(withHierarchy.get(10).getFirst(), sameInstance(child3));
    assertThat(withHierarchy.get(10).getSecond(), empty());

    withHierarchy = MuleAstUtils.recursiveStreamWithHierarchy(artifact, TOP_DOWN).collect(toList());

    assertThat(withHierarchy, hasSize(11));
    assertThat(withHierarchy.get(0).getFirst(), sameInstance(child1));
    assertThat(withHierarchy.get(0).getSecond(), empty());
    assertThat(withHierarchy.get(1).getFirst(), sameInstance(child10));
    assertThat(withHierarchy.get(1).getSecond(), hasItem(child1));
    assertThat(withHierarchy.get(2).getFirst(), sameInstance(child11));
    assertThat(withHierarchy.get(2).getSecond(), hasItem(child1));
    assertThat(withHierarchy.get(3).getFirst(), sameInstance(child110));
    assertThat(withHierarchy.get(3).getSecond(), hasItem(child11));
    assertThat(withHierarchy.get(4).getFirst(), sameInstance(child111));
    assertThat(withHierarchy.get(4).getSecond(), hasItem(child11));
    assertThat(withHierarchy.get(5).getFirst(), sameInstance(child1110));
    assertThat(withHierarchy.get(5).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(6).getFirst(), sameInstance(child1111));
    assertThat(withHierarchy.get(6).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(7).getFirst(), sameInstance(child1112));
    assertThat(withHierarchy.get(7).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(8).getFirst(), sameInstance(child2));
    assertThat(withHierarchy.get(8).getSecond(), empty());
    assertThat(withHierarchy.get(9).getFirst(), sameInstance(child22));
    assertThat(withHierarchy.get(9).getSecond(), hasItem(child2));
    assertThat(withHierarchy.get(10).getFirst(), sameInstance(child3));
    assertThat(withHierarchy.get(10).getSecond(), empty());

    withHierarchy = MuleAstUtils.recursiveStreamWithHierarchy(artifact, BOTTOM_UP).collect(toList());
    assertThat(withHierarchy, hasSize(11));
    assertThat(withHierarchy.get(0).getFirst(), sameInstance(child10));
    assertThat(withHierarchy.get(0).getSecond(), hasItem(child1));
    assertThat(withHierarchy.get(1).getFirst(), sameInstance(child110));
    assertThat(withHierarchy.get(1).getSecond(), hasItem(child11));
    assertThat(withHierarchy.get(2).getFirst(), sameInstance(child1110));
    assertThat(withHierarchy.get(2).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(3).getFirst(), sameInstance(child1111));
    assertThat(withHierarchy.get(3).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(4).getFirst(), sameInstance(child1112));
    assertThat(withHierarchy.get(4).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(5).getFirst(), sameInstance(child111));
    assertThat(withHierarchy.get(5).getSecond(), hasItem(child11));
    assertThat(withHierarchy.get(6).getFirst(), sameInstance(child11));
    assertThat(withHierarchy.get(6).getSecond(), hasItem(child1));
    assertThat(withHierarchy.get(7).getFirst(), sameInstance(child1));
    assertThat(withHierarchy.get(7).getSecond(), empty());
    assertThat(withHierarchy.get(8).getFirst(), sameInstance(child22));
    assertThat(withHierarchy.get(8).getSecond(), hasItem(child2));
    assertThat(withHierarchy.get(9).getFirst(), sameInstance(child2));
    assertThat(withHierarchy.get(9).getSecond(), empty());
    assertThat(withHierarchy.get(10).getFirst(), sameInstance(child3));
    assertThat(withHierarchy.get(10).getSecond(), empty());
  }

  @Test
  public void recursiveStreamWithHierarchyWithOrphans() {
    final ArtifactAst artifact = mock(BaseArtifactAst.class);

    final ComponentAst processor = createComponentAst();

    final ComponentAst topLevelComponent = createComponentAst(processor);

    final ComponentAst orphanProcessor = createComponentAst();

    when(artifact.topLevelComponentsStream()).thenAnswer(inv -> Stream.of(topLevelComponent));
    when(artifact.recursiveStream()).thenAnswer(inv -> Stream.of(topLevelComponent, processor, orphanProcessor));
    when(artifact.filteredComponents(any())).thenCallRealMethod();

    List<Pair<ComponentAst, List<ComponentAst>>> withHierarchy =
        MuleAstUtils.recursiveStreamWithHierarchy(artifact).collect(toList());

    assertThat(withHierarchy, hasSize(3));
    assertThat(withHierarchy.get(0).getFirst(), sameInstance(topLevelComponent));
    assertThat(withHierarchy.get(0).getSecond(), empty());
    assertThat(withHierarchy.get(1).getFirst(), sameInstance(processor));
    assertThat(withHierarchy.get(1).getSecond(), hasItem(topLevelComponent));
    assertThat(withHierarchy.get(2).getFirst(), sameInstance(orphanProcessor));
    assertThat(withHierarchy.get(2).getSecond(), not(empty()));

    withHierarchy = MuleAstUtils.recursiveStreamWithHierarchy(artifact, TOP_DOWN).collect(toList());

    assertThat(withHierarchy, hasSize(3));
    assertThat(withHierarchy.get(0).getFirst(), sameInstance(topLevelComponent));
    assertThat(withHierarchy.get(0).getSecond(), empty());
    assertThat(withHierarchy.get(1).getFirst(), sameInstance(processor));
    assertThat(withHierarchy.get(1).getSecond(), hasItem(topLevelComponent));
    assertThat(withHierarchy.get(2).getFirst(), sameInstance(orphanProcessor));
    assertThat(withHierarchy.get(2).getSecond(), not(empty()));

    withHierarchy = MuleAstUtils.recursiveStreamWithHierarchy(artifact, BOTTOM_UP).collect(toList());

    assertThat(withHierarchy, hasSize(3));
    assertThat(withHierarchy.get(0).getFirst(), sameInstance(processor));
    assertThat(withHierarchy.get(0).getSecond(), hasItem(topLevelComponent));
    assertThat(withHierarchy.get(1).getFirst(), sameInstance(topLevelComponent));
    assertThat(withHierarchy.get(1).getSecond(), empty());
    assertThat(withHierarchy.get(2).getFirst(), sameInstance(orphanProcessor));
    assertThat(withHierarchy.get(2).getSecond(), not(empty()));
  }

  @Features({@Feature(OPERATION_EXECUTION_FEATURE), @Feature(DIET_TOOLING)})
  @Stories({@Story(EXECUTE_OPERATION), @Story(TEST_CONNECTIVITY), @Story(METADATA_TYPES_RESOLUTION), @Story(VALUE_PROVIDER),
      @Story(SAMPLE_DATA)})
  @Test
  public void createAValidComponentParametrizationFromAComponentAst() {
    ComponentAst componentAst = mock(ComponentAst.class);
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    String groupName = "GROUP";
    ParameterGroupModel parameterGroupModel = mock(ParameterGroupModel.class);
    ComponentIdentifier identifier = mock(ComponentIdentifier.class);
    when(parameterGroupModel.getName()).thenReturn(groupName);

    ComponentParameterAst parameterAst = createMockedComponentParameterAst(parameterGroupModel, "param1", "value");
    ComponentParameterAst anotherParameterAst = createMockedComponentParameterAst(parameterGroupModel, "param2", "anotherValue");
    List<ComponentParameterAst> componentParameterAstList = asList(parameterAst, anotherParameterAst);
    List<ParameterGroupModel> parameterGroupModels = asList(parameterAst.getGroupModel(), anotherParameterAst.getGroupModel());

    when(componentAst.getModel(any())).thenReturn(of(parameterizedModel));
    when(parameterizedModel.getParameterGroupModels()).thenReturn(parameterGroupModels);
    when(componentAst.getParameters()).thenReturn(componentParameterAstList);
    when(componentAst.getIdentifier()).thenReturn(identifier);

    ComponentParameterization<ParameterizedModel> componentParameterization =
        createComponentParameterizationFromComponentAst(componentAst);

    assertThat(componentParameterization.getModel(), instanceOf(ParameterizedModel.class));
    assertThat(componentParameterization.getParameters(), aMapWithSize(2));
    assertThat(componentParameterization.getParameter(groupName, "param1"), is("value"));
    assertThat(componentParameterization.getParameter(groupName, "param2"), is("anotherValue"));
    assertThat(componentParameterization.getComponentIdentifier().isPresent(), is(true));
    assertThat(componentParameterization.getComponentIdentifier().get(), is(identifier));
  }

  @Features({@Feature(OPERATION_EXECUTION_FEATURE), @Feature(DIET_TOOLING)})
  @Stories({@Story(EXECUTE_OPERATION), @Story(TEST_CONNECTIVITY), @Story(METADATA_TYPES_RESOLUTION), @Story(VALUE_PROVIDER),
      @Story(SAMPLE_DATA)})
  @Test(expected = IllegalArgumentException.class)
  public void componentAstWithEmptyModelMustFail() {
    ComponentAst componentAst = mock(ComponentAst.class);
    when(componentAst.getModel(any())).thenReturn(Optional.empty());
    createComponentParameterizationFromComponentAst(componentAst);
  }

  private ComponentParameterAst createMockedComponentParameterAst(ParameterGroupModel parameterGroupModel,
                                                                  String parameterModelName,
                                                                  String value) {
    ComponentParameterAst componentParameterAst = mock(ComponentParameterAst.class);
    ParameterModel parameterModel = mock(ParameterModel.class);

    when(componentParameterAst.getGroupModel()).thenReturn(parameterGroupModel);
    when(parameterGroupModel.getParameter(parameterModelName)).thenReturn(of(parameterModel));
    when(componentParameterAst.getModel()).thenReturn(parameterModel);
    when(parameterModel.getName()).thenReturn(parameterModelName);
    when(componentParameterAst.getValue()).thenReturn(right(value));
    return componentParameterAst;
  }
}
