/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.validation;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;

import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.util.ComponentAstPredicatesFactory;

import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;

/**
 * Defines a validation for a {@link ComponentAst} within an {@link ArtifactAst}.
 * <p>
 * Runtime and modules may implement their own validations by implementing this interface.
 * <p>
 * Implementations must be stateless.
 *
 * @since 1.0
 */
public interface Validation {

  /**
   * Defines the severity of a validation failure.
   */
  public enum Level {
    /**
     * A situation that would prevent the artifact to function correctly.
     */
    ERROR,

    /**
     * A situation that even if the artifact would work correctly despite it, it would be much costlier or have some unexpected
     * effect on some other tools that rely on it.
     */
    WARN;
  }

  /**
   * @return an internal name for this validation.
   */
  String getName();

  /**
   * @return a user-friendly description of the validation.
   */
  String getDescription();

  /**
   * @return the severity level of a failure of this validation.
   */
  Level getLevel();

  /**
   * Determines if this validation has to be applied to a given component. The predicate will receive the hierarchy of the
   * component, being the component asked for the last element of the list, and the root element the first one.
   * <p>
   * Refer to {@link ComponentAstPredicatesFactory}.
   *
   * @return the predicate to determine if this validation has to be applied to a given component hierarchy.
   */
  Predicate<List<ComponentAst>> applicable();

  /**
   * Runs this validation in the provided component.
   * <p>
   * This method will only be called for components for which the {@link #applicable()} predicate was {@code true}, but only if
   * {@link #validateMany(ComponentAst, ArtifactAst)} is not implemented.
   * <p>
   * Refer to the factory methods in {@link ValidationResultItem} for creating the object to be returned.
   *
   * @param component the component under validation
   * @param artifact  the full artifact, for any context needed for the validation
   * @return {@link Optional#empty() empty} if the validation passed, or a result object describing the failure otherwise.
   */
  default Optional<ValidationResultItem> validate(ComponentAst component, ArtifactAst artifact) {
    return validateMany(component, artifact).stream().findFirst();
  }

  /**
   * Runs this validation in the provided component.
   * <p>
   * This method will only be called for components for which the {@link #applicable()} predicate was {@code true}.
   * <p>
   * Refer to the factory methods in {@link ValidationResultItem} for creating the object to be returned.
   *
   * @param component the component under validation
   * @param artifact  the full artifact, for any context needed for the validation
   * @return {@link Optional#empty() empty} if the validation passed, or a result object describing the failure otherwise.
   */
  default List<ValidationResultItem> validateMany(ComponentAst component, ArtifactAst artifact) {
    return validate(component, artifact)
        .map(r -> singletonList(r))
        .orElse(emptyList());
  }

}
