/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.validation;

import static java.util.Collections.emptyList;

import org.mule.runtime.ast.api.exception.PropertyNotFoundException;
import org.mule.runtime.ast.api.util.MuleAstUtils;

import java.util.List;
import java.util.function.Supplier;

/**
 * Determines the validations to run for a specific artifact/set of components. For instance, each extension may provide their own
 * to provide all validations for that extension.
 * <p>
 * Implementations will be discovered through SPI, using the {@link Thread#getContextClassLoader()} when
 * {@link MuleAstUtils#validate(org.mule.runtime.ast.api.ArtifactAst)} was called.
 * <p>
 * Implementations must be stateless.
 *
 * @since 1.0
 */
public interface ValidationsProvider extends Supplier<List<Validation>> {

  /**
   * @return the {@link ArtifactValidation}s to execute.
   */
  default List<ArtifactValidation> getArtifactValidations() {
    return emptyList();
  }

  /**
   * Allows for validations that check for resources in the artifact.
   * 
   * @param artifactRegionClassLoader the classLoader to use to load any resources referenced by the artifact.
   * 
   * @since 1.1
   */
  default void setArtifactRegionClassLoader(ClassLoader artifactRegionClassLoader) {
    // Nothing to do by default
  }

  /**
   * Determines whether a validation on a parameter value that has depends on a property is ignored or not.
   * <p>
   * The default behavior is to execute all validations.
   * 
   * @param ignoreParamsWithProperties {@code true} to ignore validations on parameters that have a property dependant value.
   * @since 1.1
   */
  default void setIgnoreParamsWithProperties(boolean ignoreParamsWithProperties) {
    // Nothing to do by default
  }
}
