/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.error;

import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mockito.junit.MockitoJUnit.rule;
import static org.mule.runtime.api.component.ComponentIdentifier.builder;
import static org.mule.runtime.api.meta.model.error.ErrorModelBuilder.newError;
import static org.mule.runtime.api.util.ExtensionModelTestUtils.visitableMock;
import static org.mule.runtime.ast.api.error.ErrorTypeRepositoryProvider.getCoreErrorTypeRepo;
import static org.mule.runtime.ast.internal.error.ErrorTypeRepositoryBuildingUtils.addErrorsFromExtensions;
import static org.mule.runtime.internal.dsl.DslConstants.CORE_PREFIX;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.error.ErrorModel;
import org.mule.runtime.api.meta.model.error.ErrorModelBuilder;
import org.mule.runtime.api.meta.model.operation.OperationModel;

import java.util.Optional;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.mockito.Mock;
import org.mockito.junit.MockitoRule;

public class ErrorTypeRepositoryBuildingUtilsTestCase {

  public static final String CONNECTIVITY_ERROR_IDENTIFIER = "CONNECTIVITY";
  public static final String SOURCE_RESPONSE_GENERATE_ERROR_IDENTIFIER = "SOURCE_RESPONSE_GENERATE";

  public static final ComponentIdentifier SOURCE_RESPONSE_GENERATE =
      builder().namespace(CORE_PREFIX.toUpperCase()).name(SOURCE_RESPONSE_GENERATE_ERROR_IDENTIFIER).build();

  private static final String TEST_EXTENSION_NAME = "Test Extension";
  private static final String EXTENSION_PREFIX = "test-namespace";
  private static final String ERROR_PREFIX = EXTENSION_PREFIX.toUpperCase();
  private static final String OPERATION_NAME = "operationName";
  private static final String TEST_CONNECTIVITY_ERROR_TYPE = "TEST_CONNECTIVITY";
  private static final String OAUTH_TEST_CONNECTIVITY_ERROR_TYPE = "OAUTH_CONNECTIVITY";

  private static final ErrorModel MULE_CONNECTIVITY_ERROR =
      newError(CONNECTIVITY_ERROR_IDENTIFIER, CORE_PREFIX.toUpperCase())
          .build();

  private static final ErrorModel extensionConnectivityError =
      newError(TEST_CONNECTIVITY_ERROR_TYPE, ERROR_PREFIX)
          .withParent(MULE_CONNECTIVITY_ERROR)
          .build();

  private static final ErrorModel oauthExtensionConnectivityError =
      newError(OAUTH_TEST_CONNECTIVITY_ERROR_TYPE, ERROR_PREFIX)
          .withParent(extensionConnectivityError)
          .build();

  private static final ErrorModel customErrorModel =
      newError("CUSTOM", CORE_PREFIX.toUpperCase())
          .withParent(newError(getCoreErrorTypeRepo().getAnyErrorType().getIdentifier(),
                               getCoreErrorTypeRepo().getAnyErrorType().getNamespace()).build())
          .build();

  @Rule
  public MockitoRule rule = rule().silent();

  @Mock(lenient = true)
  private ExtensionModel extensionModel;

  @Mock(lenient = true)
  private OperationModel operationWithError;

  @Mock(lenient = true)
  private OperationModel operationWithoutErrors;

  @Rule
  public ExpectedException exception = none();

  private ErrorTypeRepository typeRepository;

  @Before
  public void setUp() {
    XmlDslModel.XmlDslModelBuilder builder = XmlDslModel.builder();
    builder.setPrefix(EXTENSION_PREFIX);
    XmlDslModel xmlDslModel = builder.build();

    typeRepository = new DefaultErrorTypeRepository();

    when(extensionModel.getOperationModels()).thenReturn(asList(operationWithError, operationWithoutErrors));
    when(extensionModel.getXmlDslModel()).thenReturn(xmlDslModel);
    when(extensionModel.getName()).thenReturn(TEST_EXTENSION_NAME);

    when(operationWithError.getErrorModels()).thenReturn(singleton(extensionConnectivityError));
    when(operationWithError.getName()).thenReturn(OPERATION_NAME);
    when(operationWithError.getModelProperty(any())).thenReturn(empty());

    when(operationWithoutErrors.getName()).thenReturn("operationWithoutError");
    when(operationWithoutErrors.getErrorModels()).thenReturn(emptySet());
    when(operationWithoutErrors.getModelProperty(any())).thenReturn(empty());

    visitableMock(operationWithError, operationWithoutErrors);
  }

  @Test
  public void lookupErrorsForOperation() {
    when(extensionModel.getErrorModels()).thenReturn(singleton(extensionConnectivityError));
    addErrorsFromExtensions(singleton(extensionModel), typeRepository);

    ErrorType errorType = typeRepository.lookupErrorType(ComponentIdentifier.builder().namespace(EXTENSION_PREFIX.toUpperCase())
        .name(CONNECTIVITY_ERROR_IDENTIFIER).build()).get();

    ErrorType muleConnectivityError = errorType.getParentErrorType();
    assertThat(muleConnectivityError.getNamespace(), is(MULE_CONNECTIVITY_ERROR.getNamespace()));
    assertThat(muleConnectivityError.getIdentifier(), is(MULE_CONNECTIVITY_ERROR.getType()));

    assertThat(muleConnectivityError.getParentErrorType(), is(getCoreErrorTypeRepo().getAnyErrorType()));
  }

  @Test
  public void registerErrorTypes() {
    when(extensionModel.getErrorModels()).thenReturn(singleton(oauthExtensionConnectivityError));
    addErrorsFromExtensions(singleton(extensionModel), typeRepository);

    Optional<ErrorType> optionalOAuthType = typeRepository.lookupErrorType(builder()
        .name(OAUTH_TEST_CONNECTIVITY_ERROR_TYPE).namespace(EXTENSION_PREFIX).build());
    Optional<ErrorType> optionalConnectivityType = typeRepository.lookupErrorType(builder()
        .name(TEST_CONNECTIVITY_ERROR_TYPE).namespace(EXTENSION_PREFIX).build());

    assertThat(optionalOAuthType.isPresent(), is(true));
    assertThat(optionalConnectivityType.isPresent(), is(true));

    ErrorType parentErrorType = optionalOAuthType.get().getParentErrorType();
    assertThat(parentErrorType, is(optionalConnectivityType.get()));
  }

  @Test
  public void operationWithoutErrorsDoesntGenerateComponentMapper() {
    when(extensionModel.getOperationModels()).thenReturn(singletonList(operationWithoutErrors));

    typeRepository = spy(typeRepository);
    addErrorsFromExtensions(singleton(extensionModel), typeRepository);

    verify(typeRepository, never()).addErrorType(any(), any());
    verify(typeRepository, never()).addInternalErrorType(any(), any());
  }

  @Test
  public void operationTriesToAddInternalErrorType() {
    ErrorModel internalRepeatedError = ErrorModelBuilder.newError(SOURCE_RESPONSE_GENERATE).build();
    when(operationWithError.getErrorModels()).thenReturn(singleton(internalRepeatedError));
    when(extensionModel.getOperationModels()).thenReturn(singletonList(operationWithError));
    when(extensionModel.getErrorModels()).thenReturn(singleton(internalRepeatedError));

    addErrorsFromExtensions(singleton(extensionModel), typeRepository);

    assertThat(typeRepository.lookupErrorType(SOURCE_RESPONSE_GENERATE), is(empty()));
  }

  @Test
  public void extensionCantRegisterAMuleErrorType() {
    exception.expect(MuleRuntimeException.class);
    exception.expectMessage("The extension [" + TEST_EXTENSION_NAME
        + "] tried to register the [MULE:CUSTOM] error with [MULE] namespace, which is not allowed");
    when(operationWithError.getErrorModels()).thenReturn(singleton(customErrorModel));
    when(extensionModel.getErrorModels()).thenReturn(singleton(customErrorModel));

    addErrorsFromExtensions(singleton(extensionModel), typeRepository);
  }

}
