/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.builder;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyBoolean;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.OPERATION;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.api.meta.model.parameter.ParameterRole.BEHAVIOUR;
import static org.mule.runtime.api.util.NameUtils.hyphenize;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.AstTraversal.PARAMETER_AST;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.internal.DefaultComponentMetadataAst;
import org.mule.runtime.ast.internal.builder.BaseComponentAstBuilder;
import org.mule.runtime.ast.internal.builder.DefaultComponentAstBuilder;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.model.ExtensionModelHelper;
import org.mule.runtime.ast.internal.model.ExtensionModelHelper.ExtensionWalkerModelDelegate;
import org.mule.runtime.extension.api.declaration.type.ExtensionsTypeLoaderFactory;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

@Feature(ARTIFACT_AST)
@Story(PARAMETER_AST)
public class ComponentAstParametersTestCase {

  private static final ComponentIdentifier NS_COMP_IDENTIFIER = ComponentIdentifier.builder()
      .namespace("ns").name("comp").build();
  private static final String PARAMETER_A = "a";
  private static final String PARAMETER_B = "b";
  private static final String PARAMETER_C = "c";
  private final ClassTypeLoader TYPE_LOADER = ExtensionsTypeLoaderFactory.getDefault().createTypeLoader();

  @Rule
  public ExpectedException expectedException = none();

  private ComponentModel createParameterizedModel() {
    ComponentModel parameterizedModel = mock(ComponentModel.class);
    List<ParameterModel> parameterModels = new ArrayList<>();
    parameterModels.add(createParameterModel(PARAMETER_A));
    parameterModels.add(createParameterModel(PARAMETER_B));
    parameterModels.add(createParameterModel(PARAMETER_C));
    when(parameterizedModel.getAllParameterModels()).thenReturn(parameterModels);

    final ParameterGroupModel paramGroup = mock(ParameterGroupModel.class);
    when(paramGroup.getName()).thenReturn(DEFAULT_GROUP_NAME);
    when(paramGroup.getParameterModels()).thenReturn(parameterModels);
    when(paramGroup.isShowInDsl()).thenReturn(false);

    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(paramGroup));
    return parameterizedModel;
  }

  private ParameterModel createParameterModel(String name) {
    ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getName()).thenReturn(name);
    when(parameterModel.getType()).thenReturn(TYPE_LOADER.load(String.class));
    when(parameterModel.getRole()).thenReturn(BEHAVIOUR);
    return parameterModel;
  }

  @Test
  public void retrieveParametersAsDefinedAtExtensionModel() {
    ComponentModel parameterizedModel = createParameterizedModel();
    final BaseComponentAstBuilder baseComponentModelBuilder =
        baseComponentModelBuilder(new ExtensionModelHelper(emptySet()), NS_COMP_IDENTIFIER)
            .withParameterizedModel(parameterizedModel);

    parameterizedModel.getAllParameterModels()
        .forEach(p -> baseComponentModelBuilder.withRawParameter(p.getName(), "value-" + p.getName()));

    baseComponentModelBuilder.getGenerationInformation().withSyntax(mock(DslElementSyntax.class));

    ComponentAst componentModel = baseComponentModelBuilder.build();

    Collection<ComponentParameterAst> parameters = componentModel.getParameters();
    assertThat(parameters, not(empty()));
    assertThat(parameters.size(), is(3));

    List<String> expectedParametersName =
        parameterizedModel.getAllParameterModels().stream().map(NamedObject::getName).collect(toList());
    List<String> parametersName = parameters.stream().map(p -> p.getModel().getName()).collect(toList());
    assertThat(parametersName, is(expectedParametersName));
  }

  @Test
  @Issue("MULE-18513")
  public void paramsInShowDslGroupProperlyProcessed() {
    final ComponentAst withSpaces = paramsInShowDslGroupProperlyProcessed("GroupComponent B", "GroupComponent C");
    // Then
    Collection<ComponentParameterAst> withSpacesParameters = withSpaces.getParameters();
    assertThat(withSpacesParameters.stream().map(p -> p.getValue().getRight()).collect(toList()),
               is(asList("a", "b", "c")));
    assertThat(withSpacesParameters, hasSize(3));

    final ComponentAst camelCase = paramsInShowDslGroupProperlyProcessed("groupB", "groupC");
    // Then
    Collection<ComponentParameterAst> camelCaseParameters = camelCase.getParameters();
    assertThat(camelCaseParameters.stream().map(p -> p.getValue().getRight()).collect(toList()),
               is(asList("a", "b", "c")));
    assertThat(camelCaseParameters, hasSize(3));

    final ComponentAst singleWord = paramsInShowDslGroupProperlyProcessed("Message", "Advanced");
    // Then
    Collection<ComponentParameterAst> singleWordParameters = singleWord.getParameters();
    assertThat(singleWordParameters.stream().map(p -> p.getValue().getRight()).collect(toList()),
               is(asList("a", "b", "c")));
    assertThat(singleWordParameters, hasSize(3));
  }

  @Test
  @Issue("MULE-17711")
  public void showDslGroupNotPresentAsComponentChildren() {
    final ComponentAst withSpaces = paramsInShowDslGroupProperlyProcessed("GroupComponent B", "GroupComponent C");
    assertThat(withSpaces.directChildrenStream().count(), is(0L));

    final ComponentAst camelCase = paramsInShowDslGroupProperlyProcessed("groupB", "groupC");
    assertThat(camelCase.directChildrenStream().count(), is(0L));

    final ComponentAst singleWord = paramsInShowDslGroupProperlyProcessed("Message", "Advanced");
    assertThat(singleWord.directChildrenStream().count(), is(0L));
  }

  @Test
  @Issue("MULE-19677")
  public void showDslGroupNotPresentAsComponentChildrenWhenEmpty() {
    final ComponentAst groupEmpty = paramsInShowDslGroupProperlyProcessed("GroupComponent B", "GroupComponent C", null);
    assertThat(groupEmpty.directChildrenStream().count(), is(0L));
  }

  private ComponentAst paramsInShowDslGroupProperlyProcessed(String parameterGroupBName, String parameterGroupCName) {
    return paramsInShowDslGroupProperlyProcessed(parameterGroupBName, parameterGroupCName, "c");
  }

  private ComponentAst paramsInShowDslGroupProperlyProcessed(String parameterGroupBName, String parameterGroupCName,
                                                             String paramCValue) {
    // Given
    OperationModel parameterizedModel = createParameterizedModel(parameterGroupBName, parameterGroupCName);

    final ExtensionModel extModel = mock(ExtensionModel.class);
    final ExtensionModelHelper extModelHelper = mock(ExtensionModelHelper.class);

    final DefaultComponentAstBuilder baseCompBuilder =
        (DefaultComponentAstBuilder) baseComponentModelBuilder(extModelHelper, NS_COMP_IDENTIFIER)
            .withParameterizedModel(parameterizedModel)
            .withExtensionModel(extModel)
            .withRawParameter(PARAMETER_A, "a")
            .withRawParameter(PARAMETER_B, "b");

    final String hyphenatedParameterGroupCName = hyphenize(parameterGroupCName);
    final DslElementSyntax dslElementSyntaxComp = createCompDslSyntax(parameterGroupCName, hyphenatedParameterGroupCName);
    baseCompBuilder.getGenerationInformation().withSyntax(dslElementSyntaxComp);

    ComponentAstBuilder paramGroupCChild = baseCompBuilder.addChildComponent()
        .withIdentifier(buildFromStringRepresentation("test:" + hyphenatedParameterGroupCName))
        .withMetadata(DefaultComponentMetadataAst.builder().build());

    if (paramCValue != null) {
      paramGroupCChild = paramGroupCChild.withRawParameter(PARAMETER_C, paramCValue);
    }

    doAnswer(inv -> {
      inv.getArgument(2, ExtensionWalkerModelDelegate.class).onOperation(parameterizedModel);
      return null;
    }).when(extModelHelper).walkToComponent(eq(NS_COMP_IDENTIFIER), any(), any(), any());
    when(extModelHelper.resolveDslElementModel(eq(parameterizedModel), eq(NS_COMP_IDENTIFIER)))
        .thenReturn(dslElementSyntaxComp);
    when(extModelHelper.findComponentType(NS_COMP_IDENTIFIER)).thenReturn(OPERATION);

    // When
    return baseCompBuilder.build();
  }

  protected OperationModel createParameterizedModel(String parameterGroupBName, String parameterGroupCName) {
    OperationModel parameterizedModel = mock(OperationModel.class);

    final ParameterModel paramA = createParameterModel(PARAMETER_A);
    final ParameterModel paramB = createParameterModel(PARAMETER_B);
    final ParameterModel paramC = createParameterModel(PARAMETER_C);

    when(parameterizedModel.getAllParameterModels()).thenReturn(asList(paramA, paramB, paramC));

    final ParameterGroupModel paramGroupA = mock(ParameterGroupModel.class);
    when(paramGroupA.getName()).thenReturn(DEFAULT_GROUP_NAME);
    when(paramGroupA.getParameterModels()).thenReturn(singletonList(paramA));
    when(paramGroupA.isShowInDsl()).thenReturn(false);
    final ParameterGroupModel paramGroupB = mock(ParameterGroupModel.class);
    when(paramGroupB.getName()).thenReturn(parameterGroupBName);
    when(paramGroupB.getParameterModels()).thenReturn(singletonList(paramB));
    when(paramGroupB.isShowInDsl()).thenReturn(false);
    final ParameterGroupModel paramGroupC = mock(ParameterGroupModel.class);
    when(paramGroupC.getName()).thenReturn(parameterGroupCName);
    when(paramGroupC.getParameterModels()).thenReturn(singletonList(paramC));
    when(paramGroupC.isShowInDsl()).thenReturn(true);

    when(parameterizedModel.getParameterGroupModels()).thenReturn(asList(paramGroupA, paramGroupB, paramGroupC));
    return parameterizedModel;
  }

  protected DslElementSyntax createCompDslSyntax(String parameterGroupCName, final String hyphenatedParameterGroupCName) {
    final DslElementSyntax dslElementSyntaxComp = mock(DslElementSyntax.class);
    final DslElementSyntax dslElementSyntaxParameterGroupC = mock(DslElementSyntax.class);
    final DslElementSyntax dslElementSyntaxParamC = mock(DslElementSyntax.class);

    when(dslElementSyntaxParamC.supportsAttributeDeclaration()).thenReturn(true);

    when(dslElementSyntaxParameterGroupC.getElementName()).thenReturn(hyphenatedParameterGroupCName);
    when(dslElementSyntaxParameterGroupC.getPrefix()).thenReturn("test");
    when(dslElementSyntaxParameterGroupC.getAttribute(PARAMETER_C)).thenReturn(of(dslElementSyntaxParamC));

    when(dslElementSyntaxComp.getChild(parameterGroupCName)).thenReturn(of(dslElementSyntaxParameterGroupC));
    return dslElementSyntaxComp;
  }

  private DefaultComponentAstBuilder baseComponentModelBuilder(ExtensionModelHelper helper, ComponentIdentifier compId) {
    return (DefaultComponentAstBuilder) new DefaultComponentAstBuilder(new PropertiesResolver(),
                                                                       helper,
                                                                       emptyList(),
                                                                       0)
                                                                           .withIdentifier(compId)
                                                                           .withMetadata(DefaultComponentMetadataAst.builder()
                                                                               .build());
  }
}
