/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.error;

import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.Errors.ERROR_TYPES;

import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.mock;

import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.ast.internal.error.ErrorTypeBuilder;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import io.qameta.allure.Story;

@Story(ERROR_TYPES)
public class ErrorTypeBuilderTestCase {

  private static final String NAMESPACE = "CUSTOM";
  private static final String IDENTIFIER = "MY_ERROR";
  private static final String REPRESENTATION = "CUSTOM:MY_ERROR";

  private final ErrorTypeBuilder errorTypeBuilder = ErrorTypeBuilder.builder();
  private final ErrorType mockErrorType = mock(ErrorType.class);

  @Rule
  public ExpectedException expectedException = ExpectedException.none();

  @Test
  public void parentMustBeSet() {
    expectException("parent error type cannot be null");
    errorTypeBuilder.namespace(NAMESPACE).identifier(IDENTIFIER).build();
  }

  @Test
  public void identifierMustBeSet() {
    expectException("string representation cannot be null");
    errorTypeBuilder.namespace(NAMESPACE).parentErrorType(mockErrorType).build();
  }

  @Test
  public void namespaceMustBeSet() {
    expectException("namespace representation cannot be null");
    errorTypeBuilder.identifier(IDENTIFIER).parentErrorType(mockErrorType).build();
  }

  @Test
  public void createsExpectedTypeAndRepresentation() {
    ErrorType errorType = errorTypeBuilder.namespace(NAMESPACE).identifier(IDENTIFIER).parentErrorType(mockErrorType).build();
    assertThat(errorType.getParentErrorType(), is(mockErrorType));
    assertThat(errorType.getNamespace(), is(NAMESPACE));
    assertThat(errorType.getIdentifier(), is(IDENTIFIER));
    assertThat(errorType.toString(), is(REPRESENTATION));
  }

  private void expectException(String exceptionMessage) {
    expectedException.expect(NullPointerException.class);
    expectedException.expectMessage(containsString(exceptionMessage));
  }

}
