/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.stream;

import static java.lang.Thread.currentThread;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.mock;
import static org.mule.runtime.ast.api.util.AstTraversalDirection.BOTTOM_UP;
import static org.mule.runtime.ast.api.util.AstTraversalDirection.TOP_DOWN;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.internal.DefaultComponentMetadataAst;
import org.mule.runtime.ast.internal.builder.DefaultComponentAstBuilder;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.model.ExtensionModelHelper;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.junit.Test;

public class ComponentAstSpliteratorTestCase {

  @Test
  public void singleLevel() {
    ComponentAstBuilder rootBuilder = baseComponentBuilder("root");
    addChild(rootBuilder, "1");
    addChild(rootBuilder, "2");

    ComponentAst root = rootBuilder.build();

    assertThat(root.recursiveStream().map(c -> c.getIdentifier().getName()).collect(toList()),
               contains("root", "1", "2"));

    assertThat(root.recursiveStream(TOP_DOWN).map(c -> c.getIdentifier().getName()).collect(toList()),
               contains("root", "1", "2"));

    assertThat(root.recursiveStream(BOTTOM_UP).map(c -> c.getIdentifier().getName()).collect(toList()),
               contains("1", "2", "root"));
  }

  @Test
  public void twoLevels() {
    ComponentAstBuilder rootBuilder = baseComponentBuilder("root");

    ComponentAstBuilder child1Builder = addChild(rootBuilder, "1");
    addChild(child1Builder, "11");
    addChild(child1Builder, "12");

    ComponentAstBuilder child2Builder = addChild(rootBuilder, "2");
    addChild(child2Builder, "21");
    addChild(child2Builder, "22");

    ComponentAst root = rootBuilder.build();

    assertThat(root.recursiveStream().map(c -> c.getIdentifier().getName()).collect(toList()),
               contains("root", "1", "11", "12", "2", "21", "22"));
    assertThat(root.recursiveSpliterator().getExactSizeIfKnown(), is(7L));

    assertThat(root.recursiveStream(TOP_DOWN).map(c -> c.getIdentifier().getName()).collect(toList()),
               contains("root", "1", "11", "12", "2", "21", "22"));
    assertThat(root.recursiveSpliterator(TOP_DOWN).getExactSizeIfKnown(), is(7L));

    assertThat(root.recursiveStream(BOTTOM_UP).map(c -> c.getIdentifier().getName()).collect(toList()),
               contains("11", "12", "1", "21", "22", "2", "root"));
    assertThat(root.recursiveSpliterator(BOTTOM_UP).getExactSizeIfKnown(), is(7L));
  }

  @Test
  public void multipleLevels() {
    ComponentAstBuilder rootBuilder = baseComponentBuilder("root");

    ComponentAstBuilder child1Builder = addChild(rootBuilder, "1");

    addChild(child1Builder, "1_1");
    ComponentAstBuilder child12Builder = addChild(child1Builder, "1_2");

    addChild(child12Builder, "1_2_1");
    ComponentAstBuilder child122Builder = addChild(child12Builder, "1_2_2");

    addChild(child122Builder, "1_2_2_1");
    addChild(child122Builder, "1_2_2_2");
    addChild(child122Builder, "1_2_2_3");

    ComponentAstBuilder child2Builder = addChild(rootBuilder, "2");
    addChild(child2Builder, "2_1");

    addChild(rootBuilder, "3");

    ComponentAst root = rootBuilder.build();

    assertThat(root.recursiveStream().map(c -> c.getIdentifier().getName()).collect(toList()),
               contains("root", "1", "1_1", "1_2", "1_2_1", "1_2_2", "1_2_2_1", "1_2_2_2", "1_2_2_3", "2", "2_1", "3"));
    assertThat(root.recursiveSpliterator().getExactSizeIfKnown(), is(12L));

    assertThat(root.recursiveStream(TOP_DOWN).map(c -> c.getIdentifier().getName()).collect(toList()),
               contains("root", "1", "1_1", "1_2", "1_2_1", "1_2_2", "1_2_2_1", "1_2_2_2", "1_2_2_3", "2", "2_1", "3"));
    assertThat(root.recursiveSpliterator(TOP_DOWN).getExactSizeIfKnown(), is(12L));

    assertThat(root.recursiveStream(BOTTOM_UP).map(c -> c.getIdentifier().getName()).collect(toList()),
               contains("1_1", "1_2_1", "1_2_2_1", "1_2_2_2", "1_2_2_3", "1_2_2", "1_2", "1", "2_1", "2", "3", "root"));
    assertThat(root.recursiveSpliterator(BOTTOM_UP).getExactSizeIfKnown(), is(12L));
  }

  @Test
  public void twoLevelsDirectOnly() {
    final ComponentAstBuilder rootBuilder = baseComponentBuilder("root");

    final ComponentAstBuilder child1Builder = addChild(rootBuilder, "1");
    addChild(child1Builder, "11");
    addChild(child1Builder, "12");

    final ComponentAstBuilder child2Builder = addChild(rootBuilder, "2");
    addChild(child2Builder, "21");
    addChild(child2Builder, "22");

    final ComponentAst root = rootBuilder.build();

    final List<ComponentAst> visitedComponents = root.directChildrenStream().collect(toList());

    assertThat(visitedComponents, hasSize(2));
    assertThat(visitedComponents.stream().map(c -> c.getIdentifier().getName()).collect(toList()),
               contains("1", "2"));
  }

  @Test
  public void twoLevelsDirectOnly2() {
    final ComponentAstBuilder rootBuilder = baseComponentBuilder("root");

    final ComponentAstBuilder child1Builder = addChild(rootBuilder, "1");

    addChild(child1Builder, "11");
    ComponentAstBuilder child12Builder = addChild(child1Builder, "12");
    addChild(child12Builder, "121");

    final ComponentAstBuilder child2Builder = addChild(rootBuilder, "2");
    addChild(child2Builder, "21");
    addChild(child2Builder, "22");

    final ComponentAst root = rootBuilder.build();

    final List<ComponentAst> visitedComponents = root.directChildrenStream().collect(toList());

    assertThat(visitedComponents, hasSize(2));
    assertThat(visitedComponents.stream().map(c -> c.getIdentifier().getName()).collect(toList()),
               contains("1", "2"));
  }

  protected ComponentAstBuilder baseComponentBuilder(String name) {
    return new DefaultComponentAstBuilder(new PropertiesResolver(), new ExtensionModelHelper(emptySet()), emptyList(), 0)
        .withParameterizedModel(mock(ParameterizedModel.class))
        .withIdentifier(ComponentIdentifier.builder().namespace("ns").name(name).build())
        .withMetadata(DefaultComponentMetadataAst.builder().build());
  }

  protected ComponentAstBuilder addChild(ComponentAstBuilder builder, String name) {
    return builder.addChildComponent()
        .withParameterizedModel(mock(ParameterizedModel.class))
        .withIdentifier(ComponentIdentifier.builder().namespace("ns").name(name).build())
        .withMetadata(DefaultComponentMetadataAst.builder().build());
  }

  @Test
  public void fiveLevelsFiveChildrenEachDirectOnly() {
    final ComponentAstBuilder rootBuilder = baseComponentBuilder("root");

    addLevel(rootBuilder, 1);

    final ComponentAst root = rootBuilder.build();

    final List<Object> visitedComponents = new ArrayList<>();
    root.directChildrenStream()
        .forEach(c -> {
          visitedComponents.add(c);
        });

    assertThat(visitedComponents, hasSize(5));
  }

  @Test
  public void fiveLevelsFiveChildrenEachParallel() {
    final ComponentAstBuilder rootBuilder = baseComponentBuilder("root");

    addLevel(rootBuilder, 1);

    final ComponentAst root = rootBuilder.build();

    final List<Object> visitedComponents = new ArrayList<>();
    final Set<Thread> visitingThreads = new HashSet<>();
    root.recursiveStream().parallel().forEach(c -> {
      visitedComponents.add(c);
      visitingThreads.add(currentThread());
    });

    assertThat(visitedComponents, hasSize(3906));
  }

  protected void addLevel(final ComponentAstBuilder rootBuilder, int level) {
    for (int i = 0; i < 5; i++) {
      final ComponentAstBuilder childBuilder = addChild(rootBuilder, "" + level + "." + i);

      if (level < 5) {
        addLevel(childBuilder, level + 1);
      }
    }
  }

}
