/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.model;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.ast.internal.model.DefaultExtensionModelHelper;

import java.util.Optional;
import java.util.Set;

/**
 * Helper class to work with a set of {@link ExtensionModel}s
 * <p>
 * Contains a cache for searches within the extension models so we avoid processing each extension model twice.
 * <p>
 * It's recommended that the application only has one instance of this class to avoid processing the extension models several
 * times.
 *
 * @since 1.4
 */
public interface ExtensionModelHelper {

  /**
   * @param extensionModels the set of {@link ExtensionModel}s to work with. Usually this is the set of models configured within a
   *                        mule artifact.
   * @param dslResolvingCtx the {@link DslResolvingContext}.
   * @return the default implementation of an {@link ExtensionModelHelper}.
   */
  static ExtensionModelHelper defaultExtensionModelHelper(Set<ExtensionModel> extensionModels,
                                                          DslResolvingContext dslResolvingCtx) {
    return new DefaultExtensionModelHelper(extensionModels, dslResolvingCtx);
  }

  /**
   * Finds a {@link ConfigurationModel} within the provided set of {@link ExtensionModel}s by a {@link ComponentIdentifier}.
   *
   * @param componentId the identifier to use for the search.
   * @return the found {@link ConfigurationModel} or {@link Optional#empty()} if it couldn't be found.
   */
  Optional<ConfigurationModel> findConfigurationModel(ComponentIdentifier componentId);

}
