/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.builder;

import org.mule.api.annotation.NoImplement;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.internal.builder.LightComponentAstBuilder;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;

import java.util.Optional;
import java.util.function.Supplier;

/**
 * Provides a way of creating {@link ComponentAst} instances, configuring it and any children it may have.
 *
 * @since 1.0
 */
@NoImplement
public interface ComponentAstBuilder extends Supplier<ComponentAst> {

  /**
   * Provides a builder for {@link ComponentAst} that does not belong to an {@link ArtifactAst}.
   *
   * @return a fresh {@link ComponentAstBuilder} instance.
   */
  static ComponentAstBuilder builder() {
    return new LightComponentAstBuilder(new ParameterModelUtils());
  }

  /**
   * Adds a child component to the component being built with this builder.
   *
   * @return the builder for the newly added child component
   */
  ComponentAstBuilder addChildComponent();

  /**
   * Adds a parameter as defined in the DSL to this component builder.
   * <p>
   * When the target component is actually being {@link #build() built}, these parameter names are matched to the parameters
   * defined in the model provided by {@link #withParameterizedModel(ParameterizedModel)} to create the actual parameters map of
   * the component.
   *
   * @param paramName     the name of the parameter for which a raw value is provided.
   * @param paramRawValue the raw value of the parameter, as defined in the DSL.
   * @return this builder
   */
  ComponentAstBuilder withRawParameter(String paramName, String paramRawValue);

  /**
   * Adds a body parameter as defined in the DSL to this component builder.
   * <p>
   * When the target component is actually being {@link #build() built}, these body parameter is matched to the parameters defined
   * in the model provided by {@link #withParameterizedModel(ParameterizedModel)} to create the actual parameters map of the
   * component.
   *
   * @param parameterRawValue the raw value of the body parameter.
   * @return this builder
   */
  ComponentAstBuilder withBodyParameter(String parameterRawValue);

  /**
   * Adds an annotation as defined in the DSL to this component builder.
   * 
   * @param qName the qualified name of the annotation to be added.
   * @param value the value of the annotation.
   * @return this builder
   */
  ComponentAstBuilder withAnnotation(String name, Object value);

  /**
   *
   * @param metadata the parser metadata for this component.
   * @return this builder
   */
  ComponentAstBuilder withMetadata(ComponentMetadataAst metadata);

  /**
   *
   * @param identifier the identifier for the configuration element this object represents.
   * @return this builder
   */
  ComponentAstBuilder withIdentifier(ComponentIdentifier identifier);

  /**
   * Sets the model declaration of the target component.
   * <p>
   * This method is exclusive with {@link #withComponentModel(ComponentModel)},
   * {@link #withNestableElementModel(NestableElementModel)}, {@link #withConfigurationModel(ConfigurationModel)} and
   * {@link #withConnectionProviderModel(ConnectionProviderModel)}; only one of these may be called for a single builder.
   *
   * @param parameterizedModel the model that represents this component.
   * @return this builder
   */
  ComponentAstBuilder withParameterizedModel(ParameterizedModel parameterizedModel);

  /**
   * @return the extension model that declares the model this component represents.
   * @since 1.5
   */
  ExtensionModel getExtensionModel();

  /**
   * A component may be represented by different kinds of models, depending on the actual type of this component. Possible values
   * may be, for instance, instances of {@link ConfigurationModel}, {@link ComponentModel} or {@link ConnectionProviderModel} (but
   * not necessarily limited to those).
   *
   * @param modelClass the class of the model this method should return.
   * @return the model that represents this component.
   * @since 1.5
   */
  <M> Optional<M> getModel(Class<M> modelClass);

  /**
   * @return the identifier for the configuration element this object represents.
   * @since 1.5
   */
  ComponentIdentifier getIdentifier();

  /**
   * Builds the target component and its children.
   *
   * @return the target component
   */
  ComponentAst build();

}
