/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api;

import static java.util.Optional.empty;

import org.mule.api.annotation.NoImplement;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.util.AstTraversalDirection;
import org.mule.runtime.extension.api.dsl.syntax.resolver.DslSyntaxResolver;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.Spliterator;
import java.util.function.Predicate;
import java.util.function.UnaryOperator;
import java.util.stream.Stream;

/**
 * Represents a Mule artifact and provides access to its contents.
 *
 * @since 1.0
 */
@NoImplement
public interface ArtifactAst {

  /**
   * @return the name of the artifact represented by this object.
   * @since 1.1
   */
  String getArtifactName();

  /**
   * @return the type of the artifact represented by this object.
   * @since 1.1
   */
  ArtifactType getArtifactType();

  /**
   * The way the dependencies are calculated depends on the {@link DependencyResolutionMode} set.
   *
   * @return the {@link ExtensionModel}s this artifact depends on.
   */
  Set<ExtensionModel> dependencies();

  /**
   * Looks up the dsl syntax that was used for an extension during the ast creation.
   * 
   * @return the dsl syntax for the {@code extensionModel} if avialable, {@link Optional#empty()} if not.
   */
  default Optional<DslSyntaxResolver> dependenciesDsl(ExtensionModel extensionModel) {
    return empty();
  }

  /**
   * @return a {@link NamespaceDefinition} that represents the namespace definition of this artifact.
   */
  default NamespaceDefinition namespaceDefinition() {
    return null;
  }

  /**
   *
   * @return the {@link ArtifactAst} that is the parent of this artifact, if available.
   */
  Optional<ArtifactAst> getParent();

  /**
   * @return a {@link Stream} for this component and its children recursively that navigates the whole AST using {@link TOP_DOWN}
   *         direction.
   */
  Stream<ComponentAst> recursiveStream();

  /**
   *
   * @param direction the @{link AstTraversalDirection} used to navigate the whole AST.
   * @return a {@link Stream} for this component and its children recursively that navigates the whole AST using the given @{code
   *         direction}.
   */
  Stream<ComponentAst> recursiveStream(AstTraversalDirection direction);

  /**
   *
   * @return a {@link Spliterator} for this component and its children recursively that navigates the whole AST using
   *         {@link TOP_DOWN} direction.
   */
  Spliterator<ComponentAst> recursiveSpliterator();

  /**
   *
   * @param direction the @{link AstTraversalDirection} used to navigate the whole AST.
   * @return a {@link Spliterator} for this component and its children recursively that navigates the whole AST using the
   *         given @{code direction}.
   */
  Spliterator<ComponentAst> recursiveSpliterator(AstTraversalDirection direction);

  /**
   * @return a {@link List} for the top-level components (i.e.: configs, flows) of the artifact represented by this AST.
   */
  List<ComponentAst> topLevelComponents();

  /**
   * @return a {@link Stream} for the top-level components (i.e.: configs, flows) of the artifact represented by this AST.
   */
  Stream<ComponentAst> topLevelComponentsStream();

  /**
   *
   * @return a {@link Spliterator} for the top-level components (i.e.: configs, flows) of the artifact represented by this AST.
   */
  Spliterator<ComponentAst> topLevelComponentsSpliterator();

  /**
   * Given a {@link Predicate}, navigates the whole AST looking for all {@link ComponentAst}s satisfied by the provided
   * {@link Predicate}
   * <p>
   * Check {@link org.mule.runtime.ast.api.util.ComponentAstPredicatesFactory}, there are several {@link Predicate} already
   * defined.
   *
   * @param predicate the {@link Predicate} to test.
   * @return a {@link Stream} filtering {@link ComponentAst} that satisfied the {@code predicate}.
   */
  Stream<ComponentAst> filteredComponents(Predicate<ComponentAst> predicate);

  /**
   * Updates the mapping used to resolve property placeholders throughout the AST.
   * <p>
   * The received operator will transform a {@link String} containing keys surrounded with placeholder delimiters
   * (<code>${}</code>) to a {@link String} with those placeholders replaced with the appropriate property.
   * <p>
   * Bear in mind that a parameter resolved at one time may have a certain value, and resolving that parameter again after calling
   * this method may yield another value.
   *
   * @param newPropertiesResolver the new resolver to use for the component parameters of this artifact.
   */
  void updatePropertiesResolver(UnaryOperator<String> newPropertiesResolver);

  /**
   *
   * @return the error types
   */
  ErrorTypeRepository getErrorTypeRepository();

  /**
   * @return a collection containing the imported resources. If the imported resource contains a transitive import, it's also
   *         included in the result.
   */
  Collection<ImportedResource> getImportedResources();

  /**
   * Gets an enriched error repository. This will involve error types that are registered depending on the environment.
   *
   * @return the enriched {@code ErrorTypeRepository} instance
   */
  default ErrorTypeRepository enrichedErrorTypeRepository() {
    return getErrorTypeRepository();
  }
}
