/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.model;

import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.meta.model.util.ExtensionWalker;
import org.mule.runtime.api.meta.type.TypeCatalog;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.internal.model.DefaultExtensionModelHelper;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;
import org.mule.runtime.extension.api.dsl.syntax.resolver.DslSyntaxResolver;

import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * Helper class to work with a set of {@link ExtensionModel}s
 * <p>
 * Contains a cache for searches within the extension models so we avoid processing each extension model twice.
 * <p>
 * It's recommended that the application only has one instance of this class to avoid processing the extension models several
 * times.
 *
 * @since 1.4
 */
public interface ExtensionModelHelper {

  /**
   * @param extensionModels the set of {@link ExtensionModel}s to work with. Usually this is the set of models configured within a
   *                        mule artifact.
   * @param dslResolvingCtx the {@link DslResolvingContext}.
   * @return the default implementation of an {@link ExtensionModelHelper}.
   */
  static ExtensionModelHelper defaultExtensionModelHelper(Set<ExtensionModel> extensionModels,
                                                          DslResolvingContext dslResolvingCtx) {
    return new DefaultExtensionModelHelper(extensionModels, dslResolvingCtx);
  }

  /**
   * Finds a {@link ConfigurationModel} within the provided set of {@link ExtensionModel}s by a {@link ComponentIdentifier}.
   *
   * @param componentId the identifier to use for the search.
   * @return the found {@link ConfigurationModel} or {@link Optional#empty()} if it couldn't be found.
   */
  Optional<ConfigurationModel> findConfigurationModel(ComponentIdentifier componentId);

  /**
   * Finds a {@link ComponentModel} within the provided set of {@link ExtensionModel}s by a {@link ComponentIdentifier}.
   *
   * @param componentId the identifier to use for the search.
   * @return the found {@link ComponentModel} or {@link Optional#empty()} if it couldn't be found.
   * @since 1.5
   */
  Optional<ComponentModel> findComponentModel(ComponentIdentifier componentId);

  /**
   * Find a {@link ComponentType} for a given {@link ComponentIdentifier}
   *
   * @param componentIdentifier the identifier to use for the search.
   * @return the {@link ComponentType} associated with the configuration or an {@link Optional#empty()} if there isn't one.
   * @since 1.5
   */
  ComponentType findComponentType(ComponentIdentifier componentIdentifier);

  /**
   * Find a {@link ComponentType} for a given {@link ComponentModel}
   *
   * @param extensionComponentModel the mode to use for the search.
   * @return the {@link ComponentType} associated with the configuration or an {@link Optional#empty()} if there isn't one.
   * @since 1.5
   */
  ComponentType findComponentType(ComponentModel extensionComponentModel);

  /**
   * Find a {@link ExtensionModel} for a given {@link ComponentIdentifier}
   *
   * @param componentIdentifier the identifier to use for the search.
   * @return the {@link ExtensionModel} associated with the configuration or an {@link Optional#empty()} if there isn't one.
   * @since 1.5
   */
  Optional<ExtensionModel> lookupExtensionModelFor(ComponentIdentifier componentIdentifier);

  /**
   * Find a {@link DslElementSyntax} for a given {@link MetadataType} within the {@link ExtensionModel} for the provided namespace
   * prefix.
   *
   * @param type            the type to search the syntax for.
   * @param namespacePrefix the namespace prefix of the extension to search in.
   * @return the {@link DslElementSyntax} associated with the type or an {@link Optional#empty()} if there isn't one.
   * @since 1.5
   */
  Optional<DslElementSyntax> resolveDslElementModel(MetadataType type, String namespacePrefix);

  /**
   * Find a {@link DslElementSyntax} for a given {@link MetadataType} within the provided {@link ExtensionModel}.
   *
   * @param type           the type to search the syntax for.
   * @param extensionModel the extension to search in.
   * @return the {@link DslElementSyntax} associated with the type or an {@link Optional#empty()} if there isn't one.
   * @since 1.5
   */
  Optional<DslElementSyntax> resolveDslElementModel(MetadataType type, ExtensionModel extensionModel);

  /**
   * Find a {@link DslElementSyntax} for a given {@link NamedObject} with the provided {@link ComponentIdentifier}.
   *
   * @param component           the component to search the syntax for.
   * @param componentIdentifier the identifier to use for the search.
   * @return the {@link DslElementSyntax} associated with the component.
   * @since 1.5
   */
  DslElementSyntax resolveDslElementModel(NamedObject component, ComponentIdentifier componentIdentifier);

  /**
   * Find a {@link DslElementSyntax} for a given {@link ParameterModel} with the provided {@link ComponentIdentifier}.
   *
   * @param parameterModel      the parameterModel to search the syntax for.
   * @param componentIdentifier the identifier to use for the search.
   * @return the {@link DslElementSyntax} associated with the parameterModel.
   * @since 1.5
   */
  DslElementSyntax resolveDslElementModel(ParameterModel parameterModel, ComponentIdentifier componentIdentifier);

  /**
   * Navigates the extension model for the provided {@code componentIdentifier} and calls the corresponding method on the provided
   * {@code delegate} when found.
   * 
   * @param componentIdentifier the identifier to use for the search.
   * @param parent              an optional {@link ComponentAstBuilder} to disambiguate nested models.
   * @param delegate            the callback to execute on the found model.
   * @param currentExtension    the extension that declares the component with the given {@code componentIdentifier}.
   */
  void walkToComponent(ComponentIdentifier componentIdentifier,
                       Optional<ComponentAstBuilder> parent,
                       ExtensionWalkerModelDelegate delegate,
                       ExtensionModel currentExtension);

  /**
   * find a {@link MetadataType} for a given Java {@link Class}.
   * 
   * @param type the java class to search an associated {@link MetadataType} for.
   * @return the {@link MetadataType} associated with the Java {@link Class} or an {@link Optional#empty()} if there isn't one.
   * @since 1.5
   */
  Optional<MetadataType> findMetadataType(Class<?> type);

  /**
   * Note: Only instantiable subtypes will be returned by this method!
   * 
   * @param type the type to resolve subtypes for
   * @return the subtypes along with their {@link DslElementSyntax}, if available.
   * @since 1.5
   */
  Map<ObjectType, Optional<DslElementSyntax>> resolveSubTypes(ObjectType type);

  /**
   * Returns a {@link TypeCatalog} containing all the {@link ObjectType}s available in the current resolving context.
   * <p>
   * Any {@link ObjectType} that can be found using the {@link ExtensionModel#getTypes} method for any of the
   * {@link ExtensionModel extensions} in the context should available in the provided {@link TypeCatalog}
   *
   * @return a {@link TypeCatalog} containing all the {@link ObjectType}s available in the current resolving context.
   * @since 1.5
   */
  TypeCatalog getTypeCatalog();

  /**
   * find the DslSyntaxResolver for the given {@code extensionModel}.
   * 
   * @param extensionModel
   * @return the dsl syntax for the {@code extensionModel} if avialable, {@link Optional#empty()} if not.
   * @since 1.5
   */
  DslSyntaxResolver getDslSyntaxResolverFor(ExtensionModel extensionModel);

  /**
   * This interface is used along with an ExtensionWalker. The {@link ExtensionWalker} makes same validation/filter and then calls
   * the appropriate method form this interface if applicable.
   *
   * @since 1.5
   */
  interface ExtensionWalkerModelDelegate {

    /**
     * Handle a {@link ConfigurationModel} found in the {@link ExtensionModel} being walked.
     * 
     * @param model the model found in the {@link ExtensionModel} to handle.
     */
    void onConfiguration(ConfigurationModel model);

    /**
     * Handle a {@link ConnectionProviderModel} found in the {@link ExtensionModel} being walked.
     * 
     * @param model the model found in the {@link ExtensionModel} to handle.
     */
    void onConnectionProvider(ConnectionProviderModel model);

    /**
     * Handle a {@link OperationModel} found in the {@link ExtensionModel} being walked.
     * 
     * @param model the model found in the {@link ExtensionModel} to handle.
     */
    void onOperation(OperationModel model);

    /**
     * Handle a {@link SourceModel} found in the {@link ExtensionModel} being walked.
     * 
     * @param model the model found in the {@link ExtensionModel} to handle.
     */
    void onSource(SourceModel model);

    /**
     * Handle a {@link ConstructModel} found in the {@link ExtensionModel} being walked.
     * 
     * @param model the model found in the {@link ExtensionModel} to handle.
     */
    void onConstruct(ConstructModel model);

    /**
     * Handle a {@link NestableElementModel} found in the {@link ExtensionModel} being walked.
     * 
     * @param model the model found in the {@link ExtensionModel} to handle.
     */
    void onNestableElement(NestableElementModel model);

    /**
     * Handle a {@link MetadataType} found in the {@link ExtensionModel} being walked.
     * 
     * @param model the model found in the {@link ExtensionModel} to handle.
     */
    void onType(MetadataType type);
  }

}
