/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal;

import static java.util.Collections.unmodifiableMap;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;

import static com.github.benmanes.caffeine.cache.Caffeine.newBuilder;

import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.util.BaseComponentAst;
import org.mule.runtime.ast.internal.builder.ParameterKey;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;

import com.github.benmanes.caffeine.cache.LoadingCache;

public final class DefaultComponentAst extends BaseComponentAst {

  private final List<ComponentAst> directChildren;

  private final LoadingCache<Pair<String, String>, Optional<ComponentParameterAst>> paramResolutionCache;
  private volatile Supplier<Optional<String>> componentId;
  private volatile LazyValue<Collection<ComponentParameterAst>> allParams;

  private final Map<String, Object> annotations;

  private final ExtensionModel extensionModel;
  private final ComponentModel componentModel;
  private final NestableElementModel nestableElementModel;
  private final ConfigurationModel configurationModel;
  private final ConnectionProviderModel connectionProviderModel;
  private final ParameterizedModel parameterizedModel;
  private final MetadataType type;
  private final ComponentGenerationInformation generationInformation;

  private final ComponentMetadataAst metadata;
  private final ComponentLocation location;
  private final ComponentIdentifier identifier;
  private final ComponentType componentType;

  public DefaultComponentAst(List<ComponentAst> builtChildren,
                             Map<ParameterKey, ComponentParameterAst> parameterAsts,
                             Map<String, String> extraParams,
                             PropertiesResolver propertiesResolver,
                             Map<String, Object> annotations,
                             Supplier<Optional<String>> componentId,
                             ExtensionModel extensionModel,
                             ComponentModel componentModel, NestableElementModel nestableElementModel,
                             ConfigurationModel configurationModel, ConnectionProviderModel connectionProviderModel,
                             ParameterizedModel parameterizedModel, MetadataType type,
                             ComponentGenerationInformation generationInformation,
                             ComponentMetadataAst metadata, ComponentLocation location,
                             ComponentIdentifier identifier, ComponentType componentType) {
    this.directChildren = builtChildren;
    paramResolutionCache = initParametersCache();
    allParams = initAllParams(parameterAsts);

    resetComponentId(propertiesResolver, componentId);
    propertiesResolver.onMappingFunctionChanged(() -> resetComponentId(propertiesResolver, componentId));

    this.annotations = unmodifiableMap(annotations);

    this.extensionModel = extensionModel;
    this.componentModel = componentModel;
    this.nestableElementModel = nestableElementModel;
    this.configurationModel = configurationModel;
    this.connectionProviderModel = connectionProviderModel;
    this.parameterizedModel = parameterizedModel;
    this.type = type;
    this.generationInformation = generationInformation;

    this.metadata = metadata;
    this.location = location;
    this.identifier = identifier;
    this.componentType = componentType;
  }

  private LoadingCache<Pair<String, String>, Optional<ComponentParameterAst>> initParametersCache() {
    return newBuilder().build(key -> ofNullable(doGetParameter(getParameters(), key.getFirst(), key.getSecond())));
  }

  private void resetComponentId(PropertiesResolver propertiesResolver, Supplier<Optional<String>> componentId) {
    if (this.componentId == null || (this.componentId instanceof LazyValue && ((LazyValue) this.componentId).isComputed())) {
      this.componentId = new LazyValue<>(() -> componentId.get().map(propertiesResolver));
    }
  }

  private LazyValue<Collection<ComponentParameterAst>> initAllParams(Map<ParameterKey, ComponentParameterAst> parameterAsts) {
    return new LazyValue<>(() -> {
      if (!getModel(ParameterizedModel.class).isPresent()) {
        String id;
        if (this.getExtensionModel() != null) {
          id = this.getExtensionModel().getName() + ":" + this.toString();
        } else {
          id = "(no_ext)" + this.toString();
        }
        throw new IllegalStateException("Model for '" + id + "' (a '"
            + getModel(NamedObject.class).map(NamedObject::getName) + ")' is not parameterizable.");
      }

      return new ArrayList<>(parameterAsts.values());
    });
  }

  /////////////////////////////////////////////////////////////////////////////
  //
  // Children
  //
  /////////////////////////////////////////////////////////////////////////////

  @Override
  public List<ComponentAst> directChildren() {
    return directChildren;
  }

  /////////////////////////////////////////////////////////////////////////////
  //
  // Parameters
  //
  /////////////////////////////////////////////////////////////////////////////

  @Override
  public ComponentParameterAst getParameter(String groupName, String paramName) {
    return paramResolutionCache.get(new Pair<>(groupName, paramName)).orElse(null);
  }

  @Override
  public Collection<ComponentParameterAst> getParameters() {
    return allParams.get();
  }

  @Override
  public ExtensionModel getExtensionModel() {
    return extensionModel;
  }

  @Override
  public <M> Optional<M> getModel(Class<M> modelClass) {
    if (modelClass.isInstance(componentModel)) {
      return of(modelClass.cast(componentModel));
    }

    if (modelClass.isInstance(configurationModel)) {
      return of(modelClass.cast(configurationModel));
    }

    if (modelClass.isInstance(connectionProviderModel)) {
      return of(modelClass.cast(connectionProviderModel));
    }

    if (modelClass.isInstance(nestableElementModel)) {
      return of(modelClass.cast(nestableElementModel));
    }

    if (modelClass.isInstance(parameterizedModel)) {
      return of(modelClass.cast(parameterizedModel));
    }

    return empty();
  }

  @Override
  public MetadataType getType() {
    return type;
  }

  @Override
  public ComponentGenerationInformation getGenerationInformation() {
    return generationInformation;
  }

  /////////////////////////////////////////////////////////////////////////////
  //
  // Others
  //
  /////////////////////////////////////////////////////////////////////////////

  @Override
  public Map<String, Object> getAnnotations() {
    return annotations;
  }

  @Override
  public ComponentMetadataAst getMetadata() {
    return metadata;
  }

  @Override
  public ComponentLocation getLocation() {
    return location;
  }

  @Override
  public ComponentIdentifier getIdentifier() {
    return identifier;
  }

  @Override
  public ComponentType getComponentType() {
    return componentType;
  }

  @Override
  public Optional<String> getComponentId() {
    return componentId.get();
  }

}
