/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.builder;

import static org.mule.runtime.ast.internal.builder.ApplicationModelTypeUtils.resolveTypedComponentIdentifier;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.api.model.ExtensionModelHelper;
import org.mule.runtime.ast.internal.DefaultComponentAst;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;

public class DefaultComponentAstBuilder extends BaseComponentAstBuilder<DefaultComponentAstBuilder> {

  private final ExtensionModelHelper extModelHelper;
  private final List<DefaultComponentAstBuilder> hierarchy;

  private final int indexInParent;

  private final ComponentLocationVisitor componentLocationVisitor;

  public DefaultComponentAstBuilder(PropertiesResolver propertiesResolver, ExtensionModelHelper extModelHelper,
                                    List<DefaultComponentAstBuilder> hierarchy, int indexInParent,
                                    ComponentLocationVisitor componentLocationVisitor,
                                    ParameterModelUtils parameterModelUtils) {
    super(propertiesResolver, parameterModelUtils);
    this.extModelHelper = extModelHelper;
    this.hierarchy = hierarchy;
    this.indexInParent = indexInParent;
    this.componentLocationVisitor = componentLocationVisitor;
  }

  @Override
  public ComponentAstBuilder addChildComponent() {
    final List<DefaultComponentAstBuilder> childHierarchy = new ArrayList<>(hierarchy);
    childHierarchy.add(this);

    final DefaultComponentAstBuilder componentAstBuilder =
        new DefaultComponentAstBuilder(getPropertiesResolver(), extModelHelper, childHierarchy,
                                       (int) childComponentsStream().count(), componentLocationVisitor, getParameterModelUtils());
    addChildComponent(componentAstBuilder);
    return componentAstBuilder;
  }

  @Override
  public ComponentAstBuilder withIdentifier(ComponentIdentifier identifier) {
    extModelHelper.lookupExtensionModelFor(identifier).ifPresent(this::withExtensionModel);
    return super.withIdentifier(identifier);
  }

  @Override
  public ComponentAst get() {
    return build();
  }

  @Override
  public ComponentAst build() {
    prepareForBuild();
    return doBuild();
  }

  @Override
  public void prepareForBuild() {
    requireNonNull(getIdentifier(), "identifier");
    requireNonNull(getMetadata(), "metadata");

    if (!getModel(Object.class).isPresent()) {
      resolveTypedComponentIdentifier(this, getParent(), extModelHelper);
    }

    super.prepareForBuild();

    childComponentsStream().forEach(DefaultComponentAstBuilder::prepareForBuild);
  }

  private Optional<ComponentAstBuilder> getParent() {
    if (hierarchy.isEmpty()) {
      return empty();
    } else {
      return of(hierarchy.get(hierarchy.size() - 1));
    }
  }

  private ComponentAst doBuild() {
    final Map<ParameterKey, ComponentParameterAst> parameterAsts = createParameterAsts();
    final List<ComponentAst> builtChildren = childComponentsStream()
        .map(DefaultComponentAstBuilder::doBuild)
        .collect(toList());

    Set<String> paramNames = parameterAsts.keySet()
        .stream()
        .map(ParameterKey::getParamName)
        .collect(toSet());
    Map<String, String> extraParams = getRawParameters()
        .entrySet()
        .stream()
        .filter(e -> paramNames.contains(e.getKey()))
        .collect(toMap(Entry::getKey, Entry::getValue));

    Supplier<Optional<String>> componentId = obtainLocation(parameterAsts);
    return new DefaultComponentAst(builtChildren, parameterAsts, extraParams, getPropertiesResolver(),
                                   getAnnotations(), componentId, getExtensionModel(),
                                   getModel(ComponentModel.class).orElse(null),
                                   getModel(NestableElementModel.class).orElse(null),
                                   getModel(ConfigurationModel.class).orElse(null),
                                   getModel(ConnectionProviderModel.class).orElse(null),
                                   getModel(ParameterizedModel.class).orElse(null),
                                   getType(),
                                   getGenerationInformation().build(),
                                   getMetadata(), getLocation(), getIdentifier(), getComponentType());
  }

  @Override
  public ComponentLocation getLocation() {
    final ComponentLocation location = super.getLocation();
    if (location != null) {
      return location;
    }

    obtainLocation(createParameterAsts());

    return super.getLocation();
  }

  protected Supplier<Optional<String>> obtainLocation(final Map<ParameterKey, ComponentParameterAst> parameterAsts) {
    Supplier<Optional<String>> componentId = new LazyValue<>(() -> resolveComponentId(parameterAsts));
    setComponentId(componentId);

    componentLocationVisitor.resolveLocation(this, hierarchy);
    return componentId;
  }

  public int getIndexInParent() {
    return indexInParent;
  }

}
