/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.error;

import static org.mule.runtime.ast.internal.dsl.DslConstants.CORE_PREFIX;
import static org.mule.sdk.api.error.MuleErrors.ANY;
import static org.mule.sdk.api.error.MuleErrors.CRITICAL;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;

import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.ast.api.error.ErrorTypeBuilder;
import org.mule.runtime.ast.privileged.error.DefaultErrorType;

import java.util.Objects;

/**
 * Builder for {@link ErrorType}. This must be the only mechanism to create an instance of {@code ErrorType}.
 *
 * @since 1.0
 */
public final class DefaultErrorTypeBuilder implements ErrorTypeBuilder {

  public static final String CORE_NAMESPACE_NAME = CORE_PREFIX.toUpperCase();
  /**
   * Wild card that matches with any error and is on top of the error hierarchy for those that allow handling
   */
  public static final String ANY_IDENTIFIER = ANY.name();
  public static final String CRITICAL_IDENTIFIER = CRITICAL.name();

  private String identifier;
  private String namespace;
  private ErrorType parentErrorType;

  public static DefaultErrorTypeBuilder builder() {
    return new DefaultErrorTypeBuilder();
  }

  private DefaultErrorTypeBuilder() {}

  /**
   * Sets the error type identifier. @see {@link ErrorType#getIdentifier()}.
   *
   * The identifier must be unique within the same namespace.
   *
   * @param identifier the string representation
   * @return {@code this} builder
   */
  @Override
  public DefaultErrorTypeBuilder identifier(String identifier) {
    this.identifier = identifier;
    return this;
  }

  /**
   * Sets the error type namespace. @see {@link ErrorType#getNamespace()}
   *
   * @param namespace the error type namespace
   * @return {@code this} builder
   */
  @Override
  public DefaultErrorTypeBuilder namespace(String namespace) {
    this.namespace = namespace;
    return this;
  }

  /**
   * Sets the parent error type. @see {@link ErrorType#getParentErrorType()}
   *
   * @param parentErrorType the parent error type
   * @return {@code this} builder
   */
  @Override
  public DefaultErrorTypeBuilder parentErrorType(ErrorType parentErrorType) {
    this.parentErrorType = parentErrorType;
    return this;
  }

  /**
   * Creates a new instance of the configured error type.
   *
   * @return the error type with the provided configuration.
   */
  @Override
  public ErrorType build() {
    requireNonNull(identifier, "string representation cannot be null");
    requireNonNull(namespace, "namespace representation cannot be null");
    if (!isRoot()) {
      requireNonNull(parentErrorType, "parent error type cannot be null");
    }
    return new DefaultErrorType(identifier, namespace, parentErrorType);
  }

  private boolean isRoot() {
    return (identifier.equals(ANY_IDENTIFIER) || identifier.equals(CRITICAL_IDENTIFIER))
        && namespace.equals(CORE_NAMESPACE_NAME);
  }

  /**
   * Default and only implementation of {@link ErrorType}
   */
  // This is the previous implementaion (< 4.6). This is still here so any existing serialized instances can still be
  // deserialized.
  public static final class ErrorTypeImplementation implements ErrorType {

    private static final long serialVersionUID = -3716206147606234572L;

    private final String identifier;
    private final String namespace;
    private final ErrorType parentErrorType;

    private final String asString;

    private ErrorTypeImplementation(String identifier, String namespace, ErrorType parentErrorType) {
      this.identifier = identifier;
      this.namespace = namespace;
      this.parentErrorType = parentErrorType;

      this.asString = format("%s:%s", namespace, identifier);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getIdentifier() {
      return identifier;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getNamespace() {
      return namespace;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ErrorType getParentErrorType() {
      return parentErrorType;
    }

    @Override
    public String toString() {
      return asString;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) {
        return true;
      }
      if (o == null || getClass() != o.getClass()) {
        return false;
      }

      ErrorTypeImplementation that = (ErrorTypeImplementation) o;
      return Objects.equals(identifier, that.identifier) &&
          Objects.equals(namespace, that.namespace) &&
          Objects.equals(parentErrorType, that.parentErrorType);
    }

    @Override
    public int hashCode() {
      return Objects.hash(identifier, namespace, parentErrorType);
    }
  }

}
