/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.error;

import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import static org.mule.runtime.ast.api.error.ErrorTypeBuilder.builder;
import static org.mule.runtime.ast.internal.dsl.DslConstants.CORE_PREFIX;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.ast.api.error.ErrorTypeRepositoryProvider;

import java.util.Collection;
import java.util.Optional;
import java.util.Set;

public class TestErrorTypeRepositoryProvider implements ErrorTypeRepositoryProvider, ErrorTypeRepository {

  public static final String ANY_IDENTIFIER = "ANY";
  public static final String CRITICAL_IDENTIFIER = "CRITICAL";
  public static final String SOURCE_ERROR_IDENTIFIER = "SOURCE";
  public static final String SOURCE_RESPONSE_ERROR_IDENTIFIER = "SOURCE_RESPONSE";

  @Override
  public ErrorTypeRepository get() {
    return this;
  }

  /**
   * Error type that represents all of them that can be handled.
   */
  private static final ErrorType ANY_ERROR_TYPE =
      builder().namespace(CORE_PREFIX.toUpperCase()).identifier(ANY_IDENTIFIER).build();

  /**
   * Parent error type for errors that occur on the source of a flow.
   */
  private static final ErrorType SOURCE_ERROR_TYPE =
      builder().namespace(CORE_PREFIX.toUpperCase()).identifier(SOURCE_ERROR_IDENTIFIER)
          .parentErrorType(ANY_ERROR_TYPE).build();

  /**
   * Parent error type for errors that occur in a source processing a successful response of a flow.
   */
  private static final ErrorType SOURCE_RESPONSE_ERROR_TYPE =
      builder().namespace(CORE_PREFIX.toUpperCase()).identifier(SOURCE_RESPONSE_ERROR_IDENTIFIER)
          .parentErrorType(SOURCE_ERROR_TYPE).build();

  /**
   * Error type for which there will be no handling since it represents an error so critical it should not be handled. If such an
   * error occurs it will always be propagated. Same for it's children.
   */
  public static final ErrorType CRITICAL_ERROR_TYPE =
      builder().namespace(CORE_PREFIX.toUpperCase()).identifier(CRITICAL_IDENTIFIER)
          .parentErrorType(null).build();

  @Override
  public Collection<String> getErrorNamespaces() {
    return singleton(CORE_PREFIX.toUpperCase());
  }

  @Override
  public ErrorType getAnyErrorType() {
    return ANY_ERROR_TYPE;
  }

  @Override
  public ErrorType getSourceErrorType() {
    return SOURCE_ERROR_TYPE;
  }

  @Override
  public ErrorType getSourceResponseErrorType() {
    return SOURCE_RESPONSE_ERROR_TYPE;
  }

  @Override
  public ErrorType getCriticalErrorType() {
    return CRITICAL_ERROR_TYPE;
  }

  @Override
  public ErrorType addErrorType(ComponentIdentifier errorTypeIdentifier, ErrorType parentErrorType) {
    throw new UnsupportedOperationException();
  }

  @Override
  public ErrorType addInternalErrorType(ComponentIdentifier errorTypeIdentifier, ErrorType parentErrorType) {
    throw new UnsupportedOperationException();
  }

  @Override
  public Optional<ErrorType> lookupErrorType(ComponentIdentifier errorTypeComponentIdentifier) {
    if (errorTypeComponentIdentifier.getNamespace().equals(CORE_PREFIX.toUpperCase())) {
      return of(builder().namespace(errorTypeComponentIdentifier.getNamespace())
          .identifier(errorTypeComponentIdentifier.getName())
          .parentErrorType(ANY_ERROR_TYPE)
          .build());
    } else {
      return empty();
    }
  }

  @Override
  public Optional<ErrorType> getErrorType(ComponentIdentifier errorTypeIdentifier) {
    if (errorTypeIdentifier.getNamespace().equals(CORE_PREFIX.toUpperCase())
        && !errorTypeIdentifier.getName().equals("CUSTOM")) {
      return of(builder().namespace(errorTypeIdentifier.getNamespace())
          .identifier(errorTypeIdentifier.getName())
          .parentErrorType(ANY_ERROR_TYPE)
          .build());
    } else {
      return empty();
    }
  }

  @Override
  public Set<ErrorType> getErrorTypes() {
    return emptySet();
  }

  @Override
  public Set<ErrorType> getInternalErrorTypes() {
    return emptySet();
  }
}
