/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.util;

import static java.util.stream.StreamSupport.stream;
import static org.mule.runtime.ast.api.util.AstTraversalDirection.TOP_DOWN;
import static org.slf4j.LoggerFactory.getLogger;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;

import java.util.Collection;
import java.util.Spliterator;
import java.util.stream.Stream;

import org.slf4j.Logger;

/**
 * Base implementation of {@link ComponentAst}. It is agnostic from data, and it implements default behavior.
 *
 * @since 1.0
 */
public abstract class BaseComponentAst implements ComponentAst {

  private static final Logger LOGGER = getLogger(BaseComponentAst.class);

  @Override
  public Stream<ComponentAst> recursiveStream() {
    return recursiveStream(TOP_DOWN);
  }

  @Override
  public Spliterator<ComponentAst> recursiveSpliterator() {
    return recursiveSpliterator(TOP_DOWN);
  }

  @Override
  public Stream<ComponentAst> recursiveStream(AstTraversalDirection direction) {
    return stream(recursiveSpliterator(direction), false);
  }

  @Override
  public Spliterator<ComponentAst> recursiveSpliterator(AstTraversalDirection direction) {
    return direction.recursiveSpliterator(this);
  }

  @Override
  public Stream<ComponentAst> directChildrenStream() {
    return directChildren().stream();
  }

  @Override
  public Spliterator<ComponentAst> directChildrenSpliterator() {
    return directChildren().spliterator();
  }

  @Override
  public ComponentParameterAst getParameter(String groupName, String paramName) {
    return doGetParameter(getParameters(), groupName, paramName);
  }

  protected ComponentParameterAst doGetParameter(Collection<ComponentParameterAst> parameters, String groupName,
                                                 String paramName) {
    ComponentParameterAst found = parameters
        .stream()
        .filter(param -> matchesGroupAndName(param, groupName, paramName))
        .findFirst()
        .orElse(null);

    if (found == null && LOGGER.isDebugEnabled()) {
      parameters
          .stream()
          .filter(param -> matchesName(param, paramName))
          .findFirst()
          .ifPresent(param -> LOGGER
              .debug("Requested parameter ({}, {}) not found, but a there is a parameter with the same name: ({}, {})", groupName,
                     paramName, getGroupName(param), paramName));
    }

    return found;
  }

  /**
   * Checks if a parameter name matches with the given parameter.
   *
   * @param componentParameterAst The component parameter AST.
   * @param paramName             Parameter model name to match.
   * @return {@code true} if the name matches.
   */
  private boolean matchesName(ComponentParameterAst componentParameterAst, String paramName) {
    return componentParameterAst.getModel().getName().equals(paramName);
  }

  /**
   * Checks if a pair (group, name) matches with the given parameters.
   *
   * @param componentParameterAst The component parameter AST.
   * @param groupName             Group model name to match. The value {@code null} is interpreted as a wildcard.
   * @param paramName             Parameter model name to match.
   * @return {@code true} if the group and name match.
   */
  private boolean matchesGroupAndName(ComponentParameterAst componentParameterAst, String groupName, String paramName) {
    if (!matchesName(componentParameterAst, paramName)) {
      // name doesn't match.
      return false;
    }

    String actualGroupName = getGroupName(componentParameterAst);
    if (actualGroupName == null) {
      return true;
    }

    return actualGroupName.equals(groupName);
  }

  private String getGroupName(ComponentParameterAst componentParameterAst) {
    ParameterGroupModel groupModel = componentParameterAst.getGroupModel();
    if (groupModel == null) {
      // TODO MULE-19612: It only happens for raw parameters. Remove this if.
      return null;
    }

    return groupModel.getName();
  }

  @Override
  public String toString() {
    return getComponentId().map(n -> "" + n + "(" + getIdentifier().toString() + ")").orElse(getIdentifier().toString())
        + (getLocation() != null ? (" @ " + getLocation().getLocation()) : "");
  }
}
