/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.builder;

import static java.util.stream.Collectors.toList;

import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.internal.DefaultComponentAst;
import org.mule.runtime.ast.internal.DefaultComponentMetadataAst;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;

/**
 * Builder implementation for {@link ComponentAst} that does not belong to an {@link ArtifactAst}.
 *
 * @since 1.0
 */
public class LightComponentAstBuilder extends BaseComponentAstBuilder<LightComponentAstBuilder> {

  public LightComponentAstBuilder(ParameterModelUtils parameterModelUtils) {
    super(new PropertiesResolver(), parameterModelUtils);
  }

  public LightComponentAstBuilder(PropertiesResolver propertiesResolver, ParameterModelUtils parameterModelUtils) {
    super(propertiesResolver, parameterModelUtils);
  }

  @Override
  public ComponentAstBuilder addChildComponent() {
    final LightComponentAstBuilder componentAstBuilder =
        new LightComponentAstBuilder(getPropertiesResolver(), getParameterModelUtils());
    addChildComponent(componentAstBuilder);
    return componentAstBuilder;
  }

  @Override
  public ComponentAstBuilder withComponentModel(ComponentModel componentModel) {
    // Nothing to do
    return this;
  }

  @Override
  public ComponentAstBuilder withNestableElementModel(NestableElementModel nestableElementModel) {
    // Nothing to do
    return this;
  }

  @Override
  public ComponentAstBuilder withConfigurationModel(ConfigurationModel configurationModel) {
    // Nothing to do
    return this;
  }

  @Override
  public ComponentAstBuilder withConnectionProviderModel(ConnectionProviderModel connectionProviderModel) {
    // Nothing to do
    return this;
  }

  @Override
  public ComponentAst get() {
    return build();
  }

  @Override
  public ComponentAst build() {
    prepareForBuild();

    final Map<ParameterKey, ComponentParameterAst> parameterAsts = createParameterAsts();
    Supplier<Optional<String>> componentId = new LazyValue<>(() -> resolveComponentId(parameterAsts));
    setComponentId(componentId);

    final List<ComponentAst> builtChildren = childComponentsStream().map(BaseComponentAstBuilder::build).collect(toList());

    return new DefaultComponentAst(builtChildren, parameterAsts, getRawParameters(), new PropertiesResolver(),
                                   getAnnotations(), componentId, getExtensionModel(),
                                   getModel(ComponentModel.class).orElse(null),
                                   getModel(NestableElementModel.class).orElse(null),
                                   getModel(ConfigurationModel.class).orElse(null),
                                   getModel(ConnectionProviderModel.class).orElse(null),
                                   getModel(ParameterizedModel.class).orElse(null),
                                   getType(),
                                   getGenerationInformation().build(),
                                   resolveMetadata(), getLocation(), getIdentifier(), getComponentType());
  }

  private ComponentMetadataAst resolveMetadata() {
    if (getMetadata() != null) {
      return getMetadata();
    }

    return DefaultComponentMetadataAst.builder().build();
  }

}
