/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.util;

import static java.util.Collections.emptyList;
import static org.mule.runtime.ast.api.util.MuleAstUtils.doOnParamComponents;

import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.internal.stream.ComponentAstSpliteratorStrategy;

import java.util.ArrayList;
import java.util.List;
import java.util.Spliterator;
import java.util.function.Consumer;
import java.util.stream.Stream;

/**
 * Strategy use to navigates the whole AST
 *
 * @since 1.0.0
 */
public enum AstTraversalDirection {

  /**
   * Navigate the whole AST from top down.
   * <p>
   * e.g Given following {@link ComponentAst}:
   * <ul>
   * <li>root</li>
   * <ul>
   * <li>component-1</li>
   * <ul>
   * <li>component-1.1</li>
   * </ul>
   * <li>component-2</li>
   * <ul>
   * <li>component-2.1</li>
   * <ul>
   * <li>component-2.1.1</li>
   * </ul>
   * <li>component-2.2</li>
   * </ul>
   * </ul>
   * </ul>
   * <p>
   * Will navigate {@link ComponentAst} through:
   * <ul>
   * <li>root</li>
   * <li>component-1</li>
   * <li>component-1.1</li>
   * <li>component-2</li>
   * <li>component-2.1</li>
   * <li>component-2.1.1</li>
   * <li>component-2</li>
   * </ul>
   */
  TOP_DOWN(true) {

    @Override
    public Spliterator<ComponentAst> recursiveSpliterator(ComponentAst component) {
      return new ComponentAstSpliteratorStrategy(component, this) {

        @Override
        protected boolean tryAdvanceBeforeChildrenConsumed(Consumer<? super ComponentAst> action) {
          return tryAdvanceOverComponentAst(action);
        }

        @Override
        protected boolean tryAdvanceAfterChildrenConsumed(Consumer<? super ComponentAst> action) {
          return false;
        }
      };
    }
  },

  /**
   * Navigate the whole AST from top down.
   * <p>
   * e.g Given following {@link ComponentAst}:
   * <ul>
   * <li>root</li>
   * <ul>
   * <li>component-1</li>
   * <ul>
   * <li>component-1.1</li>
   * </ul>
   * <li>component-2</li>
   * <ul>
   * <li>component-2.1</li>
   * <ul>
   * <li>component-2.1.1</li>
   * </ul>
   * <li>component-2.2</li>
   * </ul>
   * </ul>
   * </ul>
   * <p>
   * Will navigate {@link ComponentAst} through:
   * <ul>
   * <li>component-1.1</li>
   * <li>component-1</li>
   * <li>component-2.1.1</li>
   * <li>component-2.1</li>
   * <li>component-2.2</li>
   * <li>component-2</li>
   * <li>root</li>
   * </ul>
   */
  BOTTOM_UP(false) {

    @Override
    public Spliterator<ComponentAst> recursiveSpliterator(ComponentAst component) {
      return new ComponentAstSpliteratorStrategy(component, this) {

        @Override
        protected boolean tryAdvanceBeforeChildrenConsumed(Consumer<? super ComponentAst> action) {
          return false;
        }

        @Override
        protected boolean tryAdvanceAfterChildrenConsumed(Consumer<? super ComponentAst> action) {
          return tryAdvanceOverComponentAst(action);
        }
      };
    }
  };

  private final boolean shouldEnhanceComponentBeforeChildren;

  AstTraversalDirection(boolean shouldEnhanceComponentBeforeChildren) {
    this.shouldEnhanceComponentBeforeChildren = shouldEnhanceComponentBeforeChildren;
  }

  /**
   * Creates a {@link Spliterator} for a {@link ComponentAst} that travers through this {@link ComponentAst} and its children
   * recursively by current {@link AstTraversalDirection}.
   *
   * @param component the {@link ComponentAst} to navigate.
   * @return a {@link Spliterator} for this {@code component} and its children recursively that navigates the whole AST by current
   *         {@link AstTraversalDirection}.
   */
  public abstract Spliterator<ComponentAst> recursiveSpliterator(ComponentAst component);

  /**
   * Enhances the {@link Stream} by adding hierarchy context to each {@link ComponentAst}.
   * <p>
   * The {@link Stream} is traversed in {@link AstTraversalDirection} direction
   * <p>
   * The elements of the {@link Stream} are {@link Pair}s:
   * <ul>
   * <li>the first element is the same component that would be obtained by calling {@link ArtifactAst#recursiveStream()}.</li>
   * <li>the second element is the hierarchy that was traversed to get to the first element, the first element being a root and
   * the last being the direct parent of the first element of the pair.</li>
   * </ul>
   *
   * @param components the {@link Stream} to enrich.
   * @return an enhanced {@link Stream} with hierarchy context.
   */
  public Stream<Pair<ComponentAst, List<ComponentAst>>> recursiveStreamWithHierarchy(Stream<ComponentAst> components) {
    return recursiveStreamWithHierarchy(components, emptyList());
  }

  /**
   * Enhances the {@link Stream} by adding hierarchy context to each {@link ComponentAst}.
   * <p>
   * The {@link Stream} is traversed in {@link AstTraversalDirection} direction
   * <p>
   * The elements of the {@link Stream} are {@link Pair}s:
   * <ul>
   * <li>the first element is the same component that would be obtained by calling {@link ArtifactAst#recursiveStream()}.</li>
   * <li>the second element is the hierarchy that was traversed to get to the first element, the first element being a root and
   * the last being the direct parent of the first element of the pair.</li>
   * </ul>
   *
   * @param components                   the {@link Stream} to enrich.
   * @param ignoreParameterComplexValues whether the returned {@link Stream} will contain any complex values in component
   *                                     parameters.
   * @return an enhanced {@link Stream} with hierarchy context.
   */
  public Stream<Pair<ComponentAst, List<ComponentAst>>> recursiveStreamWithHierarchy(Stream<ComponentAst> components,
                                                                                     boolean ignoreParameterComplexValues) {
    return recursiveStreamWithHierarchy(components, emptyList(), ignoreParameterComplexValues);
  }

  /**
   * Enhances the {@link Stream} by adding hierarchy context to each {@link ComponentAst}.
   * <p>
   * The {@link Stream} is traversed in {@link AstTraversalDirection} direction
   * <p>
   * The elements of the {@link Stream} are {@link Pair}s:
   * <ul>
   * <li>the first element is the same component that would be obtained by calling {@link ArtifactAst#recursiveStream()}.</li>
   * <li>the second element is the hierarchy that was traversed to get to the first element, the first element being a root and
   * the last being the direct parent of the first element of the pair.</li>
   * </ul>
   *
   * @param components      the {@link Stream} to enrich.
   * @param commonAncestors any {@link ComponentAst} hierarchy that is common for all the components in the provided
   *                        {@code components}.
   * @return an enhanced {@link Stream} with hierarchy context.
   */
  public Stream<Pair<ComponentAst, List<ComponentAst>>> recursiveStreamWithHierarchy(Stream<ComponentAst> components,
                                                                                     List<ComponentAst> commonAncestors) {
    return recursiveStreamWithHierarchy(components, commonAncestors, false);
  }

  /**
   * Enhances the {@link Stream} by adding hierarchy context to each {@link ComponentAst}.
   * <p>
   * The {@link Stream} is traversed in {@link AstTraversalDirection} direction
   * <p>
   * The elements of the {@link Stream} are {@link Pair}s:
   * <ul>
   * <li>the first element is the same component that would be obtained by calling {@link ArtifactAst#recursiveStream()}.</li>
   * <li>the second element is the hierarchy that was traversed to get to the first element, the first element being a root and
   * the last being the direct parent of the first element of the pair.</li>
   * </ul>
   *
   * @param components                   the {@link Stream} to enrich.
   * @param commonAncestors              any {@link ComponentAst} hierarchy that is common for all the components in the provided
   *                                     {@code components}.
   * @param ignoreParameterComplexValues whether the returned {@link Stream} will contain any complex values in component
   *                                     parameters.
   * @return an enhanced {@link Stream} with hierarchy context.
   */
  public Stream<Pair<ComponentAst, List<ComponentAst>>> recursiveStreamWithHierarchy(Stream<ComponentAst> components,
                                                                                     List<ComponentAst> commonAncestors,
                                                                                     boolean ignoreParameterComplexValues) {
    final List<Pair<ComponentAst, List<ComponentAst>>> enhancedComponents = new ArrayList<>();

    components.forEach(comp -> {
      if (shouldEnhanceComponentBeforeChildren) {
        enhancedComponents.add(new Pair<>(comp, new ArrayList<>(commonAncestors)));
      }

      final List<ComponentAst> currentContext = new ArrayList<>(commonAncestors);

      currentContext.add(comp);

      recursiveStreamWithHierarchy(enhancedComponents, comp, currentContext, ignoreParameterComplexValues);

      if (!shouldEnhanceComponentBeforeChildren) {
        enhancedComponents.add(new Pair<>(comp, new ArrayList<>(commonAncestors)));
      }
    });

    return enhancedComponents.stream();
  }

  private void recursiveStreamWithHierarchy(final List<Pair<ComponentAst, List<ComponentAst>>> enhancedComponents,
                                            ComponentAst component, final List<ComponentAst> currentContext,
                                            boolean ignoreParameterComplexValues) {
    Consumer<ComponentAst> action = componentInner -> {
      if (shouldEnhanceComponentBeforeChildren) {
        enhancedComponents.add(new Pair<>(componentInner,
                                          // copy the list so that the source can still be mutated
                                          new ArrayList<>(currentContext)));
      }

      final List<ComponentAst> currentContextInner = new ArrayList<>(currentContext);
      currentContextInner.add(componentInner);

      recursiveStreamWithHierarchy(enhancedComponents, componentInner, currentContextInner, ignoreParameterComplexValues);

      if (!shouldEnhanceComponentBeforeChildren) {
        enhancedComponents.add(new Pair<>(componentInner,
                                          // copy the list so that the source can still be mutated
                                          new ArrayList<>(currentContext)));
      }
    };

    component.directChildrenStream().forEach(action);
    if (!ignoreParameterComplexValues && component.getModel(ParameterizedModel.class).isPresent()) {
      component.getParameters().forEach(param -> doOnParamComponents(param, action));
    }
  }
}
