/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.stream;

import static java.util.stream.Stream.concat;
import static org.mule.runtime.ast.api.util.MuleAstUtils.doOnParamComponents;

import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.util.AstTraversalDirection;

import java.util.ArrayList;
import java.util.List;
import java.util.Spliterator;
import java.util.function.Consumer;

/**
 * A {@link ComponentAstSpliteratorStrategy} that navigates the whole AST using a {@link AstTraversalDirection} direction.
 *
 * @since 1.1.0
 */
public abstract class ComponentAstSpliteratorStrategy implements Spliterator<ComponentAst> {

  private final ComponentAst component;
  private final AstTraversalDirection direction;

  private boolean rootProcessed = false;
  private Spliterator<ComponentAst> childrenSpliterator;
  private Spliterator<ComponentAst> currentChildSpliterator;

  public ComponentAstSpliteratorStrategy(ComponentAst component, AstTraversalDirection direction) {
    this.component = component;
    this.direction = direction;
  }

  protected abstract boolean tryAdvanceBeforeChildrenConsumed(Consumer<? super ComponentAst> action);

  protected abstract boolean tryAdvanceAfterChildrenConsumed(Consumer<? super ComponentAst> action);

  protected boolean tryAdvanceOverComponentAst(Consumer<? super ComponentAst> action) {
    if (!rootProcessed) {
      rootProcessed = true;
      action.accept(component);
      return true;
    }
    return false;
  }

  @Override
  public boolean tryAdvance(Consumer<? super ComponentAst> action) {
    if (tryAdvanceBeforeChildrenConsumed(action)) {
      return true;
    }

    trySplit();

    if (currentChildSpliterator != null) {
      if (currentChildSpliterator.tryAdvance(action)) {
        return true;
      } else {
        currentChildSpliterator = null;
        return tryAdvance(action);
      }
    } else {
      if (childrenSpliterator
          .tryAdvance(componentAst -> currentChildSpliterator = componentAst.recursiveSpliterator(direction))) {
        return tryAdvance(action);
      } else {
        return tryAdvanceAfterChildrenConsumed(action);
      }
    }
  }

  @Override
  public Spliterator<ComponentAst> trySplit() {
    if (childrenSpliterator == null) {
      List<ComponentAst> paramsChildren = new ArrayList<>();

      if (component.getModel(ParameterizedModel.class).isPresent()) {
        component.getParameters()
            .forEach(param -> doOnParamComponents(param, paramsChildren::add));
      }

      childrenSpliterator = concat(component.directChildrenStream(),
                                   paramsChildren.stream())
                                       .spliterator();
    }
    return null;
  }

  @Override
  public long estimateSize() {
    return 1 + component.directChildrenStream()
        .mapToLong(inner -> inner.recursiveSpliterator(direction).estimateSize())
        .sum();
  }

  @Override
  public int characteristics() {
    return ORDERED | DISTINCT | SIZED | NONNULL | IMMUTABLE | SUBSIZED;
  }

}
