/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.model;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.collection.IsMapContaining.hasKey;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.atLeastOnce;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.reset;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.impl.DefaultObjectType;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.SubTypesModel;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.ast.api.model.ExtensionModelHelper.ExtensionWalkerModelDelegate;
import org.mule.runtime.ast.internal.builder.DefaultComponentAstBuilder;
import org.mule.runtime.ast.internal.model.DefaultExtensionModelHelper;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Description;
import io.qameta.allure.Issue;

public class DefaultExtensionModelHelperTestCase {

  private ExtensionModel extensionModel;
  private DefaultExtensionModelHelper extensionModelHelper;
  private ComponentIdentifier nestableAndConstructId;
  private ComponentIdentifier nestableFromOperationAndSourceId;
  private ComponentIdentifier parentIdentifier;
  private ComponentIdentifier operationId;
  private ComponentIdentifier sourceId;
  private ExtensionWalkerModelDelegate delegate;
  private OperationModel operationModel;
  private SourceModel sourceModel;
  private NestableElementModel nestableThatIsAlsoAConstructElementModel;
  private NestableElementModel nestableFromOperation;
  private NestableElementModel nestableFromSource;
  private ConstructModel constructModel;

  @Before
  public void setup() {
    extensionModel = mock(ExtensionModel.class);
    when(extensionModel.getName()).thenReturn("SomeTestExtensionName");

    nestableAndConstructId = mock(ComponentIdentifier.class);
    when(nestableAndConstructId.getName()).thenReturn("nestable-that-is-also-a-construct");

    nestableFromOperationAndSourceId = mock(ComponentIdentifier.class);
    when(nestableFromOperationAndSourceId.getName()).thenReturn("nestable-from-operation-and-source");

    delegate = mock(ExtensionWalkerModelDelegate.class);

    parentIdentifier = mock(ComponentIdentifier.class);
    when(parentIdentifier.getName()).thenReturn("parent-of-nestable");

    operationId = mock(ComponentIdentifier.class);
    when(operationId.getName()).thenReturn("some-operation-name");
    operationModel = mock(OperationModel.class);
    when(operationModel.getName()).thenReturn("SomeOperationName");
    when(extensionModel.getOperationModels()).thenReturn(singletonList(operationModel));

    sourceId = mock(ComponentIdentifier.class);
    when(sourceId.getName()).thenReturn("some-source-name");
    sourceModel = mock(SourceModel.class);
    when(sourceModel.getName()).thenReturn("SomeSourceName");
    when(extensionModel.getSourceModels()).thenReturn(singletonList(sourceModel));

    XmlDslModel xmlDslModel = XmlDslModel.builder().setNamespace("ns").setPrefix("prefix").build();
    when(extensionModel.getXmlDslModel()).thenReturn(xmlDslModel);

    nestableThatIsAlsoAConstructElementModel = mock(NestableElementModel.class);

    // Two nested components with the same name but different parents
    nestableFromOperation = mock(NestableElementModel.class);
    nestableFromSource = mock(NestableElementModel.class);
    when(nestableFromOperation.getName()).thenReturn("NestableFromOperationAndSource");
    when(nestableFromSource.getName()).thenReturn("NestableFromOperationAndSource");

    List operationNestedComponents = asList(nestableThatIsAlsoAConstructElementModel, nestableFromOperation);
    List sourceNestedComponents = singletonList(nestableFromSource);
    when(nestableThatIsAlsoAConstructElementModel.getName()).thenReturn("NestableThatIsAlsoAConstruct");
    when(operationModel.getNestedComponents()).thenReturn(operationNestedComponents);
    when(sourceModel.getNestedComponents()).thenReturn(sourceNestedComponents);

    constructModel = mock(ConstructModel.class);
    when(constructModel.getName()).thenReturn("NestableThatIsAlsoAConstruct");
    when(extensionModel.getConstructModel(eq("NestableThatIsAlsoAConstruct"))).thenReturn(of(constructModel));
    when(extensionModel.getConstructModels()).thenReturn(singletonList(constructModel));
  }

  @Test
  @Issue("MULE-19570")
  @Description("When a nestable and a construct have the same name, the onConstruct is called and the onNestableElement is not")
  public void whenANestableAndAConstructHaveTheSameNameTheOnConstructIsCalledAndTheOnNestableElementIsNot() {
    extensionModelHelper = new DefaultExtensionModelHelper(singleton(extensionModel));
    final DefaultComponentAstBuilder componentBuilder = mock(DefaultComponentAstBuilder.class);
    when(componentBuilder.getIdentifier()).thenReturn(parentIdentifier);

    extensionModelHelper.walkToComponent(nestableAndConstructId, of(componentBuilder), delegate, extensionModel);

    verify(delegate, never()).onNestableElement(eq(nestableThatIsAlsoAConstructElementModel));
    verify(delegate, atLeastOnce()).onConstruct(eq(constructModel));
  }

  @Test
  @Issue("W-10920842")
  public void nonInstantiableTypesFilteredFromSubTypes() {
    Set<SubTypesModel> subTypes = new HashSet<>();
    DefaultObjectType superInterface = createObjectType(false, "SuperInterface");
    DefaultObjectType subType = createObjectType(true, "SubType");
    subTypes.add(new SubTypesModel(superInterface,
                                   singleton(subType)));
    when(extensionModel.getSubTypes()).thenReturn(subTypes);
    when(extensionModel.getTypes()).thenReturn(new HashSet<>(asList(superInterface, subType)));

    extensionModelHelper = new DefaultExtensionModelHelper(singleton(extensionModel));
    Map<ObjectType, Optional<DslElementSyntax>> resolvedSubTypes = extensionModelHelper.resolveSubTypes(superInterface);

    assertThat(resolvedSubTypes, hasKey(subType));
    assertThat(resolvedSubTypes, not(hasKey(superInterface)));
  }

  @Test
  @Issue("W-11292607")
  public void nestableFromOperationAndSource() {
    extensionModelHelper = new DefaultExtensionModelHelper(singleton(extensionModel));

    // When searching the model with operation parent, the operation's nested model is found
    final DefaultComponentAstBuilder operationBuilder = mock(DefaultComponentAstBuilder.class);
    when(operationBuilder.getIdentifier()).thenReturn(operationId);
    extensionModelHelper.walkToComponent(nestableFromOperationAndSourceId, of(operationBuilder), delegate, extensionModel);
    verify(delegate, never()).onNestableElement(eq(nestableFromSource));
    verify(delegate, atLeastOnce()).onNestableElement(eq(nestableFromOperation));

    reset(delegate);
    // When searching the model with source parent, the source's nested model is found
    final DefaultComponentAstBuilder sourceBuilder = mock(DefaultComponentAstBuilder.class);
    when(sourceBuilder.getIdentifier()).thenReturn(sourceId);
    extensionModelHelper.walkToComponent(nestableFromOperationAndSourceId, of(sourceBuilder), delegate, extensionModel);
    verify(delegate, atLeastOnce()).onNestableElement(eq(nestableFromSource));
    verify(delegate, never()).onNestableElement(eq(nestableFromOperation));
  }

  protected DefaultObjectType createObjectType(boolean isInstantiable, String className) {
    Map<Class<? extends TypeAnnotation>, TypeAnnotation> extensions = new HashMap<>();
    extensions.put(TypeIdAnnotation.class, new TypeIdAnnotation(className));
    extensions.put(ClassInformationAnnotation.class,
                   new ClassInformationAnnotation(className, false, false, isInstantiable,
                                                  false, false, emptyList(),
                                                  null, emptyList(), false));

    return new DefaultObjectType(emptySet(), false, null, null, extensions);
  }
}
