/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api;

import org.mule.api.annotation.NoImplement;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.ast.api.util.AstTraversalDirection;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Spliterator;
import java.util.function.Predicate;
import java.util.stream.Stream;

/**
 * Represents the user configuration of a component (flow, config, message processor, etc) defined in an artifact.
 * <p>
 * Every {@code ComponentAst} represents the configuration of a core configuration or an extension configuration. Which
 * configuration element this object represents is identified by a {@link ComponentIdentifier} that can be retrieved using
 * {@code #getIdentifier()}.
 *
 * @since 1.0
 */
@NoImplement
public interface ComponentAst {

  /**
   * @return the identifier for the configuration element this object represents.
   */
  ComponentIdentifier getIdentifier();

  /**
   * @return a general typification of the role of this component.
   */
  ComponentType getComponentType();

  /**
   * @return the location of the component in the configuration.
   */
  ComponentLocation getLocation();

  /**
   * @return the parser metadata for this component.
   */
  ComponentMetadataAst getMetadata();

  /**
   * @return an object containing information about the generation of this component.
   */
  ComponentGenerationInformation getGenerationInformation();

  /**
   * @return the id of this component if it has one defined, or {@link Optional#empty() empty} otherwise.
   */
  Optional<String> getComponentId();

  /**
   * An annotation is defined by the DSL processor that creates the instances of {@link ComponentAst}.
   *
   * @return a {@link Map} containing the annotations for this component.
   */
  Map<String, Object> getAnnotations();

  /**
   * @return the extension model that declares the model this component represents.
   */
  ExtensionModel getExtensionModel();

  /**
   * A component may be represented by different kinds of models, depending on the actual type of this component. Possible values
   * may be, for instance, instances of {@link ConfigurationModel}, {@link ComponentModel} or {@link ConnectionProviderModel} (but
   * not necessarily limited to those).
   *
   * @param modelClass the class of the model this method should return.
   * @return the model that represents this component.
   */
  <M> Optional<M> getModel(Class<M> modelClass);

  /**
   * @return the type that represents this component, if it is represented by a type.
   */
  MetadataType getType();

  /**
   * @param groupName the name of the parameterGroup to get AST for.
   * @param paramName the name of the parameter to get AST for.
   * @return the AST of the parameter, or null if there is no parameter with the given {@code groupName} and {@code paramName}.
   */
  ComponentParameterAst getParameter(String groupName, String paramName);

  /**
   * A parameter may have a value either because it is explicitly set in the DSL, or because the extension model defines a default
   * value for it.
   *
   * @return the ASTs of all the parameters, even if they don't have a value.
   */
  Collection<ComponentParameterAst> getParameters();

  /**
   * This method will also traverse the complex parameters in each component.
   *
   * @return a {@link Stream} for this component and its children recursively that navigates the whole AST using {@link TOP_DOWN}
   *         direction.
   */
  Stream<ComponentAst> recursiveStream();

  /**
   * This method will also traverse the complex parameters in each component.
   *
   * @param direction the @{link AstTraversalDirection} used to navigate the whole AST.
   * @return a {@link Stream} for this component and its children recursively that navigates the whole AST using the given @{code
   *         direction}.
   */
  Stream<ComponentAst> recursiveStream(AstTraversalDirection direction);

  /**
   * This method will also traverse the complex parameters in each component.
   *
   * @return a {@link Spliterator} for this component and its children recursively that navigates the whole AST using
   *         {@link TOP_DOWN} direction.
   */
  Spliterator<ComponentAst> recursiveSpliterator();

  /**
   * This method will also traverse the complex parameters in each component.
   *
   * @param direction the @{link AstTraversalDirection} used to navigate the whole AST.
   * @return a {@link Spliterator} for this component and its children recursively that navigates the whole AST using the
   *         given @{code direction}.
   */
  Spliterator<ComponentAst> recursiveSpliterator(AstTraversalDirection direction);

  /**
   * This method will NOT traverse any complex parameters this component has.
   *
   * @return a {@link List} for this component's children.
   */
  List<ComponentAst> directChildren();

  /**
   * @return a {@link Stream} for this component's children.
   */
  Stream<ComponentAst> directChildrenStream();

  /**
   * This method will NOT traverse any complex parameters this component has.
   *
   * @return a {@link Spliterator} for this component's children.
   */
  Spliterator<ComponentAst> directChildrenSpliterator();

  /**
   * Returns a {@link Stream} of all the direct children which {@link ComponentAst#getIdentifier()} match the given
   * {@code namespace} and {@code name}.
   * <p>
   * Criterias are match with an AND operator, meaning that both values must match. However, setting {@code null} to any of the
   * arguments acts as a wildcard, causing that particular criteria to be ignored
   *
   * @param namespace   {@link Predicate} to match against the {@link ComponentIdentifier#getNamespace()} method
   * @param elementName {@link Predicate} to match against the {@link ComponentIdentifier#getName()} method
   * @return The direct children which identifier matches the given filters
   * @since 1.1.0
   */
  default Stream<ComponentAst> directChildrenStreamByIdentifier(String namespace, String elementName) {
    return directChildrenStream().filter(c -> {
      ComponentIdentifier identifier = c.getIdentifier();
      boolean ns = namespace != null ? namespace.equals(identifier.getNamespace()) : true;
      boolean name = elementName != null ? elementName.equals(identifier.getName()) : true;

      return ns && name;
    });
  }
}
