/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.util;

import static java.util.stream.StreamSupport.stream;
import static org.mule.runtime.ast.api.util.AstTraversalDirection.TOP_DOWN;

import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Spliterator;
import java.util.stream.Stream;

/**
 * This class provides a skeletal implementation of the {@link ComponentAst} interface, to minimize the effort required to
 * implement this interface when decorating instances of {@link ComponentAst}.
 *
 * @since 1.0
 */
public abstract class BaseComponentAstDecorator implements ComponentAst {

  private final ComponentAst decorated;

  /**
   * Creates a new decorator for the provided instance.
   *
   * @param decorated the {@link ComponentAst} instance to decorate.
   */
  public BaseComponentAstDecorator(ComponentAst decorated) {
    this.decorated = decorated;
  }

  /**
   * @return the decorated {@link ComponentAst} instance.
   */
  protected ComponentAst getDecorated() {
    return decorated;
  }

  @Override
  public Stream<ComponentAst> recursiveStream(AstTraversalDirection direction) {
    return stream(recursiveSpliterator(direction), false);
  }

  @Override
  public Stream<ComponentAst> recursiveStream() {
    return recursiveStream(TOP_DOWN);
  }

  @Override
  public Spliterator<ComponentAst> recursiveSpliterator() {
    return recursiveSpliterator(TOP_DOWN);
  }

  @Override
  public Spliterator<ComponentAst> recursiveSpliterator(AstTraversalDirection direction) {
    return direction.recursiveSpliterator(this);
  }

  @Override
  public List<ComponentAst> directChildren() {
    return decorated.directChildren();
  }

  @Override
  public Stream<ComponentAst> directChildrenStream() {
    return decorated.directChildrenStream();
  }

  @Override
  public Spliterator<ComponentAst> directChildrenSpliterator() {
    return directChildrenStream().spliterator();
  }

  @Override
  public Collection<ComponentParameterAst> getParameters() {
    return decorated.getParameters();
  }

  @Override
  public ComponentParameterAst getParameter(String groupName, String paramName) {
    return decorated.getParameter(groupName, paramName);
  }

  @Override
  public ExtensionModel getExtensionModel() {
    return decorated.getExtensionModel();
  }

  @Override
  public <M> Optional<M> getModel(Class<M> modelClass) {
    return decorated.getModel(modelClass);
  }

  @Override
  public MetadataType getType() {
    return decorated.getType();
  }

  @Override
  public Map<String, Object> getAnnotations() {
    return decorated.getAnnotations();
  }

  @Override
  public ComponentMetadataAst getMetadata() {
    return decorated.getMetadata();
  }

  @Override
  public ComponentLocation getLocation() {
    return decorated.getLocation();
  }

  @Override
  public ComponentIdentifier getIdentifier() {
    return decorated.getIdentifier();
  }

  @Override
  public ComponentType getComponentType() {
    return decorated.getComponentType();
  }

  @Override
  public Optional<String> getComponentId() {
    return decorated.getComponentId();
  }

  @Override
  public ComponentGenerationInformation getGenerationInformation() {
    return decorated.getGenerationInformation();
  }

  @Override
  public String toString() {
    return getComponentId().map(n -> "" + n + "(" + getIdentifier().toString() + ")").orElse(getIdentifier().toString())
        + (getLocation() != null ? (" @ " + getLocation().getLocation()) : "");
  }
}
