/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.builder;

import static java.util.Arrays.asList;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsIterableWithSize.iterableWithSize;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsIterableContaining.hasItem;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.hamcrest.MockitoHamcrest.argThat;

import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.CONNECTION;
import static org.mule.runtime.api.meta.ExpressionSupport.REQUIRED;
import static org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.api.meta.model.parameter.ParameterRole.BEHAVIOUR;
import static org.mule.runtime.ast.internal.builder.ApplicationModelTypeUtils.resolveTypedComponentIdentifier;
import static org.mule.runtime.ast.internal.builder.BaseComponentAstBuilder.BODY_RAW_PARAM_NAME;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.AstTraversal.PARAMETER_AST;
import static org.mule.runtime.dsl.api.xml.parser.XmlApplicationParser.IS_CDATA;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.Collections.singletonMap;

import static java.util.Optional.empty;
import static java.util.Optional.of;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.nullValue;

import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.impl.DefaultObjectType;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.ParameterDslConfiguration;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.nested.NestedRouteModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.source.SourceCallbackModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.api.model.ExtensionModelHelper.ExtensionWalkerModelDelegate;
import org.mule.runtime.ast.internal.DefaultComponentMetadataAst;
import org.mule.runtime.ast.internal.builder.ComponentLocationVisitor;
import org.mule.runtime.ast.internal.builder.DefaultComponentAstBuilder;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.model.DefaultExtensionModelHelper;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.Text;
import org.mule.runtime.extension.api.declaration.type.ExtensionsTypeLoaderFactory;
import org.mule.runtime.extension.api.property.NoWrapperModelProperty;
import org.mule.runtime.extension.api.stereotype.MuleStereotypes;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

import org.hamcrest.BaseMatcher;
import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.junit.Before;
import org.junit.Test;

import com.google.common.collect.ImmutableMap;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(PARAMETER_AST)
public class ApplicationModelTypeUtilsTestCase {

  private static final XmlDslModel XML_DSL_MODEL = XmlDslModel.builder()
      .setPrefix("mockns")
      .setNamespace("http://mockns")
      .build();

  private static final ComponentIdentifier CONFIG = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("simple-config")
      .build();

  private static final ComponentIdentifier CONN_PROVIDER = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("simple-connection-provider")
      .build();

  private static final ComponentIdentifier CONSTRUCT_CONFIG = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("construct-config")
      .build();

  private static final ComponentIdentifier OPERATION = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("simple-operation")
      .build();

  private static final ComponentIdentifier SOURCE = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("simple-source")
      .build();

  private static final ComponentIdentifier SHOW_IN_DSL_GROUP = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("param-group-in-dsl")
      .build();

  private static final ComponentIdentifier SIMPLE_POJO_PARAM_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("simple-pojo-param")
      .build();

  private static final ComponentIdentifier SIMPLE_POJO_PARAM_EXPRESSION_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("simple-pojo-param-expression")
      .build();

  private static final ComponentIdentifier SIMPLE_POJO_PARAM_IN_DSL_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("simple-pojo-param-in-dsl")
      .build();

  private static final ComponentIdentifier SIMPLE_POJO_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("simple-pojo")
      .build();

  private static final ComponentIdentifier SIMPLE_TEXT_POJO_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("simple-text-pojo")
      .build();

  private static final ComponentIdentifier SIMPLE_TEXT_PARAM_POJO_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("pojo-text-param")
      .build();

  private static final ComponentIdentifier COMPLEX_POJO_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("complex-pojo")
      .build();

  private static final ComponentIdentifier LIST_POJO_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("list-pojo")
      .build();

  private static final ComponentIdentifier LIST_TEXT_POJO_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("list-text-pojo")
      .build();

  private static final ComponentIdentifier SIMPLES_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("simples")
      .build();

  private static final ComponentIdentifier MAP_POJO_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("map-pojo")
      .build();

  private static final ComponentIdentifier SIMPLES_BY_KEY_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("by-key-simples")
      .build();

  private static final ComponentIdentifier SIMPLE_BY_KEY_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("by-key-simple")
      .build();

  private static final ComponentIdentifier MAP_STRING_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("map-string")
      .build();

  private static final ComponentIdentifier STRINGS_BY_KEY_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("by-key-strings")
      .build();

  private static final ComponentIdentifier STRING_BY_KEY_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("by-key-string")
      .build();

  private static final ComponentIdentifier SIMPLE_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("simple")
      .build();

  private static final ComponentIdentifier LIST_OF_ROUTES_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("list-of-routes")
      .build();
  private static final ComponentIdentifier ITEM_ROUTER = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("item-router")
      .build();

  private ExtensionModel extensionModel;
  private OperationModel operationModel;
  private NestedRouteModel nestedComponentModel;
  private SourceModel sourceModel;
  private DefaultExtensionModelHelper extModelHelper;
  private DefaultComponentAstBuilder mockParentAstBuilder;

  @Before
  public void before() {
    final ClassTypeLoader typeLoader = ExtensionsTypeLoaderFactory.getDefault()
        .createTypeLoader(ApplicationModelTypeUtilsTestCase.class.getClassLoader());

    extensionModel = mock(ExtensionModel.class);
    when(extensionModel.getXmlDslModel()).thenReturn(XML_DSL_MODEL);

    operationModel = mock(OperationModel.class);

    nestedComponentModel = mock(NestedRouteModel.class);
    when(nestedComponentModel.getMaxOccurs())
        .thenReturn(of(Integer.MAX_VALUE));
    List nestedComponents = singletonList(mock(NestableElementModel.class));
    when(nestedComponentModel.getNestedComponents())
        .thenReturn(nestedComponents);

    sourceModel = mock(SourceModel.class);

    ComponentIdentifier parentIdentifier = buildFromStringRepresentation("parent");
    mockParentAstBuilder = mock(DefaultComponentAstBuilder.class);
    when(mockParentAstBuilder.getIdentifier()).thenReturn(parentIdentifier);

    final MetadataType paramType = typeLoader.load(SimplePojo.class);

    final ParameterGroupModel paramGroup = mock(ParameterGroupModel.class);
    final ParameterGroupModel paramGroupInDsl = mock(ParameterGroupModel.class);
    final String paramName = "simplePojoParam";
    final String paramExpressionName = "simplePojoParamExpression";
    final String paramNameInDsl = "simplePojoParamInDsl";
    final ParameterModel simplePojoParam = createParameter(paramType, paramName);
    final ParameterModel simplePojoParamExpression = createOnlyExpressionParameter(paramType, paramExpressionName);
    final ParameterModel simplePojoParamInDsl = createParameter(paramType, paramNameInDsl);

    when(paramGroup.getParameter(paramName)).thenReturn(of(simplePojoParam));
    when(paramGroup.getParameter(paramExpressionName)).thenReturn(of(simplePojoParamExpression));
    when(paramGroup.getParameterModels()).thenReturn(asList(simplePojoParam, simplePojoParamExpression));
    when(paramGroup.getName()).thenReturn("paramGroup");
    when(paramGroupInDsl.getName()).thenReturn("paramGroupInDsl");
    when(paramGroupInDsl.isShowInDsl()).thenReturn(true);
    when(paramGroupInDsl.getName()).thenReturn("paramGroupInDsl");
    when(paramGroupInDsl.getParameter(paramNameInDsl)).thenReturn(of(simplePojoParamInDsl));
    when(paramGroupInDsl.getParameterModels()).thenReturn(singletonList(simplePojoParamInDsl));
    when(operationModel.getName()).thenReturn("simpleOperation");
    when(operationModel.getParameterGroupModels()).thenReturn(asList(paramGroup, paramGroupInDsl));
    when(operationModel.getAllParameterModels())
        .thenReturn(asList(simplePojoParam, simplePojoParamExpression, simplePojoParamInDsl));

    when(nestedComponentModel.getName()).thenReturn("listOfRoutesRouter");

    final SourceCallbackModel sourceSuccessCallbackModel = mock(SourceCallbackModel.class);
    when(sourceSuccessCallbackModel.getParameterGroupModels()).thenReturn(asList(paramGroup, paramGroupInDsl));
    when(sourceSuccessCallbackModel.getAllParameterModels())
        .thenReturn(asList(simplePojoParam, simplePojoParamExpression, simplePojoParamInDsl));

    when(sourceModel.getName()).thenReturn("simpleSource");
    when(sourceModel.getSuccessCallback()).thenReturn(of(sourceSuccessCallbackModel));
    // emulate the behavior of the default implementation from SourceModel
    when(sourceModel.getAllParameterModels())
        .thenReturn(asList(simplePojoParam, simplePojoParamExpression, simplePojoParamInDsl));

    when(extensionModel.getName()).thenReturn("mockExtension");

    when(extensionModel.getOperationModels()).thenReturn(singletonList(operationModel));
    when(extensionModel.getOperationModel("simpleOperation")).thenReturn(of(operationModel));

    when(extensionModel.getSourceModels()).thenReturn(singletonList(sourceModel));
    when(extensionModel.getSourceModel("simpleSource")).thenReturn(of(sourceModel));

    when(extensionModel.getTypes()).thenReturn(new HashSet<>(asList((ObjectType) paramType,
                                                                    (ObjectType) typeLoader.load(SimpleTextPojo.class),
                                                                    (ObjectType) typeLoader.load(ComplexPojo.class),
                                                                    (ObjectType) typeLoader.load(ListPojo.class),
                                                                    (ObjectType) typeLoader.load(ListTextPojo.class),
                                                                    (ObjectType) typeLoader.load(MapPojo.class),
                                                                    (ObjectType) typeLoader.load(MapString.class))));

    extModelHelper = spy(new DefaultExtensionModelHelper(singleton(extensionModel)));

    doAnswer(inv -> {
      final ComponentIdentifier identifier = inv.getArgument(0);
      final ExtensionWalkerModelDelegate walker = inv.getArgument(2);

      if (identifier.equals(CONFIG)) {
        ConfigurationModel configModel = mock(ConfigurationModel.class);
        when(configModel.getName()).thenReturn("simpleConfig");
        walker.onConfiguration(configModel);
      } else if (identifier.equals(CONN_PROVIDER)) {
        ConnectionProviderModel connProviderModel = mock(ConnectionProviderModel.class);
        when(connProviderModel.getName()).thenReturn("simpleConnectionProvider");
        walker.onConnectionProvider(connProviderModel);
      } else if (identifier.equals(CONSTRUCT_CONFIG)) {
        ConstructModel configModel = mock(ConstructModel.class);
        when(configModel.getName()).thenReturn("constructConfig");
        when(configModel.getStereotype()).thenReturn(MuleStereotypes.CONFIG);
        walker.onConstruct(configModel);
      } else if (identifier.equals(OPERATION)) {
        walker.onOperation(operationModel);
      } else if (identifier.equals(SOURCE)) {
        walker.onSource(sourceModel);
      } else if (identifier.equals(SIMPLE_POJO_ID)) {
        walker.onType(paramType);
      } else if (identifier.equals(SIMPLE_TEXT_POJO_ID)) {
        walker.onType(typeLoader.load(SimpleTextPojo.class));
      } else if (identifier.equals(COMPLEX_POJO_ID)) {
        walker.onType(typeLoader.load(ComplexPojo.class));
      } else if (identifier.equals(LIST_POJO_ID)) {
        walker.onType(typeLoader.load(ListPojo.class));
      } else if (identifier.equals(LIST_TEXT_POJO_ID)) {
        walker.onType(typeLoader.load(ListTextPojo.class));
      } else if (identifier.equals(MAP_POJO_ID)) {
        walker.onType(typeLoader.load(MapPojo.class));
      } else if (identifier.equals(MAP_STRING_ID)) {
        walker.onType(typeLoader.load(MapString.class));
      } else if (identifier.equals(SIMPLE_ID)) {
        walker.onType(paramType);
      } else if (identifier.equals(LIST_OF_ROUTES_ID)) {
        walker.onNestableElement(nestedComponentModel);
      }

      return null;
    })
        .when(extModelHelper).walkToComponent(any(), any(), any(), eq(extensionModel));
  }

  private ParameterModel createParameter(final MetadataType paramType, final String paramName) {
    final ParameterModel simplePojoParam = mock(ParameterModel.class);
    when(simplePojoParam.getName()).thenReturn(paramName);
    when(simplePojoParam.getType()).thenReturn(paramType);
    when(simplePojoParam.getRole()).thenReturn(BEHAVIOUR);
    when(simplePojoParam.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder()
        .allowsInlineDefinition(true)
        .allowsReferences(false)
        .allowTopLevelDefinition(false)
        .build());
    return simplePojoParam;
  }

  private ParameterModel createOnlyExpressionParameter(final MetadataType paramType, final String paramName) {
    final ParameterModel simplePojoParam = mock(ParameterModel.class);
    when(simplePojoParam.getName()).thenReturn(paramName);
    when(simplePojoParam.getType()).thenReturn(paramType);
    when(simplePojoParam.getRole()).thenReturn(BEHAVIOUR);
    when(simplePojoParam.getExpressionSupport()).thenReturn(REQUIRED);
    when(simplePojoParam.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder()
        .allowsInlineDefinition(true)
        .allowsReferences(false)
        .allowTopLevelDefinition(false)
        .build());
    return simplePojoParam;
  }

  @Test
  public void simpleParamInSimplePojo() {
    final DefaultComponentAstBuilder component = spy(createModel(SIMPLE_POJO_ID,
                                                                 singletonMap("pojoSimpleParam", "value")));

    resolveTypedComponentIdentifier(component, of(mockParentAstBuilder), extModelHelper);

    verify(component).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                    argThat(new ParameterGroupValueModelMatcher("SimplePojo")),
                                    argThat(new ParameterRawValueModelMatcher("value")),
                                    argThat(is(empty())));
    verify(component).withParameterizedModel(any());
  }

  @Test
  public void simpleTextParamInSimpleTextPojo() {
    final DefaultComponentAstBuilder textComponent = spy(createModel(SIMPLE_TEXT_PARAM_POJO_ID,
                                                                     emptyMap(),
                                                                     "myTextBody"));
    final DefaultComponentAstBuilder component = spy(createModel(SIMPLE_TEXT_POJO_ID,
                                                                 emptyMap(),
                                                                 singletonList(textComponent)));

    resolveTypedComponentIdentifier(component, of(mockParentAstBuilder), extModelHelper);

    verify(component).withParameter(argThat(new ParameterModelMatcher("pojoTextParam")),
                                    argThat(new ParameterGroupValueModelMatcher("SimpleTextPojo")),
                                    argThat(new ParameterRawValueModelMatcher("myTextBody")),
                                    argThat(is(of(SIMPLE_TEXT_PARAM_POJO_ID))));
    verify(component).withParameterizedModel(any());
  }

  @Test
  @Issue("MULE-19571")
  public void simpleTextParamInSimpleTextPojoInArray() {
    final DefaultComponentAstBuilder textComponent = spy(createModel(SIMPLE_TEXT_PARAM_POJO_ID,
                                                                     emptyMap(),
                                                                     "myTextBody"));
    final DefaultComponentAstBuilder item = spy(createModel(SIMPLE_TEXT_POJO_ID,
                                                            emptyMap(),
                                                            singletonList(textComponent)));

    final DefaultComponentAstBuilder wrapper = spy(createModel(SIMPLES_ID,
                                                               emptyMap(),
                                                               singletonList(item)));
    final DefaultComponentAstBuilder component = spy(createModel(LIST_TEXT_POJO_ID,
                                                                 emptyMap(),
                                                                 singletonList(wrapper)));

    resolveTypedComponentIdentifier(component, of(mockParentAstBuilder), extModelHelper);

    verify(item).withParameter(argThat(new ParameterModelMatcher("pojoTextParam")),
                               argThat(new ParameterGroupValueModelMatcher("SimpleTextPojo")),
                               argThat(new ParameterRawValueModelMatcher("myTextBody")),
                               argThat(is(of(SIMPLE_TEXT_PARAM_POJO_ID))));
    verify(item).withParameterizedModel(any());
  }

  @Test
  public void simpleEmptyTextParamInSimpleTextPojo() {
    final DefaultComponentAstBuilder textComponent = spy(createModel(SIMPLE_TEXT_PARAM_POJO_ID,
                                                                     emptyMap(),
                                                                     emptyList()));
    final DefaultComponentAstBuilder component = spy(createModel(SIMPLE_TEXT_POJO_ID,
                                                                 emptyMap(),
                                                                 singletonList(textComponent)));

    resolveTypedComponentIdentifier(component, of(mockParentAstBuilder), extModelHelper);

    verify(component).withParameter(argThat(new ParameterModelMatcher("pojoTextParam")),
                                    argThat(new ParameterGroupValueModelMatcher("SimpleTextPojo")),
                                    argThat(new ParameterRawValueModelMatcher(nullValue())),
                                    argThat(is(of(SIMPLE_TEXT_PARAM_POJO_ID))));
    verify(component).withParameterizedModel(any());
  }

  @Test
  public void simpleEmptyCDataTextParamInSimpleTextPojo() {
    final DefaultComponentAstBuilder textComponentBuilder = createModel(SIMPLE_TEXT_PARAM_POJO_ID,
                                                                        emptyMap(),
                                                                        emptyList());
    textComponentBuilder.withMetadata(DefaultComponentMetadataAst.builder().putParserAttribute(IS_CDATA, true).build());

    final DefaultComponentAstBuilder textComponent = spy(textComponentBuilder);
    final DefaultComponentAstBuilder component = spy(createModel(SIMPLE_TEXT_POJO_ID,
                                                                 emptyMap(),
                                                                 singletonList(textComponent)));

    resolveTypedComponentIdentifier(component, of(mockParentAstBuilder), extModelHelper);

    verify(component).withParameter(argThat(new ParameterModelMatcher("pojoTextParam")),
                                    argThat(new ParameterGroupValueModelMatcher("SimpleTextPojo")),
                                    argThat(new ParameterRawValueModelMatcher("")),
                                    argThat(is(of(SIMPLE_TEXT_PARAM_POJO_ID))));
    verify(component).withParameterizedModel(any());
  }

  @Test
  public void pojoParamInComplexPojo() {
    final DefaultComponentAstBuilder innerComponent = buildOnce(spy(createModel(SIMPLE_ID,
                                                                                singletonMap("pojoSimpleParam", "value"))));

    final DefaultComponentAstBuilder component = spy(createModel(COMPLEX_POJO_ID,
                                                                 emptyMap(),
                                                                 singletonList(innerComponent)));

    resolveTypedComponentIdentifier(component, of(mockParentAstBuilder), extModelHelper);

    verify(innerComponent).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                         argThat(new ParameterGroupValueModelMatcher("SimplePojo")),
                                         argThat(new ParameterRawValueModelMatcher("value")),
                                         argThat(is(empty())));
    verify(innerComponent).withParameterizedModel(any());

    verify(component).withParameter(argThat(new ParameterModelMatcher("simple")),
                                    argThat(new ParameterGroupValueModelMatcher("ComplexPojo")),
                                    argThat(new ParameterNestedValueModelMatcher(innerComponent)),
                                    (Iterable<ComponentIdentifier>) argThat(hasItem(SIMPLE_ID)));
    verify(component).withParameterizedModel(any());
  }

  @Test
  public void pojoParamsInListPojo() {
    final DefaultComponentAstBuilder innerComponent1 = buildOnce(spy(createModel(SIMPLE_POJO_ID,
                                                                                 singletonMap("pojoSimpleParam", "value1"))));
    final DefaultComponentAstBuilder innerComponent2 = buildOnce(spy(createModel(SIMPLE_POJO_ID,
                                                                                 singletonMap("pojoSimpleParam", "value2"))));
    final DefaultComponentAstBuilder innerComponent3 = buildOnce(spy(createModel(SIMPLE_POJO_ID,
                                                                                 singletonMap("pojoSimpleParam", "value3"))));

    final DefaultComponentAstBuilder wrapper = spy(createModel(SIMPLES_ID,
                                                               emptyMap(),
                                                               asList(innerComponent1, innerComponent2, innerComponent3)));
    final DefaultComponentAstBuilder component = spy(createModel(LIST_POJO_ID,
                                                                 emptyMap(),
                                                                 singletonList(wrapper)));

    resolveTypedComponentIdentifier(component, of(mockParentAstBuilder), extModelHelper);

    verify(innerComponent1).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                          any(),
                                          argThat(new ParameterRawValueModelMatcher("value1")),
                                          argThat(is(empty())));
    verify(innerComponent1).withParameterizedModel(any());

    verify(innerComponent2).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                          any(),
                                          argThat(new ParameterRawValueModelMatcher("value2")),
                                          argThat(is(empty())));
    verify(innerComponent2).withParameterizedModel(any());

    verify(innerComponent3).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                          any(),
                                          argThat(new ParameterRawValueModelMatcher("value3")),
                                          argThat(is(empty())));
    verify(innerComponent3).withParameterizedModel(any());

    verify(component).withParameter(argThat(new ParameterModelMatcher("simples")),
                                    argThat(new ParameterGroupValueModelMatcher("ListPojo")),
                                    argThat(new ParameterNestedManyValueModelMatcher(innerComponent1, innerComponent2,
                                                                                     innerComponent3)),
                                    argThat(is(of(SIMPLES_ID))));
    verify(component).withParameterizedModel(any());
  }

  @Test
  @Issue("MULE-19602")
  public void pojoParamsGroupsInListPojo() {
    final DefaultComponentAstBuilder innerComponent1 = buildOnce(spy(createModel(SIMPLE_POJO_ID,
                                                                                 singletonMap("pojoSimpleParam", "value1"))));
    final DefaultComponentAstBuilder innerComponent2 = buildOnce(spy(createModel(SIMPLE_POJO_ID,
                                                                                 singletonMap("pojoSimpleParam", "value2"))));
    final DefaultComponentAstBuilder innerComponent3 = buildOnce(spy(createModel(SIMPLE_POJO_ID,
                                                                                 singletonMap("pojoSimpleParam", "value3"))));

    final DefaultComponentAstBuilder wrapper = spy(createModel(SIMPLES_ID,
                                                               emptyMap(),
                                                               asList(innerComponent1, innerComponent2, innerComponent3)));
    final DefaultComponentAstBuilder component = spy(createModel(LIST_POJO_ID,
                                                                 emptyMap(),
                                                                 singletonList(wrapper)));

    resolveTypedComponentIdentifier(component, of(mockParentAstBuilder), extModelHelper);

    verify(innerComponent1).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                          argThat(new ParameterGroupValueModelMatcher("SimplePojo")),
                                          argThat(new ParameterRawValueModelMatcher("value1")),
                                          argThat(is(empty())));
    verify(innerComponent1).withParameterizedModel(any());

    verify(innerComponent2).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                          argThat(new ParameterGroupValueModelMatcher("SimplePojo")),
                                          argThat(new ParameterRawValueModelMatcher("value2")),
                                          argThat(is(empty())));
    verify(innerComponent2).withParameterizedModel(any());

    verify(innerComponent3).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                          argThat(new ParameterGroupValueModelMatcher("SimplePojo")),
                                          argThat(new ParameterRawValueModelMatcher("value3")),
                                          argThat(is(empty())));
    verify(innerComponent3).withParameterizedModel(any());
  }

  @Test
  public void pojoParamsInMapPojo() {
    final DefaultComponentAstBuilder innerComponent1 = spy(createModel(SIMPLE_POJO_ID,
                                                                       singletonMap("pojoSimpleParam", "value1")));
    final DefaultComponentAstBuilder innerComponent2 = spy(createModel(SIMPLE_POJO_ID,
                                                                       singletonMap("pojoSimpleParam", "value2")));
    final DefaultComponentAstBuilder innerComponent3 = spy(createModel(SIMPLE_POJO_ID,
                                                                       singletonMap("pojoSimpleParam", "value3")));

    final DefaultComponentAstBuilder entry1 = buildOnce(spy(createModel(SIMPLE_BY_KEY_ID,
                                                                        singletonMap("key", "1"),
                                                                        singletonList(innerComponent1))));
    final DefaultComponentAstBuilder entry2 = buildOnce(spy(createModel(SIMPLE_BY_KEY_ID,
                                                                        singletonMap("key", "2"),
                                                                        singletonList(innerComponent2))));
    final DefaultComponentAstBuilder entry3 = buildOnce(spy(createModel(SIMPLE_BY_KEY_ID,
                                                                        singletonMap("key", "3"),
                                                                        singletonList(innerComponent3))));

    final DefaultComponentAstBuilder wrapper = spy(createModel(SIMPLES_BY_KEY_ID,
                                                               emptyMap(),
                                                               asList(entry1, entry2, entry3)));
    final DefaultComponentAstBuilder component = spy(createModel(MAP_POJO_ID,
                                                                 emptyMap(),
                                                                 singletonList(wrapper)));

    resolveTypedComponentIdentifier(component, of(mockParentAstBuilder), extModelHelper);

    verify(innerComponent1).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                          any(),
                                          argThat(new ParameterRawValueModelMatcher("value1")),
                                          argThat(is(empty())));
    verify(innerComponent1).withParameterizedModel(any());

    verify(innerComponent2).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                          any(),
                                          argThat(new ParameterRawValueModelMatcher("value2")),
                                          argThat(is(empty())));
    verify(innerComponent2).withParameterizedModel(any());

    verify(innerComponent3).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                          any(),
                                          argThat(new ParameterRawValueModelMatcher("value3")),
                                          argThat(is(empty())));
    verify(innerComponent3).withParameterizedModel(any());

    verify(component).withParameter(argThat(new ParameterModelMatcher("byKeySimples")),
                                    argThat(new ParameterGroupValueModelMatcher("MapPojo")),
                                    argThat(new ParameterNestedManyValueModelMatcher(entry1, entry2, entry3)),
                                    argThat(is(of(SIMPLES_BY_KEY_ID))));
    verify(component).withParameterizedModel(any());
  }

  @Test
  @Issue("MULE-19602")
  public void pojoParamsGroupsInMapPojo() {
    final DefaultComponentAstBuilder innerComponent1 = spy(createModel(SIMPLE_POJO_ID,
                                                                       singletonMap("pojoSimpleParam", "value1")));
    final DefaultComponentAstBuilder innerComponent2 = spy(createModel(SIMPLE_POJO_ID,
                                                                       singletonMap("pojoSimpleParam", "value2")));
    final DefaultComponentAstBuilder innerComponent3 = spy(createModel(SIMPLE_POJO_ID,
                                                                       singletonMap("pojoSimpleParam", "value3")));

    final DefaultComponentAstBuilder entry1 = buildOnce(spy(createModel(SIMPLE_BY_KEY_ID,
                                                                        singletonMap("key", "1"),
                                                                        singletonList(innerComponent1))));
    final DefaultComponentAstBuilder entry2 = buildOnce(spy(createModel(SIMPLE_BY_KEY_ID,
                                                                        singletonMap("key", "2"),
                                                                        singletonList(innerComponent2))));
    final DefaultComponentAstBuilder entry3 = buildOnce(spy(createModel(SIMPLE_BY_KEY_ID,
                                                                        singletonMap("key", "3"),
                                                                        singletonList(innerComponent3))));

    final DefaultComponentAstBuilder wrapper = spy(createModel(SIMPLES_BY_KEY_ID,
                                                               emptyMap(),
                                                               asList(entry1, entry2, entry3)));
    final DefaultComponentAstBuilder component = spy(createModel(MAP_POJO_ID,
                                                                 emptyMap(),
                                                                 singletonList(wrapper)));

    resolveTypedComponentIdentifier(component, of(mockParentAstBuilder), extModelHelper);

    verify(innerComponent1).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                          argThat(new ParameterGroupValueModelMatcher("SimplePojo")),
                                          argThat(new ParameterRawValueModelMatcher("value1")),
                                          argThat(is(empty())));
    verify(innerComponent1).withParameterizedModel(any());

    verify(innerComponent2).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                          argThat(new ParameterGroupValueModelMatcher("SimplePojo")),
                                          argThat(new ParameterRawValueModelMatcher("value2")),
                                          argThat(is(empty())));
    verify(innerComponent2).withParameterizedModel(any());

    verify(innerComponent3).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                          argThat(new ParameterGroupValueModelMatcher("SimplePojo")),
                                          argThat(new ParameterRawValueModelMatcher("value3")),
                                          argThat(is(empty())));
    verify(innerComponent3).withParameterizedModel(any());
  }

  @Test
  @Issue("MULE-19655")
  public void stringEntriesInMap() {
    final DefaultComponentAstBuilder entry1 = buildOnce(spy(createModel(STRING_BY_KEY_ID,
                                                                        ImmutableMap.of("key", "1", "value", "value1"))));
    final DefaultComponentAstBuilder entry2 = buildOnce(spy(createModel(STRING_BY_KEY_ID,
                                                                        ImmutableMap.of("key", "2", "value", "value2"))));
    final DefaultComponentAstBuilder entry3 = buildOnce(spy(createModel(STRING_BY_KEY_ID,
                                                                        ImmutableMap.of("key", "3", "value", "value3"))));

    final DefaultComponentAstBuilder wrapper = spy(createModel(STRINGS_BY_KEY_ID,
                                                               emptyMap(),
                                                               asList(entry1, entry2, entry3)));
    final DefaultComponentAstBuilder component = spy(createModel(MAP_STRING_ID,
                                                                 emptyMap(),
                                                                 singletonList(wrapper)));

    resolveTypedComponentIdentifier(component, of(mockParentAstBuilder), extModelHelper);

    verify(entry1).withParameter(argThat(new ParameterModelMatcher("key")),
                                 argThat(new ParameterGroupValueModelMatcher(DEFAULT_GROUP_NAME)),
                                 argThat(new ParameterRawValueModelMatcher("1")),
                                 argThat(is(empty())));
    verify(entry1).withParameter(argThat(new ParameterModelMatcher("value")),
                                 argThat(new ParameterGroupValueModelMatcher(DEFAULT_GROUP_NAME)),
                                 argThat(new ParameterRawValueModelMatcher("value1")),
                                 argThat(is(empty())));
    verify(entry1).withParameterizedModel(any());

    verify(entry2).withParameter(argThat(new ParameterModelMatcher("key")),
                                 argThat(new ParameterGroupValueModelMatcher(DEFAULT_GROUP_NAME)),
                                 argThat(new ParameterRawValueModelMatcher("2")),
                                 argThat(is(empty())));
    verify(entry2).withParameter(argThat(new ParameterModelMatcher("value")),
                                 argThat(new ParameterGroupValueModelMatcher(DEFAULT_GROUP_NAME)),
                                 argThat(new ParameterRawValueModelMatcher("value2")),
                                 argThat(is(empty())));
    verify(entry2).withParameterizedModel(any());

    verify(entry3).withParameter(argThat(new ParameterModelMatcher("key")),
                                 argThat(new ParameterGroupValueModelMatcher(DEFAULT_GROUP_NAME)),
                                 argThat(new ParameterRawValueModelMatcher("3")),
                                 argThat(is(empty())));
    verify(entry3).withParameter(argThat(new ParameterModelMatcher("value")),
                                 argThat(new ParameterGroupValueModelMatcher(DEFAULT_GROUP_NAME)),
                                 argThat(new ParameterRawValueModelMatcher("value3")),
                                 argThat(is(empty())));
    verify(entry3).withParameterizedModel(any());
  }

  @Test
  public void pojoParamInOperation() {
    final DefaultComponentAstBuilder paramComponent = buildOnce(spy(createModel(SIMPLE_POJO_PARAM_ID,
                                                                                singletonMap("pojoSimpleParam", "value"))));

    final DefaultComponentAstBuilder operation = spy(createModel(OPERATION,
                                                                 emptyMap(),
                                                                 singletonList(paramComponent)));

    resolveTypedComponentIdentifier(operation, of(mockParentAstBuilder), extModelHelper);

    verify(paramComponent).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                         argThat(new ParameterGroupValueModelMatcher("SimplePojo")),
                                         argThat(new ParameterRawValueModelMatcher("value")),
                                         argThat(is(empty())));
    verify(paramComponent).withParameterizedModel(any());

    verify(operation).withParameter(argThat(new ParameterModelMatcher("simplePojoParam")),
                                    argThat(new ParameterGroupValueModelMatcher("paramGroup")),
                                    argThat(new ParameterNestedValueModelMatcher(paramComponent)),
                                    (Iterable<ComponentIdentifier>) argThat(hasItem(SIMPLE_POJO_PARAM_ID)));
    verify(operation).withComponentModel(operationModel);
  }

  @Test
  @Issue("MULE-19566")
  public void pojoExpressionParamInOperation() {
    final DefaultComponentAstBuilder paramComponent = buildOnce(spy(createModel(SIMPLE_POJO_PARAM_EXPRESSION_ID,
                                                                                emptyMap(),
                                                                                "#['pojoSimpleParam': 'value']")));

    final DefaultComponentAstBuilder operation = spy(createModel(OPERATION,
                                                                 emptyMap(),
                                                                 singletonList(paramComponent)));

    resolveTypedComponentIdentifier(operation, of(mockParentAstBuilder), extModelHelper);

    verify(operation).withParameter(argThat(new ParameterModelMatcher("simplePojoParamExpression")),
                                    argThat(new ParameterGroupValueModelMatcher("paramGroup")),
                                    argThat(new ParameterRawValueModelMatcher("#['pojoSimpleParam': 'value']")),
                                    argThat(is(of(SIMPLE_POJO_PARAM_EXPRESSION_ID))));
    verify(operation).withComponentModel(operationModel);
  }

  @Test
  public void pojoParamGroupShowInDslInOperation() {
    final DefaultComponentAstBuilder paramComponent = buildOnce(spy(createModel(SIMPLE_POJO_PARAM_IN_DSL_ID,
                                                                                singletonMap("pojoSimpleParam", "value"))));

    final DefaultComponentAstBuilder group = spy(createModel(SHOW_IN_DSL_GROUP,
                                                             emptyMap(),
                                                             asList(paramComponent)));

    final DefaultComponentAstBuilder operation = spy(createModel(OPERATION,
                                                                 emptyMap(),
                                                                 asList(group)));

    resolveTypedComponentIdentifier(operation, of(mockParentAstBuilder), extModelHelper);

    verify(paramComponent).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                         argThat(new ParameterGroupValueModelMatcher("SimplePojo")),
                                         argThat(new ParameterRawValueModelMatcher("value")),
                                         argThat(is(empty())));
    verify(paramComponent).withParameterizedModel(any());

    verify(operation).withParameter(argThat(new ParameterModelMatcher("simplePojoParamInDsl")),
                                    argThat(new ParameterGroupValueModelMatcher("paramGroupInDsl")),
                                    argThat(new ParameterNestedValueModelMatcher(paramComponent)),
                                    (Iterable<ComponentIdentifier>) argThat(hasItem(SIMPLE_POJO_PARAM_IN_DSL_ID)));
    verify(operation).withComponentModel(operationModel);
  }

  @Test
  @Issue("MULE-17711")
  @io.qameta.allure.Description("DSL groups are removed from the child list when a param form the group is handled")
  public void pojoParamGroupShowInDslInOperationNoGroupAsChild() {
    final DefaultComponentAstBuilder paramComponent = buildOnce(spy(createModel(SIMPLE_POJO_PARAM_IN_DSL_ID,
                                                                                singletonMap("pojoSimpleParam", "value"))));

    final DefaultComponentAstBuilder group = spy(createModel(SHOW_IN_DSL_GROUP,
                                                             emptyMap(),
                                                             singletonList(paramComponent)));

    final DefaultComponentAstBuilder operation = spy(createModel(OPERATION,
                                                                 emptyMap(),
                                                                 singletonList(group)));

    resolveTypedComponentIdentifier(operation, of(mockParentAstBuilder), extModelHelper);
    operation.prepareForBuild();

    assertThat(operation.childComponentsStream().count(), is(0L));
  }

  @Test
  public void removeChildMapFromPojo() {
    final DefaultComponentAstBuilder entryComponent1 = spy(createModel(SIMPLE_BY_KEY_ID,
                                                                       ImmutableMap.of("key", "firstKey", "value",
                                                                                       "firstValue")));
    final DefaultComponentAstBuilder entryComponent2 = spy(createModel(SIMPLE_BY_KEY_ID,
                                                                       ImmutableMap.of("key", "secondKey", "value",
                                                                                       "secondValue")));
    ParameterModel mapParameterModel = mock(ParameterModel.class);
    DefaultComponentAstBuilder operation = createOperationWithMapParameter(entryComponent1, entryComponent2, mapParameterModel);
    when(mapParameterModel.getModelProperty(NoWrapperModelProperty.class)).thenReturn(Optional.of(new NoWrapperModelProperty()));

    assertThat(operation.childComponentsStream().count(), is(2L));
    resolveTypedComponentIdentifier(operation, of(mockParentAstBuilder), extModelHelper);
    verify(operation).removeChildComponent(entryComponent1);
    verify(operation).removeChildComponent(entryComponent2);
    assertThat(operation.childComponentsStream().count(), is(0L));
  }

  private DefaultComponentAstBuilder createOperationWithMapParameter(DefaultComponentAstBuilder entryComponent1,
                                                                     DefaultComponentAstBuilder entryComponent2,
                                                                     ParameterModel mapParameterModel) {
    final DefaultComponentAstBuilder operation = spy(createModel(OPERATION,
                                                                 emptyMap(),
                                                                 asList(entryComponent1, entryComponent2)));
    when(mapParameterModel.getRole()).thenReturn(BEHAVIOUR);
    when(mapParameterModel.getName()).thenReturn("mapEntries");
    when(mapParameterModel.getExpressionSupport()).thenReturn(SUPPORTED);

    ParameterDslConfiguration dslConfig = mock(ParameterDslConfiguration.class);
    when(dslConfig.allowsReferences()).thenReturn(true);
    when(dslConfig.allowsInlineDefinition()).thenReturn(true);
    when(mapParameterModel.getDslConfiguration()).thenReturn(dslConfig);
    final ClassTypeLoader typeLoader = ExtensionsTypeLoaderFactory.getDefault()
        .createTypeLoader(ApplicationModelTypeUtilsTestCase.class.getClassLoader());
    MetadataType stringType = typeLoader.load(String.class);

    ObjectType mapType = new DefaultObjectType(new ArrayList<>(), true, stringType, MetadataFormat.JSON, new HashMap<>()) {

      @Override
      public <T extends TypeAnnotation> Optional<T> getAnnotation(Class<T> extension) {
        if (extension.equals(ClassInformationAnnotation.class)) {
          return (Optional<T>) of(new ClassInformationAnnotation(Map.class));
        } else {
          return empty();
        }
      }
    };
    when(mapParameterModel.getType()).thenReturn(mapType);
    ParameterGroupModel parameterGroupModel = mock(ParameterGroupModel.class);
    when(parameterGroupModel.getName()).thenReturn("paramGroup");
    when(parameterGroupModel.getParameter("mapEntries")).thenReturn(of(mapParameterModel));
    when(parameterGroupModel.getParameterModels()).thenReturn(singletonList(mapParameterModel));

    List<ParameterGroupModel> originalGroups = operationModel.getParameterGroupModels();
    List<ParameterGroupModel> newGroups = new ArrayList<>(originalGroups);
    newGroups.add(parameterGroupModel);
    when(operationModel.getParameterGroupModels()).thenReturn(newGroups);

    List<ParameterModel> originalParams = operationModel.getAllParameterModels();
    List<ParameterModel> newParams = new ArrayList<>(originalParams);
    newParams.add(mapParameterModel);
    when(operationModel.getAllParameterModels()).thenReturn(newParams);
    return operation;
  }

  @Test
  public void doNotRemoveChildMapFromWrappedPojo() {
    final DefaultComponentAstBuilder entryComponent1 = spy(createModel(SIMPLE_BY_KEY_ID,
                                                                       ImmutableMap.of("key", "firstKey", "value",
                                                                                       "firstValue")));
    final DefaultComponentAstBuilder entryComponent2 = spy(createModel(SIMPLE_BY_KEY_ID,
                                                                       ImmutableMap.of("key", "secondKey", "value",
                                                                                       "secondValue")));
    ParameterModel mapParameterModel = mock(ParameterModel.class);
    DefaultComponentAstBuilder operation = createOperationWithMapParameter(entryComponent1, entryComponent2, mapParameterModel);
    when(mapParameterModel.getModelProperty(NoWrapperModelProperty.class)).thenReturn(empty());

    assertThat(operation.childComponentsStream().count(), is(2L));
    resolveTypedComponentIdentifier(operation, of(mockParentAstBuilder), extModelHelper);
    assertThat(operation.childComponentsStream().count(), is(2L));
  }

  @Test
  public void pojoParamInSourceCallback() {
    final DefaultComponentAstBuilder paramComponent = buildOnce(spy(createModel(SIMPLE_POJO_PARAM_ID,
                                                                                singletonMap("pojoSimpleParam", "value"))));

    final DefaultComponentAstBuilder source = spy(createModel(SOURCE,
                                                              emptyMap(),
                                                              singletonList(paramComponent)));

    resolveTypedComponentIdentifier(source, of(mockParentAstBuilder), extModelHelper);

    verify(paramComponent).withParameter(argThat(new ParameterModelMatcher("pojoSimpleParam")),
                                         argThat(new ParameterGroupValueModelMatcher("SimplePojo")),
                                         argThat(new ParameterRawValueModelMatcher("value")),
                                         argThat(is(empty())));
    verify(paramComponent).withParameterizedModel(any());

    verify(source).withParameter(argThat(new ParameterModelMatcher("simplePojoParam")),
                                 argThat(new ParameterGroupValueModelMatcher("paramGroup")),
                                 argThat(new ParameterNestedValueModelMatcher(paramComponent)),
                                 (Iterable<ComponentIdentifier>) argThat(hasItem(SIMPLE_POJO_PARAM_ID)));
    verify(source).withComponentModel(sourceModel);
  }

  @Test
  @Issue("MULE-19744")
  public void configComponentType() {
    final DefaultComponentAstBuilder config = spy(createModel(CONFIG,
                                                              emptyMap()));
    resolveTypedComponentIdentifier(config, of(mockParentAstBuilder), extModelHelper);
    assertThat(config.getComponentType(), is(ComponentType.CONFIG));
  }

  @Test
  @Issue("MULE-19744")
  public void connectionProviderComponentType() {
    final DefaultComponentAstBuilder connProvider = buildOnce(spy(createModel(CONN_PROVIDER,
                                                                              emptyMap())));
    resolveTypedComponentIdentifier(connProvider, of(mockParentAstBuilder), extModelHelper);
    assertThat(connProvider.getComponentType(), is(CONNECTION));
  }

  @Test
  @Issue("MULE-19744")
  public void constructConfigComponentType() {
    final DefaultComponentAstBuilder config = spy(createModel(CONSTRUCT_CONFIG,
                                                              emptyMap()));
    resolveTypedComponentIdentifier(config, of(mockParentAstBuilder), extModelHelper);
    assertThat(config.getComponentType(), is(ComponentType.CONFIG));
  }

  @Test
  public void listOfRoutes() {
    final DefaultComponentAstBuilder router = spy(createModel(LIST_OF_ROUTES_ID, emptyMap()));
    router.addChildComponent(spy(createModel(ITEM_ROUTER, emptyMap())));
    router.addChildComponent(spy(createModel(ITEM_ROUTER, emptyMap())));

    resolveTypedComponentIdentifier(router, of(mockParentAstBuilder), extModelHelper);

    List<DefaultComponentAstBuilder> routes = router.childComponentsStream().collect(toList());
    assertThat(routes, iterableWithSize(2));
    routes.forEach(c -> {
      assertThat(c.getModel(ParameterizedModel.class).isPresent(), is(true));
      assertThat(c.getModel(ParameterizedModel.class).isPresent(), is(true));
    });
  }

  private DefaultComponentAstBuilder buildOnce(final DefaultComponentAstBuilder paramComponent) {
    AtomicReference<ComponentAst> paramComp = new AtomicReference<>();

    doAnswer(inv -> paramComp.updateAndGet(alreadyBuilt -> {
      if (alreadyBuilt != null) {
        return alreadyBuilt;
      } else {
        try {
          return (ComponentAst) inv.callRealMethod();
        } catch (Throwable t) {
          throw new RuntimeException(t);
        }
      }
    })).when(paramComponent).build();

    return paramComponent;
  }

  private DefaultComponentAstBuilder createModel(ComponentIdentifier identifier, Map<String, String> rawParams,
                                                 List<DefaultComponentAstBuilder> children) {
    final DefaultComponentAstBuilder component = createModel(identifier, rawParams);

    children.forEach(component::addChildComponent);

    return component;
  }

  private DefaultComponentAstBuilder createModel(ComponentIdentifier identifier, Map<String, String> rawParams,
                                                 String body) {
    return (DefaultComponentAstBuilder) createModel(identifier, rawParams)
        .withRawParameter(BODY_RAW_PARAM_NAME, body);
  }

  private DefaultComponentAstBuilder createModel(ComponentIdentifier identifier, Map<String, String> rawParams) {
    ComponentAstBuilder builder = new DefaultComponentAstBuilder(new PropertiesResolver(), extModelHelper, emptyList(), 0,
                                                                 new ComponentLocationVisitor(),
                                                                 new ParameterModelUtils())
                                                                     .withIdentifier(identifier)
                                                                     .withMetadata(DefaultComponentMetadataAst.builder().build());

    rawParams.forEach((k, v) -> builder.withRawParameter(k, v));

    return (DefaultComponentAstBuilder) builder;
  }

  public static class SimplePojo {

    @Parameter
    private String pojoSimpleParam;

  }

  public static class SimpleTextPojo {

    @Text
    @Parameter
    private String pojoTextParam;

  }

  public static class ComplexPojo {

    @Parameter
    private SimplePojo simple;

  }

  public static class ListPojo {

    @Parameter
    private List<SimplePojo> simples;

  }

  public static class ListTextPojo {

    @Parameter
    private List<SimpleTextPojo> simples;

  }

  public static class MapPojo {

    @Parameter
    private Map<String, SimplePojo> byKeySimples;

  }

  public static class MapString {

    @Parameter
    private Map<String, String> byKeyStrings;

  }

  public static abstract class AbstractPojo {

    @Parameter
    private String pojoSuperclassParam;

  }

  public static class ConcreteComplexPojo extends AbstractPojo {

    @Parameter
    private SimplePojo simpleSubclass;

  }

  public class ParameterModelMatcher extends BaseMatcher<ParameterModel> {

    private final String paramName;

    public ParameterModelMatcher(String paramName) {
      this.paramName = requireNonNull(paramName);
    }

    @Override
    public boolean matches(Object item) {
      if (item instanceof ParameterModel) {
        return paramName.equals(((ParameterModel) item).getName());
      } else {
        return false;
      }
    }

    @Override
    public void describeTo(Description description) {
      description.appendText(paramName);
    }

  }

  public class ParameterRawValueModelMatcher extends BaseMatcher<ComponentParameterAst> {

    private final Matcher<?> rawParamValueMatcher;

    public ParameterRawValueModelMatcher(String rawParamValue) {
      this(equalTo(rawParamValue));
    }

    public ParameterRawValueModelMatcher(Matcher<?> rawParamValueMatcher) {
      this.rawParamValueMatcher = rawParamValueMatcher;
    }

    @Override
    public boolean matches(Object item) {
      if (item instanceof ComponentParameterAst) {
        return rawParamValueMatcher.matches(((ComponentParameterAst) item).getRawValue());
      } else {
        return false;
      }
    }

    @Override
    public void describeTo(Description description) {
      rawParamValueMatcher.describeTo(description);
    }

  }

  public class ParameterGroupValueModelMatcher extends BaseMatcher<ParameterGroupModel> {

    private final String paramGroupName;

    public ParameterGroupValueModelMatcher(String paramGroupName) {
      this.paramGroupName = requireNonNull(paramGroupName);
    }

    @Override
    public boolean matches(Object item) {
      if (item instanceof ParameterGroupModel) {
        return paramGroupName.equals(((ParameterGroupModel) item).getName());
      } else {
        return false;
      }
    }

    @Override
    public void describeTo(Description description) {
      description.appendText(paramGroupName);
    }
  }

  public class ParameterNestedValueModelMatcher extends BaseMatcher<ComponentParameterAst> {

    private final DefaultComponentAstBuilder nestedParamValue;

    public ParameterNestedValueModelMatcher(DefaultComponentAstBuilder nestedParamValue) {
      this.nestedParamValue = requireNonNull(nestedParamValue);
    }

    @Override
    public boolean matches(Object item) {
      if (item instanceof ComponentParameterAst) {
        final Object rightValue = ((ComponentParameterAst) item).getValue().getRight();
        return nestedParamValue.build().equals(rightValue);
      } else {
        return false;
      }
    }

    @Override
    public void describeTo(Description description) {
      description.appendText(nestedParamValue.toString());
    }

  }

  public class ParameterNestedManyValueModelMatcher extends BaseMatcher<ComponentParameterAst> {

    private final List<DefaultComponentAstBuilder> nestedParamValue;

    public ParameterNestedManyValueModelMatcher(DefaultComponentAstBuilder... nestedParamValue) {
      this.nestedParamValue = asList(requireNonNull(nestedParamValue));
    }

    @Override
    public boolean matches(Object item) {
      if (item instanceof ComponentParameterAst) {
        final Object rightValue = ((ComponentParameterAst) item).getValue().getRight();
        return nestedParamValue.stream()
            .map(DefaultComponentAstBuilder::build)
            .collect(toList())
            .equals(rightValue);
      } else {
        return false;
      }
    }

    @Override
    public void describeTo(Description description) {
      description.appendText(nestedParamValue.toString());
    }

  }

}
