/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api;

import org.mule.api.annotation.NoImplement;
import org.mule.runtime.ast.internal.DefaultComponentMetadataAst;

import java.net.URI;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;

@NoImplement
public interface ComponentMetadataAst {

  public static final ComponentMetadataAst EMPTY_METADATA = DefaultComponentMetadataAst.builder().build();

  /**
   * @return the config file name in which the owner component was defined. It may be empty if the component was created
   *         programmatically.
   */
  Optional<String> getFileName();

  /**
   * @return the config file URI in which the owner component was defined. It may be empty if the component was created
   *         programmatically.
   */
  Optional<URI> getFileUri();

  /**
   * If the component with this location is in a root config file (not obtained through an {@code import} tag), the returned list
   * will be empty.
   * <p>
   * If the component with this location is obtained through an import, the first import will be the first element of the returned
   * list. Any other nested imports will be the subsequent elements of the returned list.
   * 
   * @return a {@link List} containing an element for every {@code import} tag leading to the file containing the owning
   *         component.
   * 
   * @since 1.5
   */
  List<ImportedResource> getImportChain();

  /**
   * @return the first line number in which the component was defined in the configuration file. It may be empty if the component
   *         was created programmatically. See also {@link #getParserAttributes()} and {@link AstParserAttribute} for more precise
   *         DSL-specific location information.
   */
  OptionalInt getStartLine();

  /**
   * @return the start column in which the component was defined in the configuration file. It may be empty if the component was
   *         created programmatically. See also {@link #getParserAttributes()} and {@link AstParserAttribute} for more precise
   *         DSL-specific location information.
   */
  OptionalInt getStartColumn();

  /**
   * @return the last line number in which the component was defined in the configuration file. It may be empty if the component
   *         was created programmatically. See also {@link #getParserAttributes()} and {@link AstParserAttribute} for more precise
   *         DSL-specific location information.
   */
  OptionalInt getEndLine();

  /**
   * @return the end column in which the component was defined in the configuration file. It may be empty if the component was
   *         created programmatically. See also {@link #getParserAttributes()} and {@link AstParserAttribute} for more precise
   *         DSL-specific location information.
   */
  OptionalInt getEndColumn();

  /**
   * @return the source code associated with the owner component. It may be empty if the component was created programmatically.
   */
  Optional<String> getSourceCode();

  /**
   * @return a {@code java.util.Map} with the doc metadata from the source config.
   */
  Map<String, String> getDocAttributes();

  /**
   * @return a {@code java.util.Map} with metadata from the parser.
   */
  Map<String, Object> getParserAttributes();

}
