/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.model;

import static org.mule.runtime.api.util.classloader.MuleImplementationLoaderUtils.getMuleImplementationsLoader;
import static org.mule.runtime.api.util.classloader.MuleImplementationLoaderUtils.isResolveMuleImplementationLoadersDynamically;

import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;


/**
 * Some utility methods for reusing logic when dealing with {@link ParameterModel}s.
 */
public class ParameterModelUtils {

  private static final String ALLOWS_EXPRESSION_WITHOUT_MARKERS_MODEL_PROPERTY_CLASS_NAME_OLD =
      "org.mule.runtime.module.extension.api.loader.java.property.AllowsExpressionWithoutMarkersModelProperty";
  private static final String ALLOWS_EXPRESSION_WITHOUT_MARKERS_MODEL_PROPERTY_CLASS_NAME =
      "org.mule.runtime.core.internal.extension.AllowsExpressionWithoutMarkersModelProperty";

  private static final Class<? extends ModelProperty> ALLOWS_EXPRESSION_WITHOUT_MARKERS_MODEL_PROPERTY_CLASS_OLD =
      tryLoadClass(ALLOWS_EXPRESSION_WITHOUT_MARKERS_MODEL_PROPERTY_CLASS_NAME_OLD);
  private static final Class<? extends ModelProperty> ALLOWS_EXPRESSION_WITHOUT_MARKERS_MODEL_PROPERTY_CLASS =
      tryLoadClass(ALLOWS_EXPRESSION_WITHOUT_MARKERS_MODEL_PROPERTY_CLASS_NAME);

  private final Class<? extends ModelProperty> allowsExpressionWithoutMarkersModelPropertyClass =
      getAllowsExpressionWithoutMarkersModelPropertyClass();
  private final Class<? extends ModelProperty> allowsExpressionWithoutMarkersModelPropertyClassOld =
      getAllowsExpressionWithoutMarkersModelPropertyClassOld();

  public ParameterModelUtils() {
    // nothing to do
  }

  /**
   * @param parameterModel a {@link ParameterModel} to check.
   * @return whether the given {@code parameterModel} declares that allows expressions without the expression markers. This
   *         happens in some parameters from the core extension model that require expressions, but they don't force the DSL to
   *         include the expression delimiters.
   */
  public boolean allowsExpressionWithoutMarkers(ParameterModel parameterModel) {
    return allowsExpressionWithoutMarkers(parameterModel, allowsExpressionWithoutMarkersModelPropertyClass) ||
        allowsExpressionWithoutMarkers(parameterModel, allowsExpressionWithoutMarkersModelPropertyClassOld);
  }

  private static Class<? extends ModelProperty> getAllowsExpressionWithoutMarkersModelPropertyClass() {
    if (isResolveMuleImplementationLoadersDynamically()) {
      return tryLoadClass(ALLOWS_EXPRESSION_WITHOUT_MARKERS_MODEL_PROPERTY_CLASS_NAME);
    } else {
      return ALLOWS_EXPRESSION_WITHOUT_MARKERS_MODEL_PROPERTY_CLASS;
    }
  }

  private static Class<? extends ModelProperty> getAllowsExpressionWithoutMarkersModelPropertyClassOld() {
    if (isResolveMuleImplementationLoadersDynamically()) {
      return tryLoadClass(ALLOWS_EXPRESSION_WITHOUT_MARKERS_MODEL_PROPERTY_CLASS_NAME_OLD);
    } else {
      return ALLOWS_EXPRESSION_WITHOUT_MARKERS_MODEL_PROPERTY_CLASS_OLD;
    }
  }

  private static boolean allowsExpressionWithoutMarkers(ParameterModel parameterModel, Class<? extends ModelProperty> mpClass) {
    return mpClass != null
        && parameterModel != null
        && parameterModel.getModelProperty(mpClass).isPresent();
  }

  private static Class<? extends ModelProperty> tryLoadClass(String className) {
    try {
      return (Class<? extends ModelProperty>) getMuleImplementationsLoader().loadClass(className);
    } catch (ClassNotFoundException | SecurityException e) {
      // No custom location processing
    }
    return null;
  }
}
