/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal;

import static org.mule.runtime.api.component.ComponentIdentifier.builder;
import static org.mule.runtime.ast.internal.dsl.DslConstants.CORE_PREFIX;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.AstTraversal.AST_TRAVERSAL;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;

import java.util.function.Predicate;
import java.util.stream.Stream;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

import org.mockito.Mock;
import org.mockito.junit.MockitoJUnit;
import org.mockito.junit.MockitoRule;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(AST_TRAVERSAL)
public class FilteredArtifactAstTestCase {

  public static final ComponentIdentifier ON_ERROR_CONTINUE = builder().namespace(CORE_PREFIX).name("on-error-continue").build();
  @Rule
  public final MockitoRule mockitoRule = MockitoJUnit.rule();

  @Mock
  private ArtifactAst ast;
  @Mock
  private Predicate<ComponentAst> componentFilter;
  @Mock
  private ErrorTypeRepository errorRepo;
  @Mock
  private ComponentAst component;
  @Mock
  private ComponentParameterAst parameterAst;
  @Mock
  private ErrorType parentErrorType;


  @Before
  public void setUp() throws Exception {}

  @Test
  public void gettingErrorRepositoryWithNoComponentsStillReturnsRepo() {
    final ErrorTypeRepository result = new FilteredArtifactAst(ast, componentFilter, () -> errorRepo).getErrorTypeRepository();

    assertThat(result, is(notNullValue()));
  }

  @Test
  public void getErrorTypeRepository_oneNode() {
    when(ast.topLevelComponentsStream()).thenReturn(Stream.of(component));
    when(ast.recursiveStream(any())).thenAnswer(invocation -> Stream.of(component));
    when(component.getIdentifier()).thenReturn(ON_ERROR_CONTINUE);
    when(component.getParameter(any(), any())).thenReturn(parameterAst);
    when(parameterAst.getResolvedRawValue()).thenReturn("foo,bar");
    when(errorRepo.getAnyErrorType()).thenReturn(parentErrorType);

    final ErrorTypeRepository result = new FilteredArtifactAst(ast, x -> true, () -> errorRepo).getErrorTypeRepository();
    assertThat(result, is(notNullValue()));
  }
}
