/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.api;

import static java.util.Optional.empty;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsNot.not;
import static org.mockito.Answers.CALLS_REAL_METHODS;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import static org.mule.runtime.ast.api.util.ComponentAstPredicatesFactory.equalsIdentifier;
import static org.mule.runtime.ast.api.util.ComponentAstPredicatesFactory.equalsLocation;
import static org.mule.runtime.ast.api.util.ComponentAstPredicatesFactory.equalsNamespace;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.AstTraversal.AST_TRAVERSAL;
import static org.mule.runtime.dsl.api.component.config.DefaultComponentLocation.from;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.util.BaseArtifactAst;
import org.mule.runtime.ast.api.util.BaseComponentAst;

import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Test;

@Feature(ARTIFACT_AST)
@Story(AST_TRAVERSAL)
public class ArtifactAstTestCase {

  private static final String HTTP = "http";
  private static final String LISTENER = "listener";
  private static final String EE = "ee";
  private static final String TRANSFORM = "transform";
  private static final String MULE = "mule";
  private static final String FLOW = "flow";

  private static final String CHILD_1_LOCATION = "child-1-location";
  private static final String CHILD_2_LOCATION = "child-2-location";
  private static final String CHILD_2_1_LOCATION = "child-2.1-location";
  private static final String CHILD_2_2_LOCATION = "child-2.2-location";

  @Test
  public void filterComponentsByeEqualsIdentifierPredicate() {
    ArtifactAst artifactAst = createArtifactAstWithTwoLevels();

    // Find flow components
    List<ComponentAst> components = artifactAst
        .filteredComponents(equalsIdentifier(createFlowComponentIdentifier()))
        .collect(toList());
    assertThat(components, hasSize(2));

    // Find http listener components
    components = artifactAst.filteredComponents(equalsIdentifier(createComponentIdentifier(HTTP, LISTENER)))
        .collect(toList());
    assertThat(components, hasSize(1));
  }

  @Test
  public void filterComponentsByNonExistingIdentifier() {
    ArtifactAst artifactAst = createArtifactAstWithTwoLevels();

    List<ComponentAst> components = artifactAst
        .filteredComponents(equalsIdentifier(createComponentIdentifier("NS", "NAME"))).collect(toList());
    assertThat(components, hasSize(0));
  }

  @Test
  public void filterComponentsByEqualsLocationPredicate() {
    ArtifactAst artifactAst = createArtifactAstWithTwoLevels();

    // Find child 1
    Optional<ComponentAst> componentAst =
        artifactAst.filteredComponents(equalsLocation(from(CHILD_1_LOCATION)))
            .findFirst();
    assertThat(componentAst, is(not(empty())));

    // Find child 2.2
    componentAst = artifactAst
        .filteredComponents(equalsLocation(from(CHILD_2_2_LOCATION)))
        .findFirst();
    assertThat(componentAst, is(not(empty())));
  }

  @Test
  public void filterComponentsByNonExistingLocation() {
    ArtifactAst artifactAst = createArtifactAstWithTwoLevels();
    Optional<ComponentAst> componentAst = artifactAst
        .filteredComponents(equalsLocation(from("unknown"))).findFirst();
    assertThat(componentAst, is(empty()));
  }

  @Test
  public void filterComponentsByNamespacePredicate() {
    ArtifactAst artifactAst = createArtifactAstWithTwoLevels();
    List<ComponentAst> components = artifactAst.filteredComponents(equalsNamespace(EE)).collect(toList());
    assertThat(components, hasSize(1));
  }

  @Test
  public void getDirectChildrenByName() {
    ArtifactAst artifactAst = createArtifactAstWithTwoLevels();

    ComponentAst child2 = artifactAst.recursiveStream()
        .filter(c -> c.getLocation().equals(from(CHILD_2_LOCATION)))
        .findFirst()
        .get();

    List<ComponentAst> listeners = child2.directChildrenStreamByIdentifier(HTTP, LISTENER).collect(toList());
    assertThat(listeners, hasSize(1));

    ComponentAst listener = listeners.get(0);
    assertThat(listener.getIdentifier().getNamespace(), equalTo(HTTP));
    assertThat(listener.getIdentifier().getName(), equalTo(LISTENER));
  }

  private ComponentIdentifier createFlowComponentIdentifier() {
    return ComponentIdentifier.builder().namespace(MULE).name(FLOW).build();
  }

  private ComponentIdentifier createComponentIdentifier(String namespace, String name) {
    return ComponentIdentifier.builder().namespace(namespace).name(name).build();
  }

  private ArtifactAst createArtifactAstWithTwoLevels() {
    ComponentAst child1 = mock(BaseComponentAst.class);
    when(child1.getIdentifier()).thenReturn(createFlowComponentIdentifier());
    when(child1.getLocation()).thenReturn(from(CHILD_1_LOCATION));
    when(child1.directChildrenStream()).thenAnswer(inv -> Stream.empty());
    when(child1.recursiveStream()).thenAnswer(inv -> Stream.of(child1));

    ComponentAst child21 = mock(BaseComponentAst.class);
    when(child21.getIdentifier()).thenReturn(createComponentIdentifier(HTTP, LISTENER));
    when(child21.getLocation()).thenReturn(from(CHILD_2_1_LOCATION));
    when(child21.directChildrenStream()).thenAnswer(inv -> Stream.empty());
    when(child21.recursiveStream()).thenAnswer(inv -> Stream.of(child21));

    ComponentAst child22 = mock(BaseComponentAst.class);
    when(child22.getIdentifier()).thenReturn(createComponentIdentifier(EE, TRANSFORM));
    when(child22.getLocation()).thenReturn(from(CHILD_2_2_LOCATION));
    when(child22.directChildrenStream()).thenAnswer(inv -> Stream.empty());
    when(child22.recursiveStream()).thenAnswer(inv -> Stream.of(child22));

    ComponentAst child2 = mock(BaseComponentAst.class, CALLS_REAL_METHODS);
    when(child2.getIdentifier()).thenReturn(createFlowComponentIdentifier());
    when(child2.getLocation()).thenReturn(from(CHILD_2_LOCATION));
    when(child2.directChildrenStream()).thenAnswer(inv -> Stream.of(child21, child22));
    when(child2.recursiveStream()).thenAnswer(inv -> Stream.of(child2, child21, child22));

    ArtifactAst artifactAst = mock(BaseArtifactAst.class);
    when(artifactAst.recursiveStream())
        .thenAnswer(inv -> Stream.concat(child1.recursiveStream(), child2.recursiveStream()));
    when(artifactAst.filteredComponents(any())).thenCallRealMethod();

    return artifactAst;
  }
}
