/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.api.util;

import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;
import static org.mule.runtime.api.functional.Either.left;
import static org.mule.runtime.api.functional.Either.right;
import static org.mule.runtime.ast.api.util.MuleAstUtils.areComponentsEquals;
import static org.mule.runtime.ast.api.util.MuleAstUtils.areParametersEquals;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.AstTraversal.AST_TRAVERSAL;

import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.util.BaseComponentAst;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(AST_TRAVERSAL)
public class MuleAstUtilsComparationTestCase {

  private ComponentAst compA;
  private ComponentAst compB;
  private ComponentParameterAst paramA;
  private ComponentParameterAst paramB;

  @Before
  public void setUp() {
    paramA = mock(ComponentParameterAst.class);
    paramB = mock(ComponentParameterAst.class);

    compA = mockBaseComponentAst();
    compB = mockBaseComponentAst();

  }

  private ComponentAst mockBaseComponentAst() {
    ComponentAst comp = mock(BaseComponentAst.class);
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(mock(ParameterizedModel.class)));
    return comp;
  }

  @Test
  public void componentNonBehavior() {
    when(compA.getMetadata()).thenReturn(mock(ComponentMetadataAst.class));
    when(compB.getMetadata()).thenReturn(mock(ComponentMetadataAst.class));

    assertComponentsEquals(compA, compB);
  }

  @Test
  public void componentDifferentIdentifier() {
    when(compA.getIdentifier()).thenReturn(buildFromStringRepresentation("MY:COMP"));
    when(compB.getIdentifier()).thenReturn(buildFromStringRepresentation("MY:OTHER_COMP"));

    assertComponentsNotEquals(compA, compB);
  }

  @Test
  public void componentDifferentModel() {
    when(compA.getModel(Object.class)).thenReturn(of(mock(ParameterizedModel.class)));

    assertComponentsNotEquals(compA, compB);
  }

  @Test
  public void componentDifferentParamsSize() {
    when(compA.getParameters()).thenReturn(singleton(paramA));

    assertComponentsNotEquals(compA, compB);
  }

  @Test
  public void componentDifferentParamsKeys() {
    ParameterModel paramModelA1 = mock(ParameterModel.class);
    when(paramModelA1.getName()).thenReturn("ParamA1");

    ParameterGroupModel paramGroupModelA = mock(ParameterGroupModel.class);
    when(paramGroupModelA.getName()).thenReturn("ParamGroupA");

    when(paramA.getGroupModel()).thenReturn(paramGroupModelA);
    when(paramA.getModel()).thenReturn(paramModelA1);

    ParameterModel paramModelB1 = mock(ParameterModel.class);
    when(paramModelB1.getName()).thenReturn("ParamB1");

    ParameterGroupModel paramGroupModelB = mock(ParameterGroupModel.class);
    when(paramGroupModelB.getName()).thenReturn("ParamGroupB");

    when(paramB.getGroupModel()).thenReturn(paramGroupModelB);
    when(paramB.getModel()).thenReturn(paramModelB1);

    when(compA.getParameters()).thenReturn(singleton(paramA));
    when(compB.getParameters()).thenReturn(singleton(paramB));
    when(compB.getParameter("ParamGroupA", "ParamA1")).thenReturn(null);

    assertComponentsNotEquals(compA, compB);
  }

  @Test
  public void componentDifferentParamsValues() {
    ParameterModel paramModelA1 = mock(ParameterModel.class);
    when(paramModelA1.getName()).thenReturn("ParamA1");

    ParameterGroupModel paramGroupModelA = mock(ParameterGroupModel.class);
    when(paramGroupModelA.getName()).thenReturn("ParamGroupA");

    when(paramA.getGroupModel()).thenReturn(paramGroupModelA);
    when(paramA.getModel()).thenReturn(paramModelA1);

    when(paramB.getGroupModel()).thenReturn(paramGroupModelA);
    when(paramB.getModel()).thenReturn(paramModelA1);
    when(paramB.getValue()).thenReturn(Either.empty());

    when(compA.getParameters()).thenReturn(singleton(paramA));
    when(compB.getParameters()).thenReturn(singleton(paramB));
    when(compB.getParameter("ParamGroupA", "ParamA1")).thenReturn(paramB);

    assertComponentsNotEquals(compA, compB);
  }

  @Test
  public void componentDifferentChildrenSize() {
    when(compA.directChildren()).thenReturn(emptyList());
    when(compB.directChildren()).thenReturn(singletonList(mock(ComponentAst.class)));

    assertComponentsNotEquals(compA, compB);
  }

  @Test
  public void componentDifferentChildren() {
    ComponentAst compAChild = mock(ComponentAst.class);
    ComponentAst compBChild = mock(ComponentAst.class);

    when(compAChild.getIdentifier()).thenReturn(buildFromStringRepresentation("MY:COMP"));
    when(compBChild.getIdentifier()).thenReturn(buildFromStringRepresentation("MY:OTHER_COMP"));

    when(compA.directChildren()).thenReturn(singletonList(compAChild));
    when(compB.directChildren()).thenReturn(singletonList(compBChild));

    assertComponentsNotEquals(compA, compB);
  }

  @Test
  public void paramNullValues() {
    when(paramA.getValue()).thenReturn(null);
    when(paramB.getValue()).thenReturn(null);

    assertParametersEquals(paramA, paramB);
  }

  @Test
  public void paramValueNotPresent() {
    when(paramA.getValue()).thenReturn(Either.empty());
    when(paramB.getValue()).thenReturn(Either.empty());

    assertParametersEquals(paramA, paramB);
  }

  @Test
  public void paramNullToNotPresent() {
    when(paramA.getValue()).thenReturn(null);
    when(paramB.getValue()).thenReturn(Either.empty());

    assertParametersNotEquals(paramA, paramB);
  }

  @Test
  public void paramNullToExpression() {
    when(paramA.getValue()).thenReturn(null);
    when(paramB.getValue()).thenReturn(left("#[payload]"));

    assertParametersNotEquals(paramA, paramB);
  }

  @Test
  public void paramNullToFixed() {
    when(paramA.getValue()).thenReturn(null);
    when(paramB.getValue()).thenReturn(right("value"));

    assertParametersNotEquals(paramA, paramB);
  }

  @Test
  public void paramNotPresentToExpression() {
    when(paramA.getValue()).thenReturn(Either.empty());
    when(paramB.getValue()).thenReturn(left("#[payload]"));

    assertParametersNotEquals(paramA, paramB);
  }

  @Test
  public void paramNotPresentToFixed() {
    when(paramA.getValue()).thenReturn(Either.empty());
    when(paramB.getValue()).thenReturn(right("value"));

    assertParametersNotEquals(paramA, paramB);
  }

  @Test
  public void paramFixedToExpression() {
    when(paramA.getValue()).thenReturn(left("#[payload]"));
    when(paramB.getValue()).thenReturn(right("value"));

    assertParametersNotEquals(paramA, paramB);
  }

  @Test
  public void paramSameExpressions() {
    when(paramA.getValue()).thenReturn(left("#[payload]"));
    when(paramB.getValue()).thenReturn(left("#[payload]"));

    assertParametersEquals(paramA, paramB);
  }

  @Test
  public void paramSameFixedValues() {
    when(paramA.getValue()).thenReturn(right("value"));
    when(paramB.getValue()).thenReturn(right("value"));

    assertParametersEquals(paramA, paramB);
  }

  @Test
  public void paramDifferentExpressions() {
    when(paramA.getValue()).thenReturn(left("#[payloadA]"));
    when(paramB.getValue()).thenReturn(left("#[payloadB]"));

    assertParametersNotEquals(paramA, paramB);
  }

  @Test
  public void paramDifferentFixedValues() {
    when(paramA.getValue()).thenReturn(right("valueA"));
    when(paramB.getValue()).thenReturn(right("valueB"));

    assertParametersNotEquals(paramA, paramB);
  }

  @Test
  public void paramEqualComplexValues() {
    when(compA.getIdentifier()).thenReturn(buildFromStringRepresentation("MY:COMP"));
    when(compB.getIdentifier()).thenReturn(buildFromStringRepresentation("MY:COMP"));

    when(paramA.getValue()).thenReturn(right(compA));
    when(paramB.getValue()).thenReturn(right(compB));

    assertParametersEquals(paramA, paramB);
  }

  @Test
  public void paramEqualListValue() {
    when(compB.getIdentifier()).thenReturn(buildFromStringRepresentation("MY:COMP"));

    when(paramA.getValue()).thenReturn(right(singletonList(compB)));
    when(paramB.getValue()).thenReturn(right(singletonList(compB)));

    assertParametersEquals(paramA, paramB);
  }

  @Test
  public void paramDifferentComplexValues() {
    when(compA.getIdentifier()).thenReturn(buildFromStringRepresentation("MY:COMP"));
    when(compB.getIdentifier()).thenReturn(buildFromStringRepresentation("MY:OTHER_COMP"));

    when(paramA.getValue()).thenReturn(right(compA));
    when(paramB.getValue()).thenReturn(right(compB));

    assertParametersNotEquals(paramA, paramB);
  }

  @Test
  public void paramValueSimpleToComplex() {
    when(compB.getIdentifier()).thenReturn(buildFromStringRepresentation("MY:COMP"));

    when(paramA.getValue()).thenReturn(right("valueA"));
    when(paramB.getValue()).thenReturn(right(compA));

    assertParametersNotEquals(paramA, paramB);
  }

  @Test
  public void paramValueSimpleToList() {
    when(compB.getIdentifier()).thenReturn(buildFromStringRepresentation("MY:COMP"));

    when(paramA.getValue()).thenReturn(right("valueA"));
    when(paramB.getValue()).thenReturn(right(singletonList(compB)));

    assertParametersNotEquals(paramA, paramB);
  }

  @Test
  public void paramValueComplexToList() {
    when(compA.getIdentifier()).thenReturn(buildFromStringRepresentation("MY:COMP"));
    when(compB.getIdentifier()).thenReturn(buildFromStringRepresentation("MY:COMP"));

    when(paramA.getValue()).thenReturn(right(compA));
    when(paramB.getValue()).thenReturn(right(singletonList(compB)));

    assertParametersNotEquals(paramA, paramB);
  }

  @Test
  public void paramValueListDifferentChildrenSize() {
    when(paramA.getValue()).thenReturn(right(emptyList()));
    when(paramB.getValue()).thenReturn(right(singletonList(mock(ComponentAst.class))));

    assertParametersNotEquals(paramA, paramB);
  }

  @Test
  public void paramValueListDifferentChildren() {
    ComponentAst compAChild = mock(ComponentAst.class);
    ComponentAst compBChild = mock(ComponentAst.class);

    when(compAChild.getIdentifier()).thenReturn(buildFromStringRepresentation("MY:COMP"));
    when(compBChild.getIdentifier()).thenReturn(buildFromStringRepresentation("MY:OTHER_COMP"));

    when(paramA.getValue()).thenReturn(right(singletonList(compAChild)));
    when(paramB.getValue()).thenReturn(right(singletonList(compBChild)));

    assertParametersNotEquals(paramA, paramB);
  }


  private void assertComponentsEquals(ComponentAst compA, ComponentAst compB) {
    assertThat(areComponentsEquals(compA, compB), is(true));
    assertThat(areComponentsEquals(compB, compA), is(true));
  }

  private void assertComponentsNotEquals(ComponentAst compA, ComponentAst compB) {
    assertThat(areComponentsEquals(compA, compB), is(false));
    assertThat(areComponentsEquals(compB, compA), is(false));
  }

  private void assertParametersEquals(ComponentParameterAst paramA, ComponentParameterAst paramB) {
    assertThat(areParametersEquals(paramA, paramB), is(true));
    assertThat(areParametersEquals(paramB, paramA), is(true));
  }

  private void assertParametersNotEquals(ComponentParameterAst paramA, ComponentParameterAst paramB) {
    assertThat(areParametersEquals(paramA, paramB), is(false));
    assertThat(areParametersEquals(paramB, paramA), is(false));
  }
}
