/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.param;

import static org.mule.runtime.api.functional.Either.empty;
import static org.mule.runtime.api.functional.Either.left;
import static org.mule.runtime.api.functional.Either.right;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.ast.api.util.MuleAstUtils.hasPropertyPlaceholder;

import static java.lang.String.format;

import static org.apache.commons.lang3.StringUtils.isEmpty;

import org.mule.metadata.api.model.BooleanType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.NumberType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.runtime.api.functional.Either;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ParameterResolutionException;
import org.mule.runtime.extension.api.declaration.type.annotation.LiteralTypeAnnotation;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;


public class ParamTypeResolvingVisitor extends MetadataTypeVisitor {

  private static final Map<String, Function<String, Number>> fixedNumberMappings = new HashMap<>();

  static {
    fixedNumberMappings.put(Integer.class.getName(), Integer::valueOf);
    fixedNumberMappings.put(int.class.getName(), Integer::valueOf);

    fixedNumberMappings.put(Float.class.getName(), Float::valueOf);
    fixedNumberMappings.put(float.class.getName(), Float::valueOf);

    fixedNumberMappings.put(Long.class.getName(), Long::valueOf);
    fixedNumberMappings.put(long.class.getName(), Long::valueOf);

    fixedNumberMappings.put(Byte.class.getName(), Byte::valueOf);
    fixedNumberMappings.put(byte.class.getName(), Byte::valueOf);

    fixedNumberMappings.put(Short.class.getName(), Short::valueOf);
    fixedNumberMappings.put(short.class.getName(), Short::valueOf);

    fixedNumberMappings.put(Double.class.getName(), Double::valueOf);
    fixedNumberMappings.put(double.class.getName(), Double::valueOf);

    fixedNumberMappings.put(BigDecimal.class.getName(), BigDecimal::new);
    fixedNumberMappings.put(BigInteger.class.getName(), BigInteger::new);
  }

  public static Either<String, Object> resolveParamValue(ExpressionAwareParameter param, ComponentMetadataAst metadata,
                                                         String resolvedRawValue) {
    final ParamTypeResolvingVisitor visitor = new ParamTypeResolvingVisitor(param, resolvedRawValue);

    try {
      param.getModel().getType().accept(visitor);
    } catch (Exception e) {
      throw new ParameterResolutionException(createStaticMessage(format("Exception resolving param '%s' with value '%s' at '%s:%d:%d' (%s)",
                                                                        param.getModel().getName(), resolvedRawValue,
                                                                        metadata != null ? metadata.getFileName().orElse("") : "",
                                                                        metadata != null ? metadata.getStartLine().orElse(-1)
                                                                            : -1,
                                                                        metadata != null ? metadata.getStartColumn().orElse(-1)
                                                                            : -1,
                                                                        e.toString())),
                                             e);
    }

    if (visitor.expression.get() != null) {
      return left(visitor.expression.get());
    } else if (visitor.fixedValue.get() != null) {
      return right(visitor.fixedValue.get());
    } else {
      return empty();
    }
  }

  private final ExpressionAwareParameter param;
  private final String resolvedRawValue;
  private final AtomicReference<String> expression = new AtomicReference<>();
  private final AtomicReference<Object> fixedValue = new AtomicReference<>();

  public ParamTypeResolvingVisitor(ExpressionAwareParameter param, String resolvedRawValue) {
    this.param = param;
    this.resolvedRawValue = resolvedRawValue;
  }

  @Override
  public void visitObjectField(ObjectFieldType objectFieldType) {
    objectFieldType.getValue().accept(this);
  }

  @Override
  public void visitBoolean(BooleanType booleanType) {
    doVisitPrimitive(booleanType, () -> {
      if (!isEmpty(resolvedRawValue)) {
        fixedValue.set(Boolean.valueOf(resolvedRawValue));
      }
    });
  }

  @Override
  public void visitNumber(NumberType numberType) {
    doVisitPrimitive(numberType, () -> {
      if (!isEmpty(resolvedRawValue)) {
        visitFixedNumber(resolvedRawValue, fixedValue, numberType);
      }
    });
  }

  @Override
  public void visitString(StringType stringType) {
    doVisitPrimitive(stringType, () -> {
      // Empty string is valid, do not return either.empty for this!
      fixedValue.set(resolvedRawValue);
    });
  }

  private void doVisitPrimitive(MetadataType metadataType, Runnable onFixedValue) {
    if (param.isExpression(resolvedRawValue) || hasPropertyPlaceholder(resolvedRawValue)) {
      defaultVisit(metadataType);
    } else {
      onFixedValue.run();
    }
  }

  @Override
  protected void defaultVisit(MetadataType metadataType) {
    if ((NOT_SUPPORTED.equals(param.getModel().getExpressionSupport())
        || param.getModel().getType().getAnnotation(LiteralTypeAnnotation.class).isPresent())
        && !(!param.getModel().getAllowedStereotypes().isEmpty() && resolvedRawValue != null)) {
      fixedValue.set(resolvedRawValue);
    } else {
      defaultVisitFixedValue(resolvedRawValue, expression, fixedValue);
    }
  }

  private void visitFixedNumber(String rawValue, AtomicReference<Object> value, NumberType numberType) {
    value.set(numberType.getAnnotation(ClassInformationAnnotation.class)
        .map(classInfo -> fixedNumberMappings.getOrDefault(classInfo.getClassname(), s -> null).apply(rawValue))
        .orElseGet(() -> {
          try {
            Long longValue = Long.valueOf(rawValue);
            if (longValue <= Integer.MAX_VALUE && longValue >= Integer.MIN_VALUE) {
              return longValue.intValue();
            }
            return longValue;
          } catch (NumberFormatException e) {
            return Double.valueOf(rawValue);
          }
        }));
  }

  private void defaultVisitFixedValue(String rawValue, AtomicReference<String> expression, AtomicReference<Object> fixedValue) {
    Optional<String> expressionOpt = param.extractExpression(rawValue);
    if (expressionOpt.isPresent()) {
      // For complex types that may be the result of an expression, just return the expression
      expression.set(expressionOpt.get());
    } else {
      fixedValue.set(rawValue);
    }
  }

}
