/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.validation;

import static org.mule.runtime.ast.api.util.MuleAstUtils.recursiveStreamWithHierarchy;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.validation.ArtifactValidation;
import org.mule.runtime.ast.api.validation.Validation;
import org.mule.runtime.ast.api.validation.ValidationResult;
import org.mule.runtime.ast.api.validation.ArtifactAstValidator;

import java.util.List;

import com.google.common.collect.ImmutableList;

import org.slf4j.Logger;

public class DefaultValidator implements ArtifactAstValidator {

  private static final Logger LOGGER = getLogger(DefaultValidator.class);

  private final List<ArtifactValidation> artifactValidations;
  private final List<Validation> componentValidations;

  public DefaultValidator(List<ArtifactValidation> artifactValidations, List<Validation> componentValidations) {
    this.artifactValidations = artifactValidations;
    this.componentValidations = componentValidations;
  }

  @Override
  public ValidationResult validate(ArtifactAst artifact) {
    final DefaultValidationResult result = new DefaultValidationResult();

    artifactValidations
        .stream()
        .forEach(validation -> {
          LOGGER.debug("Running validation '{}' on artifact...", validation);
          validation.validateMany(artifact)
              .forEach(result::addItem);
        });

    recursiveStreamWithHierarchy(artifact)
        .forEach(ch -> componentValidations
            .stream()
            .filter(validation -> validation.applicable()
                .test(ImmutableList.<ComponentAst>builder().addAll(ch.getSecond()).add(ch.getFirst()).build()))
            .forEach(validation -> {
              LOGGER.debug("Running validation '{}' on '{}'...", validation, ch.getFirst());
              validation.validateMany(ch.getFirst(), artifact)
                  .forEach(result::addItem);
            }));

    return result;
  }

}
