/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.builder;

import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.AstTraversal.PARAMETER_AST;

import static java.util.Arrays.asList;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.collection.IsIterableWithSize.iterableWithSize;
import static org.junit.Assert.assertThrows;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.model.impl.DefaultStringType;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.api.model.ExtensionModelHelper;
import org.mule.runtime.ast.internal.builder.ComponentLocationVisitor;
import org.mule.runtime.ast.internal.builder.DefaultComponentAstBuilder;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;

import java.util.List;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(PARAMETER_AST)
public class ComponentAstParameterBuilderTestCase {

  private static final XmlDslModel XML_DSL_MODEL = XmlDslModel.builder()
      .setPrefix("mockns")
      .setNamespace("http://mockns")
      .build();

  private static final ComponentIdentifier COMPONENT_ID = ComponentIdentifier.builder()
      .namespaceUri(XML_DSL_MODEL.getNamespace())
      .namespace(XML_DSL_MODEL.getPrefix())
      .name("component")
      .build();

  private PropertiesResolver propertiesResolver;
  private ExtensionModelHelper extensionModelHelper;
  private ComponentLocationVisitor componentLocationVisitor;
  private ParameterModelUtils parameterModelUtils;

  private ParameterGroupModel paramGroup;
  private ParameterModel simpleParam;
  private ParameterModel complexParam;
  private ParameterModel listComplexParam;

  @Before
  public void before() {
    paramGroup = mock(ParameterGroupModel.class);
    when(paramGroup.getName()).thenReturn("group");
    simpleParam = mock(ParameterModel.class);
    when(simpleParam.getType()).thenReturn(new DefaultStringType(null, null));
    when(simpleParam.getName()).thenReturn("simpleParam");
    complexParam = mock(ParameterModel.class);
    when(complexParam.getName()).thenReturn("complexParam");
    listComplexParam = mock(ParameterModel.class);
    when(listComplexParam.getName()).thenReturn("listComplexParam");

    propertiesResolver = mock(PropertiesResolver.class);

    extensionModelHelper = mock(ExtensionModelHelper.class);

    final DslElementSyntax simpleParamSyntax = mock(DslElementSyntax.class);
    when(simpleParamSyntax.getElementName()).thenReturn("simple-param");
    when(extensionModelHelper.resolveDslElementModel(simpleParam, COMPONENT_ID))
        .thenReturn(simpleParamSyntax);

    final DslElementSyntax complexParamSyntax = mock(DslElementSyntax.class);
    when(complexParamSyntax.getElementName()).thenReturn("complex-param");
    when(extensionModelHelper.resolveDslElementModel(complexParam, COMPONENT_ID))
        .thenReturn(complexParamSyntax);

    final DslElementSyntax listComplexParamSyntax = mock(DslElementSyntax.class);
    when(listComplexParamSyntax.getElementName()).thenReturn("list-complex-param");
    when(extensionModelHelper.resolveDslElementModel(listComplexParam, COMPONENT_ID))
        .thenReturn(listComplexParamSyntax);

    parameterModelUtils = mock(ParameterModelUtils.class);
    componentLocationVisitor = mock(ComponentLocationVisitor.class);
  }

  @Test
  public void syntaxTwice() {
    final DefaultComponentAstBuilder componentAstBuilder =
        new DefaultComponentAstBuilder(propertiesResolver, extensionModelHelper, null, 0, componentLocationVisitor,
                                       parameterModelUtils);

    assertThrows(IllegalStateException.class,
                 () -> componentAstBuilder.addParameter(complexParam, paramGroup, configurer -> configurer
                     .withSyntax(mock(DslElementSyntax.class))
                     .withSyntax(mock(DslElementSyntax.class))));
  }

  @Test
  public void simpleValue() {
    final ComponentAstBuilder componentAstBuilder =
        new DefaultComponentAstBuilder(propertiesResolver, extensionModelHelper, null, 0, componentLocationVisitor,
                                       parameterModelUtils)
                                           .withIdentifier(COMPONENT_ID)
                                           .withMetadata(mock(ComponentMetadataAst.class))
                                           .withParameterizedModel(mock(ParameterizedModel.class));

    componentAstBuilder.addParameter(simpleParam, paramGroup, configurer -> configurer
        .withSyntax(null)
        .withMetadata(null)
        .withValue("simple"));

    final ComponentAst built = componentAstBuilder.build();
    assertThat(built.getParameters(), iterableWithSize(1));
    final ComponentParameterAst builtParam = built.getParameter(paramGroup.getName(), simpleParam.getName());
    assertThat(builtParam.getValue().getRight(), is("simple"));
  }

  @Test
  public void complexValue() {
    final ComponentAstBuilder componentAstBuilder =
        new DefaultComponentAstBuilder(propertiesResolver, extensionModelHelper, null, 0, componentLocationVisitor,
                                       parameterModelUtils)
                                           .withIdentifier(COMPONENT_ID)
                                           .withMetadata(mock(ComponentMetadataAst.class))
                                           .withParameterizedModel(mock(ParameterizedModel.class));

    componentAstBuilder.addParameter(complexParam, paramGroup, configurer -> configurer
        .withSyntax(null)
        .withMetadata(null)
        .withValue(mockCmplexValueBuilder()));

    final ComponentAst built = componentAstBuilder.build();
    assertThat(built.getParameters(), iterableWithSize(1));
    final ComponentParameterAst builtParam = built.getParameter(paramGroup.getName(), complexParam.getName());
    assertThat(builtParam.getValue().getRight(), not(nullValue()));
    assertThat(builtParam.getValue().getRight(), instanceOf(ComponentAst.class));
  }

  @Test
  public void listComplexValue() {
    final ComponentAstBuilder componentAstBuilder =
        new DefaultComponentAstBuilder(propertiesResolver, extensionModelHelper, null, 0, componentLocationVisitor,
                                       parameterModelUtils)
                                           .withIdentifier(COMPONENT_ID)
                                           .withMetadata(mock(ComponentMetadataAst.class))
                                           .withParameterizedModel(mock(ParameterizedModel.class));

    componentAstBuilder.addParameter(listComplexParam, paramGroup, configurer -> configurer
        .withSyntax(null)
        .withMetadata(null)
        .withValue(asList(mockCmplexValueBuilder(),
                          mockCmplexValueBuilder(),
                          mockCmplexValueBuilder())));

    final ComponentAst built = componentAstBuilder.build();
    assertThat(built.getParameters(), iterableWithSize(1));

    final ComponentParameterAst builtParam = built.getParameter(paramGroup.getName(), listComplexParam.getName());
    List<ComponentAst> builtValue = (List<ComponentAst>) builtParam.getValue().getRight();
    assertThat(builtValue, iterableWithSize(3));
    assertThat(builtValue.get(0), instanceOf(ComponentAst.class));
    assertThat(builtValue.get(1), instanceOf(ComponentAst.class));
    assertThat(builtValue.get(2), instanceOf(ComponentAst.class));

  }

  private DefaultComponentAstBuilder mockCmplexValueBuilder() {
    final DefaultComponentAstBuilder builder = mock(DefaultComponentAstBuilder.class);
    when(builder.build()).thenReturn(mock(ComponentAst.class));
    return builder;
  }
}
