/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.builder.adapter;

import static org.mule.runtime.api.meta.model.TypeAnnotationModelPropertyWrapper.defaultTypeAnnotationModelPropertyWrapper;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter.createParameterizedTypeModelAdapter;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.AstTraversal.PARAMETER_AST;
import static org.mule.runtime.extension.api.util.NameUtils.getAliasName;

import static java.util.Arrays.asList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.core.IsIterableContaining.hasItem;
import static org.junit.Assert.assertThrows;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.annotation.DefaultValueAnnotation;
import org.mule.metadata.api.model.NumberType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectKeyType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.TypeAnnotationModelPropertyWrapper;
import org.mule.runtime.api.meta.model.parameter.ExclusiveParametersModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.extension.api.declaration.type.annotation.ParameterDslAnnotation;
import org.mule.runtime.internal.meta.model.DefaultTypeAnnotationModelPropertyWrapper;
import org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter;
import org.mule.runtime.ast.internal.model.DefaultExtensionModelHelper;
import org.mule.runtime.extension.api.declaration.type.annotation.ExclusiveOptionalsTypeAnnotation;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;

import javax.xml.namespace.QName;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(PARAMETER_AST)
public class ObjectTypeAsParameterGroupAdapterTestCase {

  private final static String OBJECT_FIELD_A = "A";
  private final static String OBJECT_FIELD_B = "B";
  private final static String OBJECT_FIELD_C = "C";
  private final static String OBJECT_FIELD_D = "D";
  private final static String OBJECT_FIELD_E = "E";

  private ModelProperty targetFieldModelProperty;
  private ObjectType type;
  private DefaultExtensionModelHelper extensionModelHelper;

  @Before
  public void setup() {
    targetFieldModelProperty = mock(ModelProperty.class);
    type = createObjectType();
    extensionModelHelper = mock(DefaultExtensionModelHelper.class);
  }

  @Test
  public void getParametersSortedByName() {
    MetadataTypeModelAdapter metadataTypeModelAdapter = createParameterizedTypeModelAdapter(type, extensionModelHelper);

    List<ParameterGroupModel> parameterGroupModels = metadataTypeModelAdapter.getParameterGroupModels();
    assertThat(parameterGroupModels, not(empty()));
    assertThat(parameterGroupModels.size(), is(1));

    List<ParameterModel> parameterModels = parameterGroupModels.get(0).getParameterModels();
    assertThat(parameterModels, not(empty()));
    assertThat(parameterModels.size(), is(6));

    List<String> parameters = parameterModels.stream().map(NamedObject::getName).collect(toList());
    assertThat(parameters, is(asList(OBJECT_FIELD_A, OBJECT_FIELD_B, OBJECT_FIELD_C, OBJECT_FIELD_D, OBJECT_FIELD_E, "name")));
  }

  @Test
  @Issue("MULE-19349")
  public void exclusiveParametersModel() {
    MetadataTypeModelAdapter metadataTypeModelAdapter = createParameterizedTypeModelAdapter(type, extensionModelHelper);

    final ParameterGroupModel parameterGroupModel = metadataTypeModelAdapter.getParameterGroupModels().get(0);

    assertThat(parameterGroupModel.getExclusiveParametersModels().size(), is(2));

    ExclusiveParametersModel bExclusive = parameterGroupModel.getExclusiveParametersModels().get(0);
    assertThat(bExclusive.isOneRequired(), is(true));
    assertThat(bExclusive.getExclusiveParameterNames(), contains("exclusiveJ", "exclusiveK"));

    ExclusiveParametersModel aExclusive = parameterGroupModel.getExclusiveParametersModels().get(1);
    assertThat(aExclusive.isOneRequired(), is(false));
    assertThat(aExclusive.getExclusiveParameterNames(), contains("exclusiveX", "exclusiveY"));
  }

  @Test
  @Issue("MULE-19649")
  public void typeWithNoAlias() {
    MetadataTypeModelAdapter metadataTypeModelAdapter = createParameterizedTypeModelAdapter(type, extensionModelHelper);

    final ParameterGroupModel parameterGroupModel = metadataTypeModelAdapter.getParameterGroupModels().get(0);

    assertThrows(IllegalArgumentException.class, () -> getAliasName(type));

    assertThat(parameterGroupModel.getName(), is(DEFAULT_GROUP_NAME));
  }

  @Test
  @Issue("W-12438526")
  public void fieldWithDefaultValue() {
    MetadataTypeModelAdapter metadataTypeModelAdapter = createParameterizedTypeModelAdapter(type, extensionModelHelper);

    final ParameterGroupModel parameterGroupModel = metadataTypeModelAdapter.getParameterGroupModels().get(0);

    assertThat(parameterGroupModel.getParameter(OBJECT_FIELD_C).get().getDefaultValue(), is("42"));
  }

  @Test
  @Issue("W-14665879")
  public void typeAnnotationModelPropertyWrapper() {
    MetadataTypeModelAdapter metadataTypeModelAdapter = createParameterizedTypeModelAdapter(type, extensionModelHelper);

    final ParameterModel parameterModel =
        metadataTypeModelAdapter.getParameterGroupModels().get(0).getParameter(OBJECT_FIELD_C).get();
    assertThat(parameterModel.getModelProperties(), hasItem(targetFieldModelProperty));
  }

  @Test
  @Issue("W-18586120")
  public void parametersAllowingReferences() {
    MetadataTypeModelAdapter metadataTypeModelAdapter = createParameterizedTypeModelAdapter(type, extensionModelHelper);

    final ParameterModel parameterModelC =
        metadataTypeModelAdapter.getParameterGroupModels().get(0).getParameter(OBJECT_FIELD_C).get();
    assertThat(parameterModelC.getDslConfiguration().allowsReferences(), is(true));

    final ParameterModel parameterModelD =
        metadataTypeModelAdapter.getParameterGroupModels().get(0).getParameter(OBJECT_FIELD_D).get();
    assertThat(parameterModelD.getDslConfiguration(), is(not(nullValue())));
    assertThat(parameterModelD.getDslConfiguration().allowsReferences(), is(true));

    final ParameterModel parameterModelE =
        metadataTypeModelAdapter.getParameterGroupModels().get(0).getParameter(OBJECT_FIELD_E).get();
    assertThat(parameterModelE.getDslConfiguration(), is(not(nullValue())));
    assertThat(parameterModelE.getDslConfiguration().allowsReferences(), is(false));
  }

  private ObjectType createObjectType() {
    ObjectType objectType = mock(ObjectType.class);
    Collection<ObjectFieldType> fields = new HashSet<>();
    fields.add(createObjectFieldType(OBJECT_FIELD_C));
    fields.add(createObjectFieldTypeAllowingReferences(OBJECT_FIELD_D, true));
    fields.add(createObjectFieldTypeAllowingReferences(OBJECT_FIELD_E, false));
    fields.add(createObjectFieldTypeWithExclusiveness(OBJECT_FIELD_B, false, "exclusiveX", "exclusiveY"));
    fields.add(createObjectFieldTypeWithExclusiveness(OBJECT_FIELD_A, true, "exclusiveJ", "exclusiveK"));
    when(objectType.getFields()).thenReturn(fields);
    return objectType;
  }

  private ObjectFieldType createObjectFieldTypeAllowingReferences(String name, boolean allowReferences) {
    ObjectFieldType objectFieldType = createObjectFieldType(name);

    doAnswer(invocationOnMock -> of(new ParameterDslAnnotation(false, allowReferences))).when(objectFieldType)
        .getAnnotation(ParameterDslAnnotation.class);

    return objectFieldType;
  }

  private ObjectFieldType createObjectFieldType(String name) {
    QName qName = new QName(name);

    ObjectKeyType objectKeyType = mock(ObjectKeyType.class);
    when(objectKeyType.getName()).thenReturn(qName);

    ObjectFieldType objectFieldType = mock(ObjectFieldType.class);
    when(objectFieldType.getKey()).thenReturn(objectKeyType);
    final NumberType numberFieldValue = mock(NumberType.class);

    final DefaultValueAnnotation defaultValueAnn = new DefaultValueAnnotation("42");
    final TypeAnnotationModelPropertyWrapper wrapperModelPropertyAnn =
        defaultTypeAnnotationModelPropertyWrapper(targetFieldModelProperty);
    when(numberFieldValue.getAnnotation(DefaultValueAnnotation.class))
        .thenReturn(of(defaultValueAnn));
    when(numberFieldValue.getAnnotations())
        .thenReturn(new HashSet<>(asList(defaultValueAnn)));

    when(objectFieldType.getValue()).thenReturn(numberFieldValue);
    when(objectFieldType.getAnnotation(DefaultTypeAnnotationModelPropertyWrapper.class))
        .thenReturn(of((DefaultTypeAnnotationModelPropertyWrapper) wrapperModelPropertyAnn));
    when(objectFieldType.getAnnotations())
        .thenReturn(new HashSet<>(asList(wrapperModelPropertyAnn)));

    return objectFieldType;
  }

  private ObjectFieldType createObjectFieldTypeWithExclusiveness(String name, boolean oneRequired,
                                                                 String... exclusiveParamNames) {
    QName qName = new QName(name);

    ObjectKeyType objectKeyType = mock(ObjectKeyType.class);
    when(objectKeyType.getName()).thenReturn(qName);

    ObjectFieldType objectFieldType = mock(ObjectFieldType.class);
    when(objectFieldType.getKey()).thenReturn(objectKeyType);
    when(objectFieldType.getAnnotation(ExclusiveOptionalsTypeAnnotation.class))
        .thenReturn(of(new ExclusiveOptionalsTypeAnnotation(new HashSet<>(asList(exclusiveParamNames)), oneRequired)));
    return objectFieldType;
  }
}
