/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.util;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.ast.api.ComponentAst;

import java.util.List;
import java.util.function.Predicate;

/**
 * Set of predicates to filter {@link ComponentAst}s.
 *
 * @since 1.0.0
 */
public class ComponentAstPredicatesFactory {

  private ComponentAstPredicatesFactory() {
    // Nothing to do
  }

  /**
   * Filter {@link ComponentAst} with equals {@link ComponentIdentifier}
   *
   * @param identifier the {@link ComponentIdentifier} to check for.
   * @return the predicate with which to select the desired {@code identifier}
   */
  public static Predicate<ComponentAst> equalsIdentifier(ComponentIdentifier identifier) {
    return c -> c.getIdentifier().equals(identifier);
  }

  /**
   * Filter {@link ComponentAst} with equals {@code componentId} (Ref: {@link ComponentAst#getComponentId()})
   *
   * @param componentId the {@code componentId} to check for.
   * @return the predicate with which to select the desired {@code componentId}
   */
  public static Predicate<ComponentAst> equalsComponentId(String componentId) {
    return c -> c.getComponentId().map(componentId::equals).orElse(false);
  }

  /**
   * Filter {@link ComponentAst} with equals identifier namespace
   *
   * @param namespace the namespace to check for.
   * @return the predicate with which to select the desired {@code identifier}
   */
  public static Predicate<ComponentAst> equalsNamespace(String namespace) {
    return c -> c.getIdentifier().getNamespace().equals(namespace);
  }

  /**
   * Filter {@link ComponentAst} with equals {@link ComponentLocation}
   *
   * @param location the {@link ComponentLocation} to check for.
   * @return the predicate with which to select the desired {@code location}
   */
  public static Predicate<ComponentAst> equalsLocation(ComponentLocation location) {
    return c -> location.equals(c.getLocation());
  }

  /**
   * Filter component hierarchies that represent just a top-level component.
   *
   * @return a predicate to filter just top-level elements
   */
  public static Predicate<List<ComponentAst>> topLevelElement() {
    return c -> c.size() == 1;
  }

  /**
   * Filter a component hierarchy by evaluating the provided component against the last hierarchy element.
   *
   * @param currentElementPredicate the predicate to apply to the last element of the component hierarchy being inspected.
   * @return an adapted predicate.
   */
  public static Predicate<List<ComponentAst>> currentElemement(Predicate<ComponentAst> currentElementPredicate) {
    return c -> currentElementPredicate.test(c.get(c.size() - 1));
  }

  /**
   * Filter a component hierarchy by evaluating the provided component against any hierarchy element.
   *
   * @param currentElementPredicate the predicate to apply to any element of the component hierarchy being inspected.
   * @return an adapted predicate.
   */
  public static Predicate<List<ComponentAst>> anyElementInHierarchy(Predicate<ComponentAst> currentElementPredicate) {
    return c -> c.stream().anyMatch(currentElementPredicate);
  }

}
