/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.builder;

import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.AstTraversal.PARAMETER_AST;

import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import static com.google.common.base.Objects.equal;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.impl.DefaultArrayType;
import org.mule.metadata.api.model.impl.DefaultObjectType;
import org.mule.metadata.api.model.impl.DefaultStringType;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.api.builder.ComponentParameterAstBuilder;
import org.mule.runtime.ast.internal.builder.BaseComponentAstBuilder;
import org.mule.runtime.ast.internal.builder.DefaultComponentParameterAstBuilder;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;

import org.junit.Test;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Issue("MULE-19705")
@Feature(ARTIFACT_AST)
@Story(PARAMETER_AST)
public class BaseComponentAstBuilderTestCase {

  @Test
  public void testWithParameterRemovesMatchingChildComponent_WhenAddingAParameterThatIsNotShownInDSLAndIsAnElementInsideAnArray() {
    // Given
    TestComponentAstBuilder testComponentAstBuilder = new TestComponentAstBuilder(new PropertiesResolver());

    ParameterModel parameterModel = mock(ParameterModel.class);
    DefaultArrayType componentType = new DefaultArrayType(() -> new DefaultStringType(MetadataFormat.JSON, new HashMap<>()),
                                                          MetadataFormat.JSON, new HashMap<>());
    when(parameterModel.getType()).thenReturn(componentType);

    // A child "list-item" will not be present in the component syntax
    when(parameterModel.getName()).thenReturn("listParameterModelName");

    // The list item syntax will be matched to the actual child element
    DslElementSyntax listElementSyntax = mock(DslElementSyntax.class);
    when(listElementSyntax.getElementName()).thenReturn("list-element-name");
    when(listElementSyntax.getPrefix()).thenReturn("list-prefix");
    when(listElementSyntax.getNamespace()).thenReturn("list-namespace");

    // There will be a matching type as a generic of the component syntax
    DslElementSyntax componentSyntax = mock(DslElementSyntax.class);
    when(componentSyntax.getGeneric(any(DefaultStringType.class))).thenReturn(Optional.of(listElementSyntax));
    testComponentAstBuilder.getGenerationInformation().withSyntax(componentSyntax);

    // This is the actual child element that should be removed
    BaseComponentAstBuilder component = mock(BaseComponentAstBuilder.class);
    when(component.getIdentifier()).thenReturn(ComponentIdentifier.builder().name("list-element-name").namespace("list-prefix")
        .namespaceUri("list-namespace").build());
    testComponentAstBuilder.addChildComponent(component);

    ComponentParameterAst componentParameterAst = mock(ComponentParameterAst.class);
    Optional<ComponentIdentifier> componentIdentifierOptional = Optional.empty();

    ParameterGroupModel parameterGroupModel = mock(ParameterGroupModel.class);
    when(parameterGroupModel.isShowInDsl()).thenReturn(false);
    when(parameterGroupModel.getParameterModels()).thenReturn(singletonList(parameterModel));

    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));

    // When
    testComponentAstBuilder.withParameterizedModel(parameterizedModel);
    testComponentAstBuilder.withParameter(parameterModel, parameterGroupModel, componentParameterAst,
                                          componentIdentifierOptional);
    testComponentAstBuilder.prepareForBuild();

    // Then
    assertThat(testComponentAstBuilder.childComponentsStream().count(), is(0L));
  }

  @Test
  public void testWithParameterRemovesMatchingChildComponent_WhenAddingAParameterThatIsNotShownInDSLAndIsAMap() {
    // Given
    TestComponentAstBuilder testComponentAstBuilder = new TestComponentAstBuilder(new PropertiesResolver());

    // It is a map with a String key and a String value
    DefaultObjectType objectType = new DefaultObjectType(new ArrayList<>(), false, null, MetadataFormat.JSON,
                                                         new HashMap<>()) {

      @Override
      public <T extends TypeAnnotation> Optional<T> getAnnotation(Class<T> extension) {
        ClassInformationAnnotation classInformationAnnotation = mock(ClassInformationAnnotation.class);
        when(classInformationAnnotation.isMap()).thenReturn(true);
        List<String> genericTypes = new ArrayList<>();
        genericTypes.add(String.class.getName());
        genericTypes.add(String.class.getName());
        when(classInformationAnnotation.getGenericTypes()).thenReturn(genericTypes);
        return (Optional<T>) Optional.of(classInformationAnnotation);
      }
    };

    ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getType()).thenReturn(objectType);

    // A child "list-item" will not be present in the component syntax
    when(parameterModel.getName()).thenReturn("parameterModelName");

    // The map syntax will be matched to the actual child element
    DslElementSyntax mapSyntax = mock(DslElementSyntax.class);
    when(mapSyntax.getElementName()).thenReturn("element-name");
    when(mapSyntax.getPrefix()).thenReturn("prefix");
    when(mapSyntax.getNamespace()).thenReturn("namespace");

    // There will be a matching child of component syntax
    DslElementSyntax componentSyntax = mock(DslElementSyntax.class);
    when(componentSyntax.getChild("parameterModelName")).thenReturn(Optional.of(mapSyntax));
    testComponentAstBuilder.getGenerationInformation().withSyntax(componentSyntax);

    // This is the actual child element that should be removed
    BaseComponentAstBuilder component = mock(BaseComponentAstBuilder.class);
    when(component.getIdentifier())
        .thenReturn(ComponentIdentifier.builder().name("element-name").namespace("prefix").namespaceUri("namespace").build());
    testComponentAstBuilder.addChildComponent(component);

    ComponentParameterAst componentParameterAst = mock(ComponentParameterAst.class);
    Optional<ComponentIdentifier> componentIdentifierOptional = Optional.empty();

    ParameterGroupModel parameterGroupModel = mock(ParameterGroupModel.class);
    when(parameterGroupModel.isShowInDsl()).thenReturn(false);
    when(parameterGroupModel.getParameterModels()).thenReturn(singletonList(parameterModel));

    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));

    // When
    testComponentAstBuilder.withParameterizedModel(parameterizedModel);
    testComponentAstBuilder.withParameter(parameterModel, parameterGroupModel, componentParameterAst,
                                          componentIdentifierOptional);
    testComponentAstBuilder.prepareForBuild();

    // Then
    assertThat(testComponentAstBuilder.childComponentsStream().count(), is(0L));
  }

  @Test
  @Issue("W-10923083")
  @Description("Checks to return the placeholder value when the property resolver can resolve it.")
  public void getComponentIdReturnsThePlaceholderValue() {
    final String placeholder = "${key}";
    final String placeholderValue = "value";
    final PropertiesResolver propertiesResolver = new PropertiesResolver();
    propertiesResolver.setMappingFunction(getSimpleMappingFunction());
    final TestComponentAstBuilder testComponentAstBuilder = new TestComponentAstBuilder(propertiesResolver);
    final Supplier<Optional<String>> componentId = new LazyValue<>(() -> of(placeholder));

    testComponentAstBuilder.setComponentId(componentId);

    assertThat(testComponentAstBuilder.getComponentId().get(), is(placeholderValue));
  }

  @Test
  @Issue("W-10923083")
  @Description("Checks to return the placeholder when the property resolver can not resolve it.")
  public void getComponentIdReturnsThePlaceholder() {
    final String placeholder = "${KEY}";
    final PropertiesResolver propertiesResolver = new PropertiesResolver();
    propertiesResolver.setMappingFunction(getSimpleMappingFunction());
    final TestComponentAstBuilder testComponentAstBuilder = new TestComponentAstBuilder(propertiesResolver);
    final Supplier<Optional<String>> componentId = new LazyValue<>(() -> of(placeholder));

    testComponentAstBuilder.setComponentId(componentId);

    assertThat(testComponentAstBuilder.getComponentId().get(), is(placeholder));
  }

  private UnaryOperator<String> getSimpleMappingFunction() {
    final String placeholder = "${key}";
    final String placeholderValue = "value";
    return key -> {
      if (equal(placeholder, key)) {
        return placeholderValue;
      }
      return key;
    };
  }

  private class TestComponentAstBuilder extends BaseComponentAstBuilder {

    public TestComponentAstBuilder(PropertiesResolver propertiesResolver) {
      super(propertiesResolver, new ParameterModelUtils());
    }

    @Override
    public ComponentAstBuilder addChildComponent() {
      return null;
    }

    @Override
    public ComponentAstBuilder withTypedModel(MetadataType type) {
      return null;
    }

    @Override
    public ComponentAst build() {
      return null;
    }

    @Override
    public ComponentAst get() {
      return null;
    }

    @Override
    public ComponentParameterAst addParameter(ParameterModel paramModel, ParameterGroupModel parameterGroupModel,
                                              Consumer<ComponentParameterAstBuilder> paramConfigurer) {
      final DefaultComponentParameterAstBuilder paramBuilder = new DefaultComponentParameterAstBuilder();
      paramConfigurer.accept(paramBuilder);

      final ComponentParameterAst paramValue = paramBuilder.build();
      withParameter(paramModel, parameterGroupModel, paramValue, empty());
      return paramValue;
    }
  }
}
