/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.util;

import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.api.util.classloader.MuleImplementationLoaderUtils.getMuleImplementationsLoader;
import static org.mule.runtime.api.util.classloader.MuleImplementationLoaderUtils.isResolveMuleImplementationLoadersDynamically;
import static org.mule.runtime.ast.api.ArtifactType.APPLICATION;
import static org.mule.runtime.ast.api.error.ErrorTypeRepositoryProvider.getCoreErrorTypeRepo;
import static org.mule.runtime.ast.api.util.AstTraversalDirection.TOP_DOWN;
import static org.mule.runtime.ast.internal.dsl.DslConstants.CORE_PREFIX;
import static org.mule.runtime.ast.internal.dsl.DslConstants.KEY_ATTRIBUTE_NAME;
import static org.mule.runtime.ast.internal.dsl.DslConstants.VALUE_ATTRIBUTE_NAME;
import static org.mule.runtime.extension.api.ExtensionConstants.ERROR_MAPPINGS_PARAMETER_NAME;
import static org.mule.runtime.extension.api.util.ExtensionMetadataTypeUtils.isMap;
import static org.mule.runtime.extension.api.util.ExtensionModelUtils.getGroupAndParametersPairs;

import static java.lang.Thread.currentThread;
import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;

import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.SimpleType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.api.parameterization.ComponentParameterization;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ArtifactType;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.validation.ArtifactAstValidator;
import org.mule.runtime.ast.api.validation.ArtifactAstValidatorBuilder;
import org.mule.runtime.ast.api.validation.Validation;
import org.mule.runtime.ast.api.validation.ValidationResult;
import org.mule.runtime.ast.api.validation.ValidationsProvider;
import org.mule.runtime.ast.internal.FilteredArtifactAst;
import org.mule.runtime.ast.internal.builder.LightComponentAstBuilder;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;
import org.mule.runtime.ast.internal.validation.DefaultValidatorBuilder;

import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.UnaryOperator;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Set of utility methods for working with {@link ArtifactAst}s and {@link ComponentAst}s.
 *
 * @since 1.0
 */
public final class MuleAstUtils {

  private static final Logger LOGGER = LoggerFactory.getLogger(MuleAstUtils.class);

  private static final String DEFAULT_EXPRESSION_PREFIX = "#[";
  private static final String DEFAULT_EXPRESSION_SUFFIX = "]";

  private static final ExtensionModel CORE_EXT_MODEL;

  static {
    CORE_EXT_MODEL = getCoreExtensionModel();
  }

  private static ExtensionModel getCoreExtensionModel() {
    ExtensionModel coreExtModel = null;
    try {
      Class<?> coreExtModelProviderClass =
          getMuleImplementationsLoader().loadClass("org.mule.runtime.core.api.extension.provider.MuleExtensionModelProvider");
      coreExtModel = (ExtensionModel) coreExtModelProviderClass.getMethod("getExtensionModel").invoke(null);
    } catch (ClassNotFoundException | IllegalAccessException | IllegalArgumentException | InvocationTargetException
        | NoSuchMethodException | SecurityException e) {
      LOGGER.warn("{}: {}", e.getClass().getName(), e.getMessage());
    }

    return coreExtModel;
  }

  private static final ArtifactAst EMPTY_ARTIFACT = new BaseArtifactAst() {

    @Override
    public String getArtifactName() {
      return "<empty>";
    }

    @Override
    public ArtifactType getArtifactType() {
      return APPLICATION;
    }

    @Override
    public Set<ExtensionModel> dependencies() {
      ExtensionModel coreExtModel = CORE_EXT_MODEL;
      if (isResolveMuleImplementationLoadersDynamically()) {
        coreExtModel = getCoreExtensionModel();
      }
      if (coreExtModel == null) {
        return Collections.emptySet();
      } else {
        return singleton(coreExtModel);
      }
    }

    @Override
    public Optional<ArtifactAst> getParent() {
      return empty();
    }

    @Override
    public List<ComponentAst> topLevelComponents() {
      return emptyList();
    }

    @Override
    public void updatePropertiesResolver(UnaryOperator<String> newPropertiesResolver) {
      // Nothing to do
    }

    @Override
    public ErrorTypeRepository getErrorTypeRepository() {
      return getCoreErrorTypeRepo();
    }

    @Override
    public Collection<ImportedResource> getImportedResources() {
      return emptyList();
    }

    @Override
    public String toString() {
      return "(empty artifactAst)";
    }
  };

  private MuleAstUtils() {
    // Nothing to do
  }

  /**
   * Given an {@link ArtifactAst}, return all those components within it that are not accessible from a top-level component.
   *
   * @param appModel the model to determine orphan components from.
   * @return components within the provided {@code appModel} that are not accessible from a top-level component.
   */
  public static Set<ComponentAst> resolveOrphanComponents(ArtifactAst appModel) {
    final Set<ComponentAst> notOrphanComponents = appModel.topLevelComponentsStream()
        .filter(componentAst -> componentAst.getModel(ParameterizedModel.class).isPresent())
        .flatMap(ComponentAst::recursiveStream)
        .collect(toSet());
    notOrphanComponents.addAll(appModel
        .filteredComponents(cm -> !notOrphanComponents.contains(cm))
        .flatMap(ComponentAst::directChildrenStream)
        .collect(toSet()));

    return appModel
        .filteredComponents(cm -> !notOrphanComponents.contains(cm))
        .collect(toSet());
  }

  /**
   * Provides a singleton empty {@link ArtifactAst} instance.
   *
   * @return an {@link ArtifactAst} without components.
   */
  public static ArtifactAst emptyArtifact() {
    return EMPTY_ARTIFACT;
  }

  /**
   * Given a {@link ComponentAst}, lookup for a parameter that references an object with a certain stereotype.
   *
   * @param referent            the {@link ComponentAst} that will be checked for the reference.
   * @param referenceStereotype the {@link StereotypeModel} to check for.
   * @return any {@link ComponentParameterAst} representing a parameter of {@code referent} that references an object assignable
   *         to {@code referenceStereotype}, if it exists.
   */
  public static Optional<ComponentParameterAst> parameterOfType(ComponentAst referent, StereotypeModel referenceStereotype) {
    return referent.getModel(ParameterizedModel.class)
        .flatMap(parameterized -> getGroupAndParametersPairs(parameterized)
            .filter(pair -> pair.getSecond().getAllowedStereotypes()
                .stream()
                .anyMatch(allowed -> allowed.isAssignableTo(referenceStereotype)))
            .map(pair -> referent.getParameter(pair.getFirst().getName(), pair.getSecond().getName()))
            .findAny());
  }

  /**
   * Given a {@link ComponentAst}, lookup for all parameters that reference an object with a certain stereotype.
   *
   * @param referent            the {@link ComponentAst} that will be checked for the reference.
   * @param referenceStereotype the {@link StereotypeModel} to check for.
   * @return all {@link ComponentParameterAst}s representing parameters of {@code referent} that reference an object assignable to
   *         {@code referenceStereotype}.
   */
  public static List<ComponentParameterAst> parametersOfType(ComponentAst referent, StereotypeModel referenceStereotype) {
    return referent.getModel(ParameterizedModel.class)
        .map(parameterized -> getGroupAndParametersPairs(parameterized)
            .filter(pair -> pair.getSecond().getAllowedStereotypes()
                .stream()
                .anyMatch(allowed -> allowed.isAssignableTo(referenceStereotype)))
            .map(pair -> referent.getParameter(pair.getFirst().getName(), pair.getSecond().getName()))
            .collect(toList()))
        .orElse(emptyList());
  }

  /**
   * Performs a given action on the {@link ComponentAst}s that form the value of the provided parameter.
   *
   * @param param  the parameter on which complex parameter values the {@code action} will be called.
   * @param action the callback to perform on each found parameter item.
   */
  public static void doOnParamComponents(ComponentParameterAst param, Consumer<ComponentAst> action) {
    if (param.getModel().getName().equals(ERROR_MAPPINGS_PARAMETER_NAME)) {
      return;
    }
    param.getModel().getType().accept(new MetadataTypeVisitor() {

      @Override
      public void visitArrayType(ArrayType arrayType) {
        final Object value = param.getValue().getRight();
        if (value != null && value instanceof Collection) {
          ((Iterable<ComponentAst>) value).forEach(this::doAction);
        }
      }

      @Override
      public void visitObject(ObjectType objectType) {
        final Object value = param.getValue().getRight();
        if (isMap(objectType)) {
          if (value != null && value instanceof Collection) {
            ((Iterable<ComponentAst>) value).forEach(this::doAction);
          }
        } else {
          if (value instanceof ComponentAst) {
            doAction(value);
          }
        }
      }

      private void doAction(Object component) {
        if (component != null && component instanceof ComponentAst) {
          action.accept((ComponentAst) component);
        }
      }
    });
  }

  /**
   * Similar to {@link ArtifactAst#recursiveStream()}, but it enhances the {@link Stream} by adding hierarchy context to each
   * item. The {@link ArtifactAst} is traversed in {@link TOP_DOWN} direction.
   * <p>
   * The elements of the {@link Stream} are {@link Pair}s:
   * <ul>
   * <li>the first element is the same component that would be obtained by calling {@link ArtifactAst#recursiveStream()}.</li>
   * <li>the second element is the hierarchy that was traversed to get to the first element, the first element being a root and
   * the last being the direct parent of the first element of the pair.</li>
   * </ul>
   *
   * @param artifact the {@link ArtifactAst} to generate a {@link Stream} for.
   * @return an enhanced {@link Stream} with hierarchy context.
   */
  public static Stream<Pair<ComponentAst, List<ComponentAst>>> recursiveStreamWithHierarchy(ArtifactAst artifact) {
    return recursiveStreamWithHierarchy(artifact, TOP_DOWN, false);
  }

  /**
   * Similar to {@link ArtifactAst#recursiveStream()}, but it enhances the {@link Stream} by adding hierarchy context to each
   * item. The {@link ArtifactAst} is traversed in {@link AstTraversalDirection} direction
   * <p>
   * The elements of the {@link Stream} are {@link Pair}s:
   * <ul>
   * <li>the first element is the same component that would be obtained by calling {@link ArtifactAst#recursiveStream()}.</li>
   * <li>the second element is the hierarchy that was traversed to get to the first element, the first element being a root and
   * the last being the direct parent of the first element of the pair.</li>
   * </ul>
   *
   * @param artifact                     the {@link ArtifactAst} to generate a {@link Stream} for.
   * @param direction                    the {@link AstTraversalDirection} used to navigate the {@code artifact}.
   * @param ignoreParameterComplexValues whether the returned {@link Stream} will contain any complex values in component
   *                                     parameters.
   * @return an enhanced {@link Stream} with hierarchy context.
   */
  public static Stream<Pair<ComponentAst, List<ComponentAst>>> recursiveStreamWithHierarchy(ArtifactAst artifact,
                                                                                            AstTraversalDirection direction) {
    return recursiveStreamWithHierarchy(artifact, direction, false);
  }

  /**
   * Similar to {@link ArtifactAst#recursiveStream()}, but it enhances the {@link Stream} by adding hierarchy context to each
   * item. The {@link ArtifactAst} is traversed in {@link AstTraversalDirection} direction
   * <p>
   * The elements of the {@link Stream} are {@link Pair}s:
   * <ul>
   * <li>the first element is the same component that would be obtained by calling {@link ArtifactAst#recursiveStream()}.</li>
   * <li>the second element is the hierarchy that was traversed to get to the first element, the first element being a root and
   * the last being the direct parent of the first element of the pair.</li>
   * </ul>
   *
   * @param artifact                     the {@link ArtifactAst} to generate a {@link Stream} for.
   * @param direction                    the {@link AstTraversalDirection} used to navigate the {@code artifact}.
   * @param ignoreParameterComplexValues whether the returned {@link Stream} will contain any complex values in component
   *                                     parameters.
   * @return an enhanced {@link Stream} with hierarchy context.
   */
  public static Stream<Pair<ComponentAst, List<ComponentAst>>> recursiveStreamWithHierarchy(ArtifactAst artifact,
                                                                                            AstTraversalDirection direction,
                                                                                            boolean ignoreParameterComplexValues) {

    final Set<ComponentAst> orphanComponents = resolveOrphanComponents(artifact);

    if (orphanComponents.isEmpty()) {
      return direction.recursiveStreamWithHierarchy(artifact.topLevelComponentsStream(), ignoreParameterComplexValues);
    } else {
      // Do not break code that checks if a component is top-level by checking for an empty hierarchy for orphans.
      final ComponentAst orphanPhantomContainer = new LightComponentAstBuilder(new ParameterModelUtils())
          .withIdentifier(ComponentIdentifier.builder()
              .namespace(CORE_PREFIX)
              .name("orphan-container")
              .build())
          .build();

      return Stream.concat(direction.recursiveStreamWithHierarchy(artifact.topLevelComponentsStream()),
                           direction.recursiveStreamWithHierarchy(orphanComponents.stream(),
                                                                  singletonList(orphanPhantomContainer)));
    }
  }

  /**
   * Determines if a value depends on a property, as defined by the
   *
   * <pre>
   * ${...}
   * </pre>
   *
   * markers.
   *
   * @param v the value to evaluate.
   * @return {@code true} if the provided value contains a property placeholder, {@code false} otherwise.
   *
   * @since 1.1
   */
  public static boolean hasPropertyPlaceholder(String v) {
    // TODO W-12175130 properly implement this
    return v != null && v.contains("${");
  }

  /**
   * Provides a way to execute validations on an {@link ArtifactAst}.
   *
   * @return a builder for an {@link ArtifactAstValidator}.
   * @since 1.1
   */
  public static ArtifactAstValidatorBuilder validatorBuilder() {
    return new DefaultValidatorBuilder();
  }

  /**
   * Provides a way to create a {@link ComponentParameterization} from a {@link ComponentAst}.
   *
   * @param componentAst the {@link ComponentAst} to be transformed.
   * @return a {@link ComponentParameterization}.
   * @throws IllegalArgumentException if the {@link ComponentAst} model is not a {@link ParameterizedModel}
   * @since 1.1
   */
  public static <T extends ParameterizedModel> ComponentParameterization<T> createComponentParameterizationFromComponentAst(ComponentAst componentAst) {
    return componentAst.getModel(ParameterizedModel.class).map(model -> {
      ComponentParameterization.Builder<T> builder = ComponentParameterization.<T>builder((T) model);
      componentAst.getParameters()
          .forEach(parameter -> createComponentParameterizationFromComponentAstParameter(parameter.getValue(), parameter
              .getModel().getType())
                  .ifPresent(v -> builder.withParameter(parameter.getGroupModel(), parameter.getModel(), v)));
      builder.withComponentIdentifier(componentAst.getIdentifier());
      return builder.build();
    }).orElseThrow(() -> new IllegalArgumentException("ComponentAst model must be a ParameterizedModel"));
  }

  private static Optional<Object> createComponentParameterizationFromComponentAstParameter(final Either<String, Object> value,
                                                                                           final MetadataType type) {
    return value.mapLeft(expr -> DEFAULT_EXPRESSION_PREFIX + expr + DEFAULT_EXPRESSION_SUFFIX)
        .mapRight(item -> createComponentParameterizationFromFixedValueComponentAstParameter(item, type))
        .getValue();
  }

  private static Object createComponentParameterizationFromFixedValueComponentAstParameter(Object v, MetadataType metadataType) {
    Object result = v;
    if (v instanceof ComponentAst) {
      result = createComponentParameterizationFromComponentAst((ComponentAst) v);
    } else if (v instanceof Collection) {
      if (isMap(metadataType)) {
        ObjectType mapType = (ObjectType) metadataType;
        MetadataType entryType = mapType.getOpenRestriction().get();

        final AtomicReference<Object> mapResult = new AtomicReference<>();
        entryType.accept(new MetadataTypeVisitor() {

          @Override
          public void visitSimpleType(SimpleType simpleType) {
            mapResult.set(((Collection<ComponentAst>) v).stream()
                .collect(toMap(item -> item.getParameter(DEFAULT_GROUP_NAME, KEY_ATTRIBUTE_NAME).getValue().getRight(),
                               item -> createComponentParameterizationFromComponentAstParameter(item
                                   .getParameter(DEFAULT_GROUP_NAME, VALUE_ATTRIBUTE_NAME).getValue(), entryType)
                                       .orElse(null))));
          }

          @Override
          protected void defaultVisit(MetadataType metadataType) {
            mapResult.set(((Collection<ComponentAst>) v).stream()
                .collect(toMap(item -> item.getParameter(DEFAULT_GROUP_NAME, KEY_ATTRIBUTE_NAME).getValue().getRight(),
                               MuleAstUtils::createComponentParameterizationFromComponentAst)));
          }
        });

        result = mapResult.get();
      } else {
        ArrayType arrayType = (ArrayType) metadataType;
        MetadataType itemsType = arrayType.getType();

        final AtomicReference<Object> arrayResult = new AtomicReference<>();
        itemsType.accept(new MetadataTypeVisitor() {

          @Override
          public void visitSimpleType(SimpleType simpleType) {
            arrayResult.set(((Collection<ComponentAst>) v).stream()
                .map(item -> createComponentParameterizationFromComponentAstParameter(item
                    .getParameter(DEFAULT_GROUP_NAME, VALUE_ATTRIBUTE_NAME).getValue(), itemsType)
                        .orElse(null))
                .collect(toList()));
          }

          @Override
          protected void defaultVisit(MetadataType metadataType) {
            arrayResult.set(((Collection<ComponentAst>) v).stream()
                .map(MuleAstUtils::createComponentParameterizationFromComponentAst)
                .collect(toList()));
          }
        });

        result = arrayResult.get();
      }

    }
    return result;
  }

  /**
   * Runs all validations available from the current {@link Thread#getContextClassLoader()}.
   * <p>
   * All validations will be run even if failures are found in some. The result will contain all failures.
   *
   * @param artifact the artifact to validate
   * @return the result of the validation
   *
   * @deprecated since 1.1, use {@link #validatorBuilder()} instead.
   */
  @Deprecated
  public static ValidationResult validate(ArtifactAst artifact) {
    return validate(artifact, v -> {
    }, v -> true, currentThread().getContextClassLoader());
  }

  /**
   * Runs all validations available from the current {@link Thread#getContextClassLoader()}.
   * <p>
   * All validations will be run even if failures are found in some. The result will contain all failures.
   *
   * @param artifact           the artifact to validate
   * @param validationEnricher a postprocessor to apply on every discovered {@link ValidationsProvider}.
   * @return the result of the validation
   *
   * @deprecated since 1.1, use {@link #validatorBuilder()} instead.
   */
  @Deprecated
  public static ValidationResult validate(ArtifactAst artifact, Consumer<ValidationsProvider> validationEnricher) {
    return validate(artifact, validationEnricher, v -> true, currentThread().getContextClassLoader());
  }

  /**
   * Runs all validations available from the current {@link Thread#getContextClassLoader()}.
   * <p>
   * All validations will be run even if failures are found in some. The result will contain all failures.
   *
   * @param artifact          the artifact to validate
   * @param validationsFilter additional filter to apply on the discovered validations before evaluating them.
   * @return the result of the validation
   *
   * @deprecated since 1.1, use {@link #validatorBuilder()} instead.
   */
  @Deprecated
  public static ValidationResult validate(ArtifactAst artifact, Predicate<Validation> validationsFilter) {
    return validate(artifact, v -> {
    }, validationsFilter, currentThread().getContextClassLoader());
  }

  /**
   * Runs all validations available from the current {@link Thread#getContextClassLoader()}.
   * <p>
   * All validations will be run even if failures are found in some. The result will contain all failures.
   *
   * @param artifact                  the artifact to validate
   * @param validationEnricher        a postprocessor to apply on every discovered {@link ValidationsProvider}.
   * @param validationsFilter         additional filter to apply on the discovered validations before evaluating them.
   * @param artifactRegionClassLoader the classLoader to use to load any resources referenced by the artifact
   * @return the result of the validation
   *
   * @since 1.1
   *
   * @deprecated since 1.1, use {@link #validatorBuilder()} instead.
   */
  @Deprecated
  public static ValidationResult validate(ArtifactAst artifact, Consumer<ValidationsProvider> validationEnricher,
                                          Predicate<Validation> validationsFilter,
                                          ClassLoader artifactRegionClassLoader) {
    return validatorBuilder()
        .withValidationEnricher(validationEnricher)
        .withValidationsFilter(validationsFilter)
        .withArtifactRegionClassLoader(artifactRegionClassLoader)
        .build()
        .validate(artifact);
  }

  /**
   * @param artifactAst     full {@link ArtifactAst} to filter.
   * @param componentFilter a {@link Predicate} to filter the {@link ComponentAst AST components}.
   * @return an {@link ArtifactAst} with the components filtered by the given {@code componentFilter}.
   * @since 1.4
   */
  public static ArtifactAst filteredArtifactAst(ArtifactAst artifactAst, Predicate<ComponentAst> componentFilter) {
    return new FilteredArtifactAst(artifactAst, componentFilter);
  }

  /**
   * Compares the values of the given {@link ComponentAst}s for equality behavior-wise.
   * <p>
   * `behavior-wise` comparison means that attributes of the {@link ComponentAst} that do not affect the behavior of the component
   * (for instance the parsing metadata) are NOT taken into account.
   *
   * @param compA one component to compare, non-null.
   * @param compB the other component to compare, non-null.
   * @return {@code true} if the given components are equals behavior-wise, {@code false} otherwise.
   * @since 1.5
   */
  public static boolean areComponentsEquals(ComponentAst compA, ComponentAst compB) {
    if (!Objects.equals(compA.getIdentifier(), compB.getIdentifier())) {
      return false;
    }

    if (compA.getModel(Object.class).orElse(null) != compB.getModel(Object.class).orElse(null)) {
      return false;
    }

    if (compA.getParameters().size() != compB.getParameters().size()) {
      return false;
    }

    if (compA.directChildren().size() != compB.directChildren().size()) {
      return false;
    }

    for (ComponentParameterAst paramA : compA.getParameters()) {
      ComponentParameterAst paramB = compB.getParameter(paramA.getGroupModel().getName(), paramA.getModel().getName());

      if (paramB == null) {
        return false;
      }

      if (!areParametersEquals(paramA, paramB)) {
        return false;
      }
    }

    Iterator<ComponentAst> childrenIteratorB = compB.directChildren().iterator();
    for (ComponentAst childA : compA.directChildren()) {
      ComponentAst childB = childrenIteratorB.next();

      if (!areComponentsEquals(childA, childB)) {
        return false;
      }
    }

    return true;
  }

  /**
   * Compares the values of the given {@link ComponentParameterAst}s for equality.
   *
   * @param paramA one parameter to compare, non-null.
   * @param paramB the other parameter to compare, non-null.
   * @return {@code true} if the values of the given params are equals, {@code false} otherwise.
   * @since 1.5
   */
  public static boolean areParametersEquals(ComponentParameterAst paramA, ComponentParameterAst paramB) {
    if (paramA.getValue() == null && paramB.getValue() == null) {
      return true;
    }

    if (paramA.getValue() != null && paramB.getValue() != null) {
      if (paramA.getValue().getValue().isPresent()) {
        if (!paramB.getValue().getValue().isPresent()) {
          return false;
        }

        return paramA.getValue().reduce(exprA -> paramB.getValue().isLeft() && exprA.equals(paramB.getValue().getLeft()),
                                        fixedValueA -> {
                                          if (!paramB.getValue().isRight()) {
                                            return false;
                                          }

                                          return areParametersValuesEquals(fixedValueA, paramB.getValue().getRight());
                                        });
      } else {
        return !paramB.getValue().getValue().isPresent();
      }
    }

    return false;
  }

  private static boolean areParametersValuesEquals(Object valueA, Object valueB) {
    if (valueA instanceof ComponentAst) {
      if (!(valueB instanceof ComponentAst)) {
        return false;
      }

      return areComponentsEquals((ComponentAst) valueA, (ComponentAst) valueB);
    }

    if (valueA instanceof Collection) {
      if (!(valueB instanceof Collection)) {
        return false;
      }

      if (((Collection) valueA).size() != ((Collection) valueB).size()) {
        return false;
      }

      Iterator<ComponentAst> childrenIteratorB = ((Collection) valueB).iterator();
      for (Object childA : ((Collection) valueA)) {
        Object childB = childrenIteratorB.next();

        if (!areParametersValuesEquals(childA, childB)) {
          return false;
        }
      }
    }

    return valueA.equals(valueB);
  }

}
