/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.builder;

import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.api.builder.ComponentParameterAstBuilder;
import org.mule.runtime.ast.internal.DefaultComponentGenerationInformation;
import org.mule.runtime.ast.internal.DefaultComponentParameterAst;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;

import java.util.List;

// complex
public class DefaultComponentParameterAstBuilder implements ComponentParameterAstBuilder {

  private ParameterModel paramModel;
  private DefaultComponentAstBuilder ownerComponent;

  private ParameterGroupModel parameterGroupModel;
  private ComponentGenerationInformation generationInformation;
  private ComponentMetadataAst metadata;

  private String rawValue;
  private DefaultComponentAstBuilder complexValue;
  private List<ComponentAstBuilder> complexListValue;

  public ComponentParameterAstBuilder withParamModel(ParameterModel paramModel, ParameterGroupModel parameterGroupModel) {
    this.paramModel = paramModel;
    this.parameterGroupModel = parameterGroupModel;

    return this;
  }

  public ComponentParameterAstBuilder withOwnerComponent(ComponentAstBuilder ownerComponent) {
    this.ownerComponent = (DefaultComponentAstBuilder) ownerComponent;

    return this;
  }

  @Override
  public ComponentParameterAstBuilder withSyntax(DslElementSyntax syntax) {
    if (this.generationInformation != null) {
      throw new IllegalStateException("'syntax' already set");
    }

    this.generationInformation = DefaultComponentGenerationInformation
        .builder()
        .withSyntax(syntax)
        .build();

    return this;
  }

  @Override
  public ComponentParameterAstBuilder withMetadata(ComponentMetadataAst metadata) {
    this.metadata = metadata;

    return this;
  }

  @Override
  public ComponentParameterAstBuilder withValue(String rawValue) {
    this.rawValue = rawValue;

    return this;
  }

  @Override
  public ComponentParameterAstBuilder withValue(ComponentAstBuilder complexValue) {
    this.complexValue = (DefaultComponentAstBuilder) complexValue;

    return this;
  }

  @Override
  public ComponentParameterAstBuilder withValue(List<ComponentAstBuilder> complexListValue) {
    this.complexListValue = complexListValue;

    return this;
  }

  @Override
  public ComponentParameterAst build() {
    if (complexValue != null) {
      return new DefaultComponentParameterAst(complexValue,
                                              paramModel,
                                              parameterGroupModel,
                                              metadata,
                                              generationInformation,
                                              ownerComponent.getPropertiesResolver(),
                                              ownerComponent.getParameterModelUtils());
    } else if (complexListValue != null) {
      return new DefaultComponentParameterAst(complexListValue,
                                              paramModel,
                                              parameterGroupModel,
                                              metadata,
                                              generationInformation,
                                              ownerComponent.getPropertiesResolver(),
                                              ownerComponent.getParameterModelUtils());
    }

    // for null, consider it as a simple value param with null value
    return new DefaultComponentParameterAst(rawValue,
                                            paramModel,
                                            parameterGroupModel,
                                            metadata,
                                            generationInformation,
                                            ownerComponent.getPropertiesResolver(),
                                            ownerComponent.getParameterModelUtils());
  }

  public DefaultComponentAstBuilder getComplexValue() {
    return complexValue;
  }

  public boolean isSimpleValue() {
    return rawValue != null;
  }
}
