/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.builder;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnit;
import org.mockito.junit.MockitoRule;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.meta.model.EnrichableModel;
import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.dsl.api.component.config.DefaultComponentLocation;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.stream.Stream;

import static java.util.Arrays.asList;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

public class ComponentLocationVisitorTestCase {

  private static final ComponentIdentifier COMPONENT_IDENTIFIER = ComponentIdentifier.builder()
      .name("fuzzbutt")
      .namespace("xmlns:fuzzy")
      .namespaceUri("http://fuzzbutt.com")
      .build();
  private static final Optional<String> COMPONENT_ID = Optional.of("componentId");
  private static final Optional<String> PARENT_COMPONENT_ID = Optional.of("parentId");
  public static final String CHILD_COMPONENT_INDEX = "-1";
  private static ComponentLocationVisitor visitor;

  @Rule
  public final MockitoRule mockitoRule = MockitoJUnit.rule();

  @Mock
  private DefaultComponentAstBuilder componentModel;
  @Mock
  private ComponentMetadataAst componentMetadata;
  @Mock
  private DefaultComponentAstBuilder parent;
  @Mock
  private DefaultComponentLocation parentLocation;
  @Mock(extraInterfaces = {ConstructModel.class, SourceModel.class, ConnectionProviderModel.class, EnrichableModel.class})
  private ConfigurationModel parentConfigurationModel;
  @Captor
  private ArgumentCaptor<ComponentLocation> locationCaptor;
  @Captor
  private ArgumentCaptor<Optional<TypedComponentIdentifier>> componentIdentifierCaptor;

  @Before
  public void setup() {
    when(componentModel.getMetadata()).thenReturn(componentMetadata);
    when(componentModel.getIdentifier()).thenReturn(COMPONENT_IDENTIFIER);
    visitor = new ComponentLocationVisitor();
  }

  @Test
  public void resolveLocationWithNoHierarchy() {
    // Type has to be set
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.CONFIG);
    when(componentModel.getComponentId()).thenReturn(COMPONENT_ID);

    visitor.resolveLocation(componentModel, asList());

    verify(componentModel).withLocation(locationCaptor.capture());
    assertThat(locationCaptor.getValue().getLocation(), is("componentId"));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_Chain() {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.CHAIN);
    setUpParent(parent, TypedComponentIdentifier.ComponentType.CONFIG);

    visitor.resolveLocation(componentModel, asList(parent));

    verify(componentModel).withLocation(locationCaptor.capture());
    assertThat(locationCaptor.getValue().getLocation(), is("parentId"));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_RootParent() {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.CONFIG);
    setUpParent(parent, TypedComponentIdentifier.ComponentType.CONFIG);
    setUpTopLevelParent();

    visitor.resolveLocation(componentModel, asList(parent));

    verify(componentModel).withLocation(any());
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_Config() {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.CONFIG);
    setUpParent(parent, TypedComponentIdentifier.ComponentType.CONFIG);

    visitor.resolveLocation(componentModel, asList(parent));

    verify(parentLocation).appendLocationPart(eq("0"), componentIdentifierCaptor.capture(), any(), (OptionalInt) any(), any());
    assertThat(componentIdentifierCaptor.getValue().isPresent(), is(true));
    assertThat(componentIdentifierCaptor.getValue().get().getType(), is(TypedComponentIdentifier.ComponentType.CONFIG));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_Config_customModel()
      throws InvocationTargetException, IllegalAccessException {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.CONFIG);
    when(componentModel.getModel(EnrichableModel.class)).thenReturn(Optional.of(parentConfigurationModel));
    final ModelProperty modelProperty = mock(ModelProperty.class);
    when(parentConfigurationModel.getModelProperty(any())).thenReturn(Optional.of(modelProperty));
    setUpParent(parent, TypedComponentIdentifier.ComponentType.CONFIG);
    final ComponentLocationVisitor visitorSpy = spy(visitor);
    doReturn(ModelPropertyHarness.class).when(visitorSpy).getModelPropertyClass();
    doReturn(true).when(visitorSpy).isModelIndexed(any());
    doReturn("CustomModelPath").when(visitorSpy).getCustomModelLocationPath(any());
    DefaultComponentLocation newProcessor = mock(DefaultComponentLocation.class);
    when(parentLocation.appendLocationPart(any(), any(), any(), (OptionalInt) any(), any())).thenReturn(newProcessor);
    final DefaultComponentLocation innerLocation = mock(DefaultComponentLocation.class);
    when(newProcessor.appendLocationPart(any(), any(), any(), (OptionalInt) any(), any())).thenReturn(innerLocation);


    visitorSpy.resolveLocation(componentModel, asList(parent));

    verify(newProcessor).appendLocationPart(eq("0"), componentIdentifierCaptor.capture(), any(), (OptionalInt) any(), any());
    assertThat(componentIdentifierCaptor.getValue().isPresent(), is(true));
    assertThat(componentIdentifierCaptor.getValue().get().getType(), is(TypedComponentIdentifier.ComponentType.CONFIG));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_Config_customModel_badConfig()
      throws InvocationTargetException, IllegalAccessException {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.CONFIG);
    when(componentModel.getModel(EnrichableModel.class)).thenReturn(Optional.of(parentConfigurationModel));
    final ModelProperty modelProperty = mock(ModelProperty.class);
    when(parentConfigurationModel.getModelProperty(any())).thenReturn(Optional.of(modelProperty));
    setUpParent(parent, TypedComponentIdentifier.ComponentType.CONFIG);
    final ComponentLocationVisitor visitorSpy = spy(visitor);
    doReturn(ModelPropertyHarness.class).when(visitorSpy).getModelPropertyClass();
    doReturn(true).when(visitorSpy).isModelIndexed(any());
    doThrow(new ComponentLocationVisitor.BadCustomModelPropertyConfigException(new NullPointerException("Test"))).when(visitorSpy)
        .getCustomModelLocationPath(any());

    visitorSpy.resolveLocation(componentModel, asList(parent));

    verify(parentLocation).appendLocationPart(eq("0"), componentIdentifierCaptor.capture(), any(), (OptionalInt) any(), any());
    assertThat(componentIdentifierCaptor.getValue().isPresent(), is(true));
    assertThat(componentIdentifierCaptor.getValue().get().getType(), is(TypedComponentIdentifier.ComponentType.CONFIG));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_Config_customModel_notIndexed()
      throws InvocationTargetException, IllegalAccessException {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.CONFIG);
    when(componentModel.getModel(EnrichableModel.class)).thenReturn(Optional.of(parentConfigurationModel));
    final ModelProperty modelProperty = mock(ModelProperty.class);
    when(parentConfigurationModel.getModelProperty(any())).thenReturn(Optional.of(modelProperty));
    setUpParent(parent, TypedComponentIdentifier.ComponentType.CONFIG);
    final ComponentLocationVisitor visitorSpy = spy(visitor);
    doReturn(ModelPropertyHarness.class).when(visitorSpy).getModelPropertyClass();
    doReturn(false).when(visitorSpy).isModelIndexed(any());
    doReturn("CustomModelPath").when(visitorSpy).getCustomModelLocationPath(any());
    DefaultComponentLocation newProcessor = mock(DefaultComponentLocation.class);
    when(parentLocation.appendLocationPart(any(), any(), any(), (OptionalInt) any(), any())).thenReturn(newProcessor);

    visitorSpy.resolveLocation(componentModel, asList(parent));

    verify(parentLocation).appendLocationPart(eq("CustomModelPath"), componentIdentifierCaptor.capture(), any(),
                                              (OptionalInt) any(), any());
    assertThat(componentIdentifierCaptor.getValue().isPresent(), is(true));
    assertThat(componentIdentifierCaptor.getValue().get().getType(), is(TypedComponentIdentifier.ComponentType.CONFIG));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_Config_deprecatedCustomModel()
      throws InvocationTargetException, IllegalAccessException {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.CONFIG);
    when(componentModel.getModel(EnrichableModel.class)).thenReturn(Optional.of(parentConfigurationModel));
    final ModelProperty modelProperty = mock(ModelProperty.class);
    when(parentConfigurationModel.getModelProperty(any())).thenReturn(Optional.of(modelProperty));
    setUpParent(parent, TypedComponentIdentifier.ComponentType.CONFIG);
    final ComponentLocationVisitor visitorSpy = spy(visitor);
    doReturn(ModelPropertyHarness.class).when(visitorSpy).getDeprecatedModelPropertyClass();
    doReturn(true).when(visitorSpy).isDeprecatedModelIndexed(any());
    doReturn("CustomModelPath").when(visitorSpy).getDeprecatedLocationPath(any());
    DefaultComponentLocation newProcessor = mock(DefaultComponentLocation.class);
    when(parentLocation.appendLocationPart(any(), any(), any(), (OptionalInt) any(), any())).thenReturn(newProcessor);
    final DefaultComponentLocation innerLocation = mock(DefaultComponentLocation.class);
    when(newProcessor.appendLocationPart(any(), any(), any(), (OptionalInt) any(), any())).thenReturn(innerLocation);


    visitorSpy.resolveLocation(componentModel, asList(parent));

    verify(newProcessor).appendLocationPart(eq("0"), componentIdentifierCaptor.capture(), any(), (OptionalInt) any(), any());
    assertThat(componentIdentifierCaptor.getValue().isPresent(), is(true));
    assertThat(componentIdentifierCaptor.getValue().get().getType(), is(TypedComponentIdentifier.ComponentType.CONFIG));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_ErrorHandler() {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.ERROR_HANDLER);
    setUpParent(parent, TypedComponentIdentifier.ComponentType.CONFIG);

    visitor.resolveLocation(componentModel, asList(parent));

    verify(parentLocation).appendLocationPart(eq("errorHandler"), componentIdentifierCaptor.capture(), any(), (OptionalInt) any(),
                                              any());
    assertThat(componentIdentifierCaptor.getValue().isPresent(), is(true));
    assertThat(componentIdentifierCaptor.getValue().get().getType(), is(TypedComponentIdentifier.ComponentType.ERROR_HANDLER));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_ErrorTemplate() {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.ON_ERROR);
    setUpParent(parent, TypedComponentIdentifier.ComponentType.CONFIG);
    when(parent.childComponentsStream()).thenReturn(Stream.of(componentModel));

    visitor.resolveLocation(componentModel, asList(parent));

    verify(parentLocation).appendLocationPart(eq("0"), componentIdentifierCaptor.capture(), any(), (OptionalInt) any(),
                                              any());
    assertThat(componentIdentifierCaptor.getValue().isPresent(), is(true));
    assertThat(componentIdentifierCaptor.getValue().get().getType(), is(TypedComponentIdentifier.ComponentType.ON_ERROR));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_Processor() {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.OPERATION);
    setUpParent(parent, TypedComponentIdentifier.ComponentType.CONFIG);
    when(parent.getModel(SourceModel.class)).thenReturn(Optional.empty());
    DefaultComponentLocation newProcessor = mock(DefaultComponentLocation.class);
    when(parentLocation.appendProcessorsPart()).thenReturn(newProcessor);

    visitor.resolveLocation(componentModel, asList(parent));

    verify(newProcessor).appendLocationPart(eq(CHILD_COMPONENT_INDEX), componentIdentifierCaptor.capture(), any(),
                                            (OptionalInt) any(), any());
    assertThat(componentIdentifierCaptor.getValue().isPresent(), is(true));
    assertThat(componentIdentifierCaptor.getValue().get().getType(), is(TypedComponentIdentifier.ComponentType.OPERATION));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_Connection() {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.CONFIG);
    setUpParent(parent, TypedComponentIdentifier.ComponentType.CONFIG);
    when(parent.getModel(SourceModel.class)).thenReturn(Optional.empty());
    when(componentModel.getModel(ConnectionProviderModel.class))
        .thenReturn(Optional.of((ConnectionProviderModel) parentConfigurationModel));

    visitor.resolveLocation(componentModel, asList(parent));

    verify(parentLocation).appendConnectionPart(componentIdentifierCaptor.capture(), any(), any(), any());
    assertThat(componentIdentifierCaptor.getValue().isPresent(), is(true));
    assertThat(componentIdentifierCaptor.getValue().get().getType(), is(TypedComponentIdentifier.ComponentType.CONFIG));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_Flow_Processor() {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.OPERATION);
    setUpParent(parent, TypedComponentIdentifier.ComponentType.CONFIG);
    when(parent.getModel(SourceModel.class)).thenReturn(Optional.empty());
    DefaultComponentLocation newProcessor = mock(DefaultComponentLocation.class);
    when(parentLocation.appendLocationPart(any(), any(), any(), (OptionalInt) any(), any())).thenReturn(newProcessor);
    setUpTopLevelParent();

    visitor.resolveLocation(componentModel, asList(parent));

    verify(newProcessor).appendLocationPart(eq(CHILD_COMPONENT_INDEX), componentIdentifierCaptor.capture(), any(),
                                            (OptionalInt) any(), any());
    assertThat(componentIdentifierCaptor.getValue().isPresent(), is(true));
    assertThat(componentIdentifierCaptor.getValue().get().getType(), is(TypedComponentIdentifier.ComponentType.OPERATION));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_MessageSource() {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.SOURCE);
    setUpParent(parent, TypedComponentIdentifier.ComponentType.CONFIG);
    when(parent.getModel(SourceModel.class)).thenReturn(Optional.empty());
    DefaultComponentLocation newProcessor = mock(DefaultComponentLocation.class);
    when(parentLocation.appendProcessorsPart()).thenReturn(newProcessor);
    setUpTopLevelParent();

    visitor.resolveLocation(componentModel, asList(parent));

    verify(parentLocation).appendLocationPart(eq("source"), componentIdentifierCaptor.capture(), any(),
                                              (OptionalInt) any(), any());
    assertThat(componentIdentifierCaptor.getValue().isPresent(), is(true));
    assertThat(componentIdentifierCaptor.getValue().get().getType(), is(TypedComponentIdentifier.ComponentType.SOURCE));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_Root_ErrorHandler() {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.ERROR_HANDLER);
    setUpParent(parent, TypedComponentIdentifier.ComponentType.CONFIG);
    when(parent.getModel(SourceModel.class)).thenReturn(Optional.empty());
    setUpTopLevelParent();

    visitor.resolveLocation(componentModel, asList(parent));

    verify(parentLocation).appendLocationPart(eq("errorHandler"), componentIdentifierCaptor.capture(), any(),
                                              (OptionalInt) any(), any());
    assertThat(componentIdentifierCaptor.getValue().isPresent(), is(true));
    assertThat(componentIdentifierCaptor.getValue().get().getType(), is(TypedComponentIdentifier.ComponentType.ERROR_HANDLER));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_Config_ParentRouter() {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.CONFIG);
    setUpParent(parent, TypedComponentIdentifier.ComponentType.ROUTER);

    visitor.resolveLocation(componentModel, asList(parent));

    verify(parentLocation).appendLocationPart(eq("0"), componentIdentifierCaptor.capture(), any(), (OptionalInt) any(), any());
    assertThat(componentIdentifierCaptor.getValue().isPresent(), is(true));
    assertThat(componentIdentifierCaptor.getValue().get().getType(), is(TypedComponentIdentifier.ComponentType.CONFIG));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_Config_ParentAndChildRouter() {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.ROUTE);
    setUpParent(parent, TypedComponentIdentifier.ComponentType.ROUTER);
    final DefaultComponentLocation newLocation = mock(DefaultComponentLocation.class);
    when(parentLocation.appendRoutePart()).thenReturn(newLocation);

    visitor.resolveLocation(componentModel, asList(parent));

    verify(parentLocation).appendRoutePart();
    verify(newLocation).appendLocationPart(eq(CHILD_COMPONENT_INDEX), componentIdentifierCaptor.capture(), any(),
                                           (OptionalInt) any(), any());
    assertThat(componentIdentifierCaptor.getValue().isPresent(), is(true));
    assertThat(componentIdentifierCaptor.getValue().get().getType(), is(TypedComponentIdentifier.ComponentType.ROUTE));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneThatHasAConfigurationModel_Config_ParentRouter_ChildProcessor() {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.OPERATION);
    setUpParent(parent, TypedComponentIdentifier.ComponentType.ROUTER);
    final DefaultComponentLocation newLocation = mock(DefaultComponentLocation.class);
    when(parentLocation.appendRoutePart()).thenReturn(newLocation);
    final DefaultComponentLocation routeLocation = mock(DefaultComponentLocation.class);
    when(newLocation.appendLocationPart(any(), any(), any(), (OptionalInt) any(), any())).thenReturn(routeLocation);
    final DefaultComponentLocation newProcessorLocation = mock(DefaultComponentLocation.class);
    when(routeLocation.appendProcessorsPart()).thenReturn(newProcessorLocation);

    visitor.resolveLocation(componentModel, asList(parent));

    verify(parentLocation).appendRoutePart();
    verify(newLocation).appendLocationPart(eq(CHILD_COMPONENT_INDEX), any(), any(), eq(OptionalInt.empty()),
                                           eq(OptionalInt.empty()));
    verify(routeLocation).appendProcessorsPart();
    verify(newProcessorLocation).appendLocationPart(eq("0"), componentIdentifierCaptor.capture(), any(), (OptionalInt) any(),
                                                    any());
    assertThat(componentIdentifierCaptor.getValue().isPresent(), is(true));
    assertThat(componentIdentifierCaptor.getValue().get().getType(), is(TypedComponentIdentifier.ComponentType.OPERATION));
  }

  @Test
  public void resolveLocationWithHierarchyOfOneNotRoot() {
    when(componentModel.getComponentType()).thenReturn(TypedComponentIdentifier.ComponentType.CONFIG);
    setUpParent(parent, TypedComponentIdentifier.ComponentType.CONFIG);

    visitor.resolveLocation(componentModel, asList(parent));

    verify(parentLocation).appendLocationPart(eq("0"), componentIdentifierCaptor.capture(), any(), (OptionalInt) any(), any());
    assertThat(componentIdentifierCaptor.getValue().isPresent(), is(true));
    assertThat(componentIdentifierCaptor.getValue().get().getType(), is(TypedComponentIdentifier.ComponentType.CONFIG));
  }

  @Test
  public void checkThatGetBooleanBehavesProperly() throws NoSuchMethodException {
    final Method m = ModelPropertyHarness.class.getDeclaredMethod("isPublic");
    final ModelPropertyHarness model = new ModelPropertyHarness();
    assertThat(ComponentLocationVisitor.getBoolean(m, model), is(true));
    assertThat(ComponentLocationVisitor.getBoolean(null, model), is(false));
  }

  @Test
  public void checkThatGetStringBehavesProperly() throws NoSuchMethodException {
    final Method m = ModelPropertyHarness.class.getDeclaredMethod("getName");
    final ModelPropertyHarness model = new ModelPropertyHarness();
    assertThat(ComponentLocationVisitor.getString(m, model), is("TestPropertyModel"));
    assertThat(ComponentLocationVisitor.getString(null, model), is(nullValue()));
  }

  private void setUpParent(DefaultComponentAstBuilder parent, TypedComponentIdentifier.ComponentType config) {
    when(parent.getComponentType()).thenReturn(config);
    when(parent.getComponentId()).thenReturn(PARENT_COMPONENT_ID);
    when(parent.getModel(ConfigurationModel.class)).thenReturn(Optional.of(parentConfigurationModel));
    when(parent.getLocation()).thenReturn(parentLocation);
  }

  private void setUpTopLevelParent() {
    when(parent.getModel(ConstructModel.class)).thenReturn(Optional.of((ConstructModel) parentConfigurationModel));
    when(((ConstructModel) parentConfigurationModel).allowsTopLevelDeclaration()).thenReturn(true);
  }


  static class ModelPropertyHarness implements ModelProperty {

    @Override
    public String getName() {
      return "TestPropertyModel";
    }

    @Override
    public boolean isPublic() {
      return true;
    }
  }
}
