/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.error;

import static org.mule.runtime.api.component.ComponentIdentifier.builder;
import static org.mule.runtime.api.functional.Either.right;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.ERROR_MAPPINGS;
import static org.mule.runtime.ast.api.error.ErrorTypeRepositoryProvider.getCoreErrorTypeRepo;
import static org.mule.runtime.ast.internal.error.ErrorTypeRepositoryBuildingUtils.ERROR_TYPE_PARAM;
import static org.mule.runtime.ast.internal.error.ErrorTypeRepositoryBuildingUtils.RAISE_ERROR_IDENTIFIER;
import static org.mule.runtime.ast.internal.error.ErrorTypeRepositoryBuildingUtils.addErrorsFromArtifact;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.Errors.ERROR_TYPES;
import static org.mule.runtime.extension.api.ExtensionConstants.ERROR_MAPPINGS_PARAMETER_NAME;

import static java.util.Collections.singletonList;
import static java.util.Optional.of;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.error.ErrorTypeRepositoryProvider;
import org.mule.runtime.ast.internal.DefaultComponentMetadataAst;
import org.mule.runtime.extension.api.error.ErrorMapping;

import java.util.ServiceLoader;
import java.util.stream.Stream;

import org.junit.Before;
import org.junit.Test;

import org.mockito.MockedStatic;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(ERROR_TYPES)
public class ErrorTypeRepositoryBuildingUtilsTestCase {

  private ComponentMetadataAst metadata;
  private OperationModel someOperationModel;
  private ArtifactAst artifactAst;
  private ComponentAst chain;

  private ErrorTypeRepository errorTypeRepository;

  @Before
  public void setUp() {
    metadata = DefaultComponentMetadataAst.builder()
        .setFileName("mule-app.xml")
        .build();

    someOperationModel = mock(OperationModel.class);

    chain = mock(ComponentAst.class);
    when(chain.getMetadata()).thenReturn(metadata);
    artifactAst = mock(ArtifactAst.class);
    when(artifactAst.topLevelComponentsStream()).thenReturn(Stream.of(chain));

    errorTypeRepository = mock(ErrorTypeRepository.class);
  }

  private void withMockCoreErrorTypeRepository(Runnable test) {
    try (MockedStatic<ServiceLoader> loaderClass = mockStatic(ServiceLoader.class)) {
      ServiceLoader fromRuntimeLoader = mock(ServiceLoader.class);
      when(fromRuntimeLoader.iterator())
          .thenAnswer(inv -> singletonList((ErrorTypeRepositoryProvider) (() -> errorTypeRepository)).iterator());

      loaderClass.when(() -> ServiceLoader.load(eq(ErrorTypeRepositoryProvider.class), any(ClassLoader.class)))
          .thenReturn(fromRuntimeLoader);

      test.run();
    }
  }

  @Test
  public void errorTypesFromRaiseError() {
    final ComponentParameterAst errorTypeParam = mock(ComponentParameterAst.class);
    when(errorTypeParam.getValue()).thenReturn(right("TEST:RAISED_ERROR"));

    final ComponentAst raiseError = mock(ComponentAst.class);
    when(raiseError.getIdentifier()).thenReturn(RAISE_ERROR_IDENTIFIER);
    when(raiseError.getParameter(DEFAULT_GROUP_NAME, ERROR_TYPE_PARAM)).thenReturn(errorTypeParam);
    when(raiseError.getMetadata()).thenReturn(metadata);
    when(raiseError.getLocation()).thenReturn(mock(ComponentLocation.class));

    when(chain.directChildrenStream()).thenReturn(Stream.of(raiseError));

    withMockCoreErrorTypeRepository(() -> {
      addErrorsFromArtifact(artifactAst, errorTypeRepository);

      verify(errorTypeRepository).addErrorType(builder().namespace("TEST").name("RAISED_ERROR").build(),
                                               getCoreErrorTypeRepo().getAnyErrorType());
    });
  }

  @Test
  public void errorTypesFromErrorMappings() {
    final ComponentParameterAst errorMappingsParam = mock(ComponentParameterAst.class);
    when(errorMappingsParam.getValue())
        .thenReturn(right(singletonList(new ErrorMapping("TEST-EXT:OP_ERROR", "TEST:MAPPED_ERROR"))));

    final ComponentAst someOperation = mock(ComponentAst.class);
    when(someOperation.getIdentifier()).thenReturn(builder().namespace("test-ext").name("operation").build());
    when(someOperation.getParameter(ERROR_MAPPINGS, ERROR_MAPPINGS_PARAMETER_NAME)).thenReturn(errorMappingsParam);
    when(someOperation.getMetadata()).thenReturn(metadata);
    when(someOperation.getLocation()).thenReturn(mock(ComponentLocation.class));
    when(someOperation.getModel(OperationModel.class)).thenReturn(of(someOperationModel));

    when(chain.directChildrenStream()).thenReturn(Stream.of(someOperation));

    withMockCoreErrorTypeRepository(() -> {
      addErrorsFromArtifact(artifactAst, errorTypeRepository);

      verify(errorTypeRepository).addErrorType(builder().namespace("TEST").name("MAPPED_ERROR").build(),
                                               getCoreErrorTypeRepo().getAnyErrorType());
    });
  }

  @Test
  public void coreErrorTypesFromRaiseError() {
    final ComponentParameterAst errorTypeParam = mock(ComponentParameterAst.class);
    when(errorTypeParam.getValue()).thenReturn(right("RAISED_ERROR"));

    final ComponentAst raiseError = mock(ComponentAst.class);
    when(raiseError.getIdentifier()).thenReturn(RAISE_ERROR_IDENTIFIER);
    when(raiseError.getParameter(DEFAULT_GROUP_NAME, ERROR_TYPE_PARAM)).thenReturn(errorTypeParam);
    when(raiseError.getMetadata()).thenReturn(metadata);
    when(raiseError.getLocation()).thenReturn(mock(ComponentLocation.class));

    when(chain.directChildrenStream()).thenReturn(Stream.of(raiseError));

    withMockCoreErrorTypeRepository(() -> {
      addErrorsFromArtifact(artifactAst, errorTypeRepository);

      verify(errorTypeRepository, never()).addErrorType(any(), any());
    });
  }

  @Test
  public void coreErrorTypesFromErrorMappings() {
    final ComponentParameterAst errorMappingsParam = mock(ComponentParameterAst.class);
    when(errorMappingsParam.getValue())
        .thenReturn(right(singletonList(new ErrorMapping("TEST-EXT:OP_ERROR", "MAPPED_ERROR"))));

    final ComponentAst someOperation = mock(ComponentAst.class);
    when(someOperation.getIdentifier()).thenReturn(builder().namespace("test-ext").name("operation").build());
    when(someOperation.getParameter(ERROR_MAPPINGS, ERROR_MAPPINGS_PARAMETER_NAME)).thenReturn(errorMappingsParam);
    when(someOperation.getMetadata()).thenReturn(metadata);
    when(someOperation.getLocation()).thenReturn(mock(ComponentLocation.class));
    when(someOperation.getModel(OperationModel.class)).thenReturn(of(someOperationModel));

    when(chain.directChildrenStream()).thenReturn(Stream.of(someOperation));

    withMockCoreErrorTypeRepository(() -> {
      addErrorsFromArtifact(artifactAst, errorTypeRepository);

      verify(errorTypeRepository, never()).addErrorType(any(), any());
    });
  }
}
