/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.api.util;

import static org.mule.runtime.api.functional.Either.right;
import static org.mule.runtime.ast.api.util.MuleArtifactAstCopyUtils.copyComponentTreeRecursively;
import static org.mule.runtime.ast.api.util.MuleArtifactAstCopyUtils.copyRecursively;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.AstTraversal.AST_TRAVERSAL;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.Optional.of;
import static java.util.function.UnaryOperator.identity;
import static java.util.stream.Collectors.toList;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.instanceOf;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.CoreMatchers.sameInstance;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.util.AstTraversalDirection;
import org.mule.runtime.ast.api.util.BaseArtifactAst;
import org.mule.runtime.ast.api.util.BaseComponentAst;
import org.mule.runtime.ast.api.util.BaseComponentAstDecorator;

import java.util.Collections;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Stream;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(AST_TRAVERSAL)
public class MuleArtifactAstCopyUtilsTestCase {

  private ArtifactAst artifact;

  private ComponentAst processor;

  private ComponentAst topLevelComponent;

  private ComponentAst randomComponent;

  private ComponentAst withParamComponent;
  private ComponentAst complexParamValue;

  @Before
  public void after() {
    processor = mock(BaseComponentAst.class);
    when(processor.directChildren()).thenAnswer(inv -> emptyList());
    when(processor.directChildrenStream()).thenAnswer(inv -> Stream.empty());
    when(processor.recursiveStream()).thenCallRealMethod();
    when(processor.recursiveStream(any(AstTraversalDirection.class))).thenAnswer(inv -> Stream.of(processor));

    topLevelComponent = mock(BaseComponentAst.class);
    when(topLevelComponent.directChildren()).thenAnswer(inv -> singletonList(processor));
    when(topLevelComponent.directChildrenStream()).thenAnswer(inv -> Stream.of(processor));
    when(topLevelComponent.recursiveStream()).thenCallRealMethod();
    when(topLevelComponent.recursiveStream(any(AstTraversalDirection.class)))
        .thenAnswer(inv -> Stream.of(topLevelComponent, processor));

    randomComponent = mock(BaseComponentAst.class);
    when(randomComponent.directChildren()).thenAnswer(inv -> emptyList());
    when(randomComponent.directChildrenStream()).thenAnswer(inv -> Stream.empty());
    when(randomComponent.recursiveStream()).thenCallRealMethod();
    when(randomComponent.recursiveStream(any(AstTraversalDirection.class))).thenAnswer(inv -> Stream.of(randomComponent));

    artifact = mock(BaseArtifactAst.class);
    when(artifact.recursiveStream()).thenCallRealMethod();
    when(artifact.topLevelComponentsStream()).thenAnswer(inv -> Stream.of(topLevelComponent));
    when(artifact.recursiveStream(any(AstTraversalDirection.class)))
        .thenAnswer(inv -> artifact.topLevelComponentsStream().flatMap(c -> c.recursiveStream(inv.getArgument(0))));

    withParamComponent = mock(BaseComponentAst.class);
    when(withParamComponent.getModel(ParameterizedModel.class)).thenReturn(of(mock(ParameterizedModel.class)));
    when(withParamComponent.directChildren()).thenAnswer(inv -> singletonList(processor));
    when(withParamComponent.directChildrenStream()).thenAnswer(inv -> Stream.of(processor));
    when(withParamComponent.recursiveStream()).thenCallRealMethod();
    when(withParamComponent.recursiveStream(any(AstTraversalDirection.class)))
        .thenAnswer(inv -> Stream.of(withParamComponent, processor));

    complexParamValue = mock(BaseComponentAst.class);

    ObjectType objectType = mock(ObjectType.class);
    doAnswer(inv -> {
      inv.getArgument(0, MetadataTypeVisitor.class).visitObject(objectType);
      return null;
    }).when(objectType).accept(any());

    ParameterModel complexParamModel = mock(ParameterModel.class);
    when(complexParamModel.getName()).thenReturn("complexParam");
    when(complexParamModel.getType()).thenReturn(objectType);

    ComponentParameterAst complexParam = mock(ComponentParameterAst.class);
    when(complexParam.getValue()).thenReturn(right(complexParamValue));
    when(complexParam.getModel()).thenReturn(complexParamModel);

    when(withParamComponent.getParameters()).thenReturn(singletonList(complexParam));
  }

  @Test
  public void copyComponentIdentityMap() {
    final ComponentAst copy = copyComponentTreeRecursively(topLevelComponent, identity());
    assertThat(copy, sameInstance(topLevelComponent));
  }

  @Test
  public void copyComponentMapChild() {
    final ComponentAst copy = copyComponentTreeRecursively(topLevelComponent, c -> {
      if (c == processor) {
        return new BaseComponentAstDecorator(c) {};
      } else {
        return c;
      }
    });
    assertThat(copy, not(sameInstance(topLevelComponent)));
    assertThat(copy.directChildrenStream().findFirst().get(), instanceOf(BaseComponentAstDecorator.class));
  }

  @Test
  public void copyComponentMapParent() {
    final ComponentAst copy = copyComponentTreeRecursively(topLevelComponent, c -> {
      if (c == topLevelComponent) {
        return new BaseComponentAstDecorator(c) {};
      } else {
        return c;
      }
    });
    assertThat(copy, not(sameInstance(topLevelComponent)));
    assertThat(copy.directChildrenStream().findFirst().get(), sameInstance(processor));
  }

  @Test
  public void copyComponentHierarchy() {
    AtomicReference<ComponentAst> mappedParent = new AtomicReference<>();
    AtomicReference<ComponentAst> mappedChild = new AtomicReference<>();

    final ComponentAst copy = copyComponentTreeRecursively(topLevelComponent, c -> {
      if (c == processor) {
        final BaseComponentAstDecorator baseComponentAstDecorator = new BaseComponentAstDecorator(c) {};
        mappedChild.set(baseComponentAstDecorator);
        return baseComponentAstDecorator;
      } else {
        final BaseComponentAstDecorator baseComponentAstDecorator = new BaseComponentAstDecorator(c) {};
        mappedParent.set(baseComponentAstDecorator);
        return baseComponentAstDecorator;
      }
    });

    assertThat(copy.recursiveStream().collect(toList()),
               equalTo(asList(mappedParent.get(), mappedChild.get())));
  }

  @Test
  public void addComponentToArtifact() {
    final ArtifactAst copy = copyRecursively(artifact, identity(), () -> singletonList(randomComponent), c -> false);

    assertThat(copy.recursiveStream().collect(toList()),
               equalTo(asList(topLevelComponent, processor, randomComponent)));
    assertThat(copy.topLevelComponentsStream().collect(toList()),
               equalTo(asList(topLevelComponent, randomComponent)));
  }

  @Test
  public void removeComponentToArtifact() {
    final ArtifactAst copyMoreTopLevel = copyRecursively(artifact, identity(), () -> singletonList(randomComponent), c -> false);
    final ArtifactAst copy = copyRecursively(copyMoreTopLevel, identity(), Collections::emptyList, c -> c == topLevelComponent);

    assertThat(copy.topLevelComponentsStream().collect(toList()),
               equalTo(asList(randomComponent)));
    assertThat(copy.recursiveStream().collect(toList()),
               equalTo(asList(randomComponent)));

  }

  @Test
  @Issue("MULE-19572")
  public void complexParamMapped() {
    final ComponentAst copy = copyComponentTreeRecursively(withParamComponent, c -> {
      if (c == complexParamValue) {
        return new BaseComponentAstDecorator(c) {};
      } else {
        return c;
      }
    });
    ComponentParameterAst complexParamCopy = copy.getParameters().iterator().next();
    assertThat(complexParamCopy.getValue().getRight(), not(sameInstance(complexParamValue)));
    assertThat(complexParamCopy.getValue().getRight(), instanceOf(BaseComponentAstDecorator.class));
  }

  @Test
  @Issue("W-14626100")
  public void complexParameterMappedTwiceShouldReturnSameAst() {
    BaseComponentAstDecorator expected = new BaseComponentAstDecorator(complexParamValue) {};
    final ComponentAst firstCopy = copyComponentTreeRecursively(withParamComponent, c -> {
      if (c == complexParamValue) {
        return expected;
      } else {
        return c;
      }
    });
    // The AST should not be mutated at all if no transformations happened as part of the copy.
    final ComponentAst secondCopy = copyComponentTreeRecursively(firstCopy, c -> {
      if (c == complexParamValue) {
        return new BaseComponentAstDecorator(complexParamValue) {};
      } else {
        return c;
      }
    });
    assertThat(firstCopy, is(sameInstance(secondCopy)));
  }

  @Test(expected = UnsupportedOperationException.class)
  @Issue("W-11208606")
  public void artifactCopyTopLevelElementsImmutable() {
    final ArtifactAst copy = copyRecursively(artifact, identity(), () -> singletonList(randomComponent), c -> false);

    copy.topLevelComponents().clear();
  }

  @Test(expected = UnsupportedOperationException.class)
  @Issue("W-11208606")
  public void componentCopyDirectChildrenImmutable() {
    final ComponentAst copy =
        copyComponentTreeRecursively(topLevelComponent,
                                     comp -> comp == processor ? mock(ComponentAst.class) : comp);

    copy.directChildren().clear();
  }
}
