/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.error;

import static java.util.Optional.empty;
import static java.util.stream.Collectors.toSet;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.ast.api.ArtifactAst;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

/**
 * This is the actual instance returned by {@link ArtifactAst#getErrorTypeRepository()} default implementation.
 *
 * @since 1.0
 */
public final class CompositeErrorTypeRepository implements ErrorTypeRepository {

  private final List<ErrorTypeRepository> children;

  public CompositeErrorTypeRepository(List<ErrorTypeRepository> children) {
    this.children = children;
  }

  @Override
  public ErrorType addErrorType(ComponentIdentifier errorTypeIdentifier, ErrorType parentErrorType) {
    throw new UnsupportedOperationException();
  }

  @Override
  public ErrorType addInternalErrorType(ComponentIdentifier errorTypeIdentifier, ErrorType parentErrorType) {
    throw new UnsupportedOperationException();
  }

  @Override
  public Optional<ErrorType> lookupErrorType(ComponentIdentifier errorTypeComponentIdentifier) {
    for (ErrorTypeRepository child : children) {
      final Optional<ErrorType> lookupErrorType = child.lookupErrorType(errorTypeComponentIdentifier);
      if (lookupErrorType.isPresent()) {
        return lookupErrorType;
      }
    }

    return empty();
  }

  @Override
  public Optional<ErrorType> getErrorType(ComponentIdentifier errorTypeIdentifier) {
    return children.stream()
        .map(child -> child.getErrorType(errorTypeIdentifier))
        .filter(Optional::isPresent)
        .map(Optional::get)
        .findFirst();
  }

  @Override
  public Collection<String> getErrorNamespaces() {
    return children.stream()
        .flatMap(c -> c.getErrorNamespaces().stream())
        .collect(toSet());
  }

  @Override
  public ErrorType getAnyErrorType() {
    return children.stream()
        .map(ErrorTypeRepository::getAnyErrorType)
        .filter(Objects::nonNull)
        .findFirst()
        .orElse(null);
  }

  @Override
  public ErrorType getSourceErrorType() {
    return children.stream()
        .map(ErrorTypeRepository::getSourceErrorType)
        .filter(Objects::nonNull)
        .findFirst()
        .orElse(null);
  }

  @Override
  public ErrorType getSourceResponseErrorType() {
    return children.stream()
        .map(ErrorTypeRepository::getSourceResponseErrorType)
        .filter(Objects::nonNull)
        .findFirst()
        .orElse(null);
  }

  @Override
  public ErrorType getCriticalErrorType() {
    return children.stream()
        .map(ErrorTypeRepository::getCriticalErrorType)
        .filter(Objects::nonNull)
        .findFirst()
        .orElse(null);
  }

  @Override
  public Set<ErrorType> getErrorTypes() {
    return children
        .stream()
        .flatMap(repo -> repo.getErrorTypes().stream())
        .collect(toSet());
  }

  @Override
  public Set<ErrorType> getInternalErrorTypes() {
    return children
        .stream()
        .flatMap(repo -> repo.getInternalErrorTypes().stream())
        .collect(toSet());
  }
}
