/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.builder;

import static java.util.Collections.singleton;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnit;
import org.mockito.junit.MockitoRule;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ArtifactType;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.builder.ComponentMetadataAstBuilder;
import org.mule.runtime.ast.api.model.ExtensionModelHelper;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;

import java.util.Optional;
import java.util.function.UnaryOperator;

import static org.hamcrest.CoreMatchers.hasItem;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

public class DefaultArtifactAstBuilderTestCase {

  public static final String NAME = "Name";
  @Rule
  public final MockitoRule mockitoRule = MockitoJUnit.rule();

  private DefaultArtifactAstBuilder builder;
  @Mock
  private ExtensionModel extensionModel;
  @Mock
  private ArtifactAst artifactAst;
  @Mock
  private UnaryOperator<String> unaryOperator;
  @Mock
  private ExtensionModelHelper extensionModelHelper;
  @Mock
  private ComponentLocationVisitor componentLocationVisitor;
  @Mock
  private ParameterModelUtils parameterModelUtils;
  @Mock
  private ComponentAst component;
  @Mock
  private ErrorTypeRepository errorTypeRepo;
  @Mock
  private ImportedResource resource;

  @Before
  public void setUp() throws Exception {
    when(extensionModel.getName()).thenReturn("mule");
    builder = new DefaultArtifactAstBuilder(
                                            NAME,
                                            ArtifactType.APPLICATION,
                                            singleton(extensionModel),
                                            Optional.of(artifactAst),
                                            unaryOperator,
                                            extensionModelHelper,
                                            componentLocationVisitor,
                                            parameterModelUtils);
  }

  @Test
  public void buildWithNoSetup() {
    final ArtifactAst result = builder.build();

    assertThat(result.getArtifactName(), is(NAME));
  }

  @Test
  public void addConstructedTopLevelComponent() {
    final ArtifactAst result = builder.addTopLevelComponent(component).build();

    assertThat(result.getArtifactName(), is(NAME));
    assertThat(result.topLevelComponents().size(), is(1));
  }

  @Test
  public void withImportedResource() {
    final ArtifactAst result = builder.withImportedResource(resource).build();

    assertThat(result.getImportedResources().size(), is(1));
    assertThat(result.getImportedResources(), hasItem(resource));
    verify(resource).updatePropertiesResolver(any());
  }

  @Test
  public void createMetadataBuilder() {
    final ComponentMetadataAstBuilder result = builder.createMetadataBuilder();

    assertThat(result.build(), is(notNullValue()));
  }
}
