/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api;

import static java.lang.System.getProperty;
import static java.util.Collections.unmodifiableSet;
import static java.util.stream.Collectors.toSet;
import static org.mule.runtime.api.util.MuleSystemProperties.SYSTEM_PROPERTY_PREFIX;

import org.mule.runtime.api.meta.model.ExtensionModel;

import java.util.HashSet;
import java.util.Set;
import java.util.function.Supplier;

/**
 * Provides different ways that the dependencies of an {@link ArtifactAst} may be calculated when calling
 * {@link ArtifactAst#dependencies()}.
 * <p>
 * The setting of this mode is done through a system property that is read whenever it is needed.
 */
public enum DependencyResolutionMode {

  /**
   * Only the extensions that are being actually used by the artifact are returned.
   */
  MINIMAL() {

    @Override
    public Set<ExtensionModel> resolveDependencies(Supplier<Set<ExtensionModel>> extensionModelsSupplier,
                                                   ArtifactAst artifactAst) {
      final Set<String> namespaceUris = artifactAst.recursiveStream()
          .map(c -> c.getIdentifier().getNamespaceUri())
          .collect(toSet());

      return unmodifiableSet(extensionModelsSupplier.get()
          .stream()
          .filter(em -> namespaceUris.contains(em.getXmlDslModel().getNamespace()))
          .collect(toSet()));
    }
  },

  /**
   * All the dependencies that the artifact is compiled with are returned.
   */
  COMPILED() {

    @Override
    public Set<ExtensionModel> resolveDependencies(Supplier<Set<ExtensionModel>> extensionModelsSupplier,
                                                   ArtifactAst artifactAst) {
      return extensionModelsSupplier.get();
    }
  },

  /**
   * Same as {@link #COMPILED}, but also includes the dependencies of the parent artifact
   */
  WITH_PARENT() {

    @Override
    public Set<ExtensionModel> resolveDependencies(Supplier<Set<ExtensionModel>> extensionModelsSupplier,
                                                   ArtifactAst artifactAst) {
      Set<ExtensionModel> extensions = new HashSet<>(extensionModelsSupplier.get());
      artifactAst.getParent()
          .map(ArtifactAst::dependencies)
          .ifPresent(extensions::addAll);

      return unmodifiableSet(extensions);
    }
  };

  /**
   * Reads a system property and obtains the proper value from that.
   *
   * @return the resolved {@link DependencyResolutionMode}
   * @throws IllegalArgumentException if the system property contains an invalid value.
   */
  public static DependencyResolutionMode getDependencyResolutionMode() {
    final String sysProp = getProperty(SYSTEM_PROPERTY_PREFIX + DependencyResolutionMode.class.getName());
    if (sysProp == null) {
      return COMPILED;
    } else {
      return DependencyResolutionMode.valueOf(sysProp);
    }
  }

  /**
   * Generates a set of {@link ExtensionModel}s according to the resolution mode type. For more details, see each implementation.
   *
   * @param extensionModelsSupplier A supplier from the declaration of the components added by the {@link ArtifactAstBuilder}.
   * @param artifactAst             The artifact AST.
   * @return The set of dependencies to be returned by {@link ArtifactAst#dependencies()}.
   */
  public abstract Set<ExtensionModel> resolveDependencies(Supplier<Set<ExtensionModel>> extensionModelsSupplier,
                                                          ArtifactAst artifactAst);
}
