/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.builder;

import static org.mule.runtime.ast.api.ArtifactType.APPLICATION;

import static java.util.function.UnaryOperator.identity;

import org.mule.api.annotation.NoImplement;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ArtifactType;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.NamespaceDefinition;
import org.mule.runtime.ast.api.model.ExtensionModelHelper;
import org.mule.runtime.ast.internal.builder.ComponentLocationVisitor;
import org.mule.runtime.ast.internal.builder.DefaultArtifactAstBuilder;
import org.mule.runtime.ast.internal.model.DefaultExtensionModelHelper;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;

import java.util.Optional;
import java.util.Set;
import java.util.function.UnaryOperator;

/**
 * Provides a way of creating {@link ArtifactAst} instances, adding and configuring its components.
 *
 * @since 1.0
 */
@NoImplement
public interface ArtifactAstBuilder {

  static ArtifactAstBuilder builder(Set<ExtensionModel> extensionsModels,
                                    Optional<ArtifactAst> parentArtifact) {
    return builder(extensionsModels, parentArtifact, identity());
  }

  static ArtifactAstBuilder builder(Set<ExtensionModel> extensionsModels,
                                    Optional<ArtifactAst> parentArtifact,
                                    ExtensionModelHelper extModelHelper) {
    return builder(extensionsModels, parentArtifact, identity(), extModelHelper);
  }

  /**
   * Instantiates a new raw builder, whose components will be declared in the provided {@code extensionModels}.
   *
   * @param extensionsModels       contain the declaration of the components to be added to the target {@link ArtifactAst}.
   * @param parentArtifact         the {@link ArtifactAst} that is the parent of the target artifact.
   * @param basePropertiesResolver the resolver for properties that may be found during the basic building of the AST (for
   *                               instance, the names of components used to generate locations).
   * @return the newly created builder
   */
  static ArtifactAstBuilder builder(Set<ExtensionModel> extensionsModels,
                                    Optional<ArtifactAst> parentArtifact,
                                    UnaryOperator<String> basePropertiesResolver) {
    return builder(extensionsModels, parentArtifact,
                   basePropertiesResolver,
                   new DefaultExtensionModelHelper(extensionsModels));
  }

  /**
   * Instantiates a new raw builder, whose components will be declared in the provided {@code extensionModels}.
   *
   * @param extensionsModels       contain the declaration of the components to be added to the target {@link ArtifactAst}.
   * @param parentArtifact         the {@link ArtifactAst} that is the parent of the target artifact.
   * @param basePropertiesResolver the resolver for properties that may be found during the basic building of the AST (for
   *                               instance, the names of components used to generate locations).
   * @return the newly created builder
   */
  static ArtifactAstBuilder builder(Set<ExtensionModel> extensionsModels,
                                    Optional<ArtifactAst> parentArtifact,
                                    UnaryOperator<String> basePropertiesResolver,
                                    ExtensionModelHelper extModelHelper) {
    return builder("<app>", APPLICATION, extensionsModels, parentArtifact,
                   basePropertiesResolver,
                   extModelHelper);
  }

  static ArtifactAstBuilder builder(String artifactName,
                                    ArtifactType artifactType,
                                    Set<ExtensionModel> extensionsModels,
                                    Optional<ArtifactAst> parentArtifact,
                                    UnaryOperator<String> basePropertiesResolver) {
    return new DefaultArtifactAstBuilder(artifactName, artifactType, extensionsModels,
                                         parentArtifact,
                                         basePropertiesResolver,
                                         new DefaultExtensionModelHelper(extensionsModels),
                                         new ComponentLocationVisitor(),
                                         new ParameterModelUtils());
  }

  static ArtifactAstBuilder builder(String artifactName,
                                    ArtifactType artifactType,
                                    Set<ExtensionModel> extensionsModels,
                                    Optional<ArtifactAst> parentArtifact,
                                    UnaryOperator<String> basePropertiesResolver,
                                    ExtensionModelHelper extModelHelper) {
    return new DefaultArtifactAstBuilder(artifactName, artifactType, extensionsModels,
                                         parentArtifact,
                                         basePropertiesResolver,
                                         extModelHelper,
                                         new ComponentLocationVisitor(),
                                         new ParameterModelUtils());
  }

  /**
   * Adds a top-level component to the artifact being built with this builder.
   *
   * @return the builder for the newly added top-level component
   */
  ComponentAstBuilder addTopLevelComponent();

  /**
   * @return A properly configured {@link ComponentMetadataAstBuilder} to use with this instance.
   */
  ComponentMetadataAstBuilder createMetadataBuilder();

  /**
   * Builds the target artifact and its inner components.
   *
   * @return the target artifact.
   */
  ArtifactAst build();

  /**
   * Adds a {@link NamespaceDefinition} to the Artifact being built.
   *
   * @param namespaceDefinition the namespace definition to be added.
   */
  void withNamespaceDefinition(NamespaceDefinition namespaceDefinition);

  /**
   * Adds a resource to be imported by the resulting {@link ArtifactAst}.
   *
   * @param resource The resource to be imported by the resulting {@link ArtifactAst}.
   * @return the same ast builder.
   */
  ArtifactAstBuilder withImportedResource(ImportedResource resource);

}
