/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.api.util;

import static org.mule.runtime.api.functional.Either.left;
import static org.mule.runtime.api.functional.Either.right;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.ast.api.util.AstTraversalDirection.BOTTOM_UP;
import static org.mule.runtime.ast.api.util.AstTraversalDirection.TOP_DOWN;
import static org.mule.runtime.ast.api.util.MuleAstUtils.createComponentParameterizationFromComponentAst;
import static org.mule.runtime.ast.api.util.MuleAstUtils.parameterOfType;
import static org.mule.runtime.ast.api.util.MuleAstUtils.parametersOfType;
import static org.mule.runtime.ast.api.util.MuleAstUtils.resolveOrphanComponents;
import static org.mule.runtime.ast.internal.dsl.DslConstants.KEY_ATTRIBUTE_NAME;
import static org.mule.runtime.ast.internal.dsl.DslConstants.VALUE_ATTRIBUTE_NAME;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.AstTraversal.AST_TRAVERSAL;
import static org.mule.runtime.ast.test.AllureConstants.DietToolingFeature.DIET_TOOLING;
import static org.mule.runtime.ast.test.AllureConstants.DietToolingFeature.Stories.METADATA_TYPES_RESOLUTION;
import static org.mule.runtime.ast.test.AllureConstants.DietToolingFeature.Stories.SAMPLE_DATA;
import static org.mule.runtime.ast.test.AllureConstants.DietToolingFeature.Stories.TEST_CONNECTIVITY;
import static org.mule.runtime.ast.test.AllureConstants.DietToolingFeature.Stories.VALUE_PROVIDER;
import static org.mule.runtime.ast.test.AllureConstants.OperationExecutionFeature.OPERATION_EXECUTION_FEATURE;
import static org.mule.runtime.ast.test.AllureConstants.OperationExecutionFeature.Stories.EXECUTE_OPERATION;
import static org.mule.runtime.ast.test.api.util.TestUtils.createMockParameterGroup;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;

import static org.hamcrest.CoreMatchers.instanceOf;
import static org.hamcrest.CoreMatchers.sameInstance;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.aMapWithSize;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.collection.IsEmptyCollection.empty;
import static org.hamcrest.collection.IsMapContaining.hasKey;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsIterableContaining.hasItem;
import static org.hamcrest.core.IsIterableContaining.hasItems;
import static org.hamcrest.core.IsNot.not;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.impl.DefaultArrayType;
import org.mule.metadata.api.model.impl.DefaultStringType;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.stereotype.ImmutableStereotypeModel;
import org.mule.runtime.api.parameterization.ComponentParameterization;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.util.AstTraversalDirection;
import org.mule.runtime.ast.api.util.BaseArtifactAst;
import org.mule.runtime.ast.api.util.BaseComponentAst;
import org.mule.runtime.ast.api.util.MuleAstUtils;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Features;
import io.qameta.allure.Issue;
import io.qameta.allure.Stories;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(AST_TRAVERSAL)
public class MuleAstUtilsTestCase {

  @Test
  public void orphanComponentsTopLevelWithChildren() {
    final ArtifactAst artifact = mock(BaseArtifactAst.class);

    final ComponentAst processor = mockBaseComponentAst();
    when(processor.directChildrenStream()).thenAnswer(inv -> Stream.empty());
    when(processor.recursiveStream()).thenAnswer(inv -> Stream.of(processor));

    final ComponentAst topLevelComponent = mockBaseComponentAst();
    when(topLevelComponent.directChildrenStream()).thenAnswer(inv -> Stream.of(processor));
    when(topLevelComponent.recursiveStream()).thenAnswer(inv -> Stream.of(topLevelComponent, processor));

    when(artifact.topLevelComponentsStream()).thenAnswer(inv -> Stream.of(topLevelComponent));
    when(artifact.recursiveStream()).thenAnswer(inv -> Stream.of(topLevelComponent, processor));
    when(artifact.filteredComponents(any())).thenCallRealMethod();

    final Set<ComponentAst> orphanComponents = resolveOrphanComponents(artifact);

    assertThat(orphanComponents, not(hasItem(topLevelComponent)));
    assertThat(orphanComponents, not(hasItem(processor)));
  }

  private ComponentAst mockBaseComponentAst() {
    ComponentAst theMock = mock(BaseComponentAst.class);
    when(theMock.getModel(ParameterizedModel.class)).thenReturn(of(mock(ParameterizedModel.class)));
    return theMock;
  }

  @Test
  public void orphanComponentsSingleProcessor() {
    final ArtifactAst artifact = mock(BaseArtifactAst.class);

    final ComponentAst orphanProcessor = mockBaseComponentAst();
    when(orphanProcessor.directChildrenStream()).thenAnswer(inv -> Stream.empty());
    when(orphanProcessor.recursiveStream()).thenAnswer(inv -> Stream.of(orphanProcessor));

    when(artifact.topLevelComponentsStream()).thenAnswer(inv -> Stream.empty());
    when(artifact.recursiveStream()).thenAnswer(inv -> Stream.of(orphanProcessor));
    when(artifact.filteredComponents(any())).thenCallRealMethod();

    final Set<ComponentAst> orphanComponents = resolveOrphanComponents(artifact);

    assertThat(orphanComponents, hasItem(orphanProcessor));
  }

  @Test
  public void orphanComponentsNestedProcessor() {
    final ArtifactAst artifact = mock(BaseArtifactAst.class);

    final ComponentAst nestedProcessor = mockBaseComponentAst();
    when(nestedProcessor.directChildrenStream()).thenAnswer(inv -> Stream.empty());
    when(nestedProcessor.recursiveStream()).thenAnswer(inv -> Stream.of(nestedProcessor));

    final ComponentAst orphanRouter = mockBaseComponentAst();
    when(orphanRouter.directChildrenStream()).thenAnswer(inv -> Stream.of(nestedProcessor));
    when(orphanRouter.recursiveStream()).thenAnswer(inv -> Stream.of(orphanRouter, nestedProcessor));

    when(artifact.topLevelComponentsStream()).thenAnswer(inv -> Stream.empty());
    when(artifact.recursiveStream()).thenAnswer(inv -> Stream.of(nestedProcessor, orphanRouter));
    when(artifact.filteredComponents(any())).thenCallRealMethod();

    final Set<ComponentAst> orphanComponents = resolveOrphanComponents(artifact);

    assertThat(orphanComponents, hasItem(orphanRouter));
    assertThat(orphanComponents, not(hasItem(nestedProcessor)));
  }

  @Test
  public void parameterOfTypeSameStereotype() {
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_OWN", "MINE", null);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    ParameterGroupModel groupModel = createMockParameterGroup(DEFAULT_GROUP_NAME, parameterModel);

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);

    final ComponentAst comp = mockBaseComponentAst();
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter(anyString(), eq("myOwnParam"))).thenReturn(parameter);

    assertThat(parameterOfType(comp, myOwnStereotype).get(), sameInstance(parameter));
  }

  @Test
  public void parameterOfTypeParentStereotype() {
    final ImmutableStereotypeModel myOwnParentStereotype = new ImmutableStereotypeModel("MY_OWN", "MINE", null);
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_PRECIOUS", "MINE", myOwnParentStereotype);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    ParameterGroupModel groupModel = createMockParameterGroup(DEFAULT_GROUP_NAME, parameterModel);

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);

    final ComponentAst comp = mockBaseComponentAst();
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter(anyString(), eq("myOwnParam"))).thenReturn(parameter);

    assertThat(parameterOfType(comp, myOwnParentStereotype).get(), sameInstance(parameter));
  }

  @Test
  public void parameterOfTypeNotMatchingStereotype() {
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_PRECIOUS", "MINE", null);
    final ImmutableStereotypeModel myOtherStereotype = new ImmutableStereotypeModel("A_SWORD", "MINE", null);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getAllParameterModels()).thenReturn(singletonList(parameterModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);

    final ComponentAst comp = mockBaseComponentAst();
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter(anyString(), eq("myOwnParam"))).thenReturn(parameter);

    assertThat(parameterOfType(comp, myOtherStereotype).isPresent(), is(false));
  }

  @Test
  public void parametersOfTypeSameStereotype() {
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_OWN", "MINE", null);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    ParameterGroupModel groupModel = createMockParameterGroup(DEFAULT_GROUP_NAME, parameterModel);

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);

    final ComponentAst comp = mockBaseComponentAst();
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter(anyString(), eq("myOwnParam"))).thenReturn(parameter);

    assertThat(parametersOfType(comp, myOwnStereotype), hasItems(sameInstance(parameter)));
  }

  @Test
  public void parametersOfTypeSameStereotypeMultiple() {
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_OWN", "MINE", null);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    final ParameterModel parameter2Model = mock(ParameterModel.class);
    when(parameter2Model.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameter2Model.getName()).thenReturn("myOwnParam2");

    ParameterGroupModel groupModel = createMockParameterGroup(DEFAULT_GROUP_NAME, parameterModel, parameter2Model);

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);
    final ComponentParameterAst parameter2 = mock(ComponentParameterAst.class);

    final ComponentAst comp = mockBaseComponentAst();
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter(anyString(), eq("myOwnParam"))).thenReturn(parameter);
    when(comp.getParameter(anyString(), eq("myOwnParam2"))).thenReturn(parameter2);

    assertThat(parametersOfType(comp, myOwnStereotype), hasItems(sameInstance(parameter), sameInstance(parameter2)));
  }

  @Test
  public void parametersOfTypeParentStereotype() {
    final ImmutableStereotypeModel myOwnParentStereotype = new ImmutableStereotypeModel("MY_OWN", "MINE", null);
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_PRECIOUS", "MINE", myOwnParentStereotype);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    final ParameterGroupModel groupModel = createMockParameterGroup(DEFAULT_GROUP_NAME, parameterModel);

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);

    final ComponentAst comp = mockBaseComponentAst();
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter(anyString(), eq("myOwnParam"))).thenReturn(parameter);

    assertThat(parametersOfType(comp, myOwnParentStereotype), hasItems(sameInstance(parameter)));
  }

  @Test
  public void parametersOfTypeNotMatchingStereotype() {
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_PRECIOUS", "MINE", null);
    final ImmutableStereotypeModel myOtherStereotype = new ImmutableStereotypeModel("A_SWORD", "MINE", null);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getAllParameterModels()).thenReturn(singletonList(parameterModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);

    final ComponentAst comp = mockBaseComponentAst();
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter(anyString(), eq("myOwnParam"))).thenReturn(parameter);

    assertThat(parametersOfType(comp, myOtherStereotype), hasSize(0));
  }

  private ComponentAst createComponentAst(ComponentAst... innerComponents) {
    ComponentAst component = mockBaseComponentAst();
    when(component.directChildrenStream()).thenAnswer(inv -> Stream.of(innerComponents));
    when(component.recursiveStream()).thenCallRealMethod();
    when(component.recursiveStream(any(AstTraversalDirection.class))).thenAnswer(inv -> {
      AstTraversalDirection direction = inv.getArgument(0);
      return StreamSupport.stream(component.recursiveSpliterator(direction), false);
    });
    when(component.recursiveSpliterator()).thenCallRealMethod();
    when(component.recursiveSpliterator(any(AstTraversalDirection.class))).thenAnswer(inv -> {
      AstTraversalDirection direction = inv.getArgument(0);
      return direction.recursiveSpliterator(component);
    });
    return component;
  }

  @Test
  public void recursiveStreamWithHierarchy() {
    ComponentAst child1110 = createComponentAst();

    ComponentAst child1111 = createComponentAst();

    ComponentAst child1112 = createComponentAst();

    ComponentAst child111 = createComponentAst(child1110, child1111, child1112);

    ComponentAst child110 = createComponentAst();

    ComponentAst child10 = createComponentAst();

    ComponentAst child11 = createComponentAst(child110, child111);

    ComponentAst child1 = createComponentAst(child10, child11);

    ComponentAst child22 = createComponentAst();

    ComponentAst child2 = createComponentAst(child22);

    ComponentAst child3 = createComponentAst();

    ArtifactAst artifact = mock(BaseArtifactAst.class);

    when(artifact.recursiveStream())
        .thenAnswer(inv -> Stream.concat(child1.recursiveStream(), child2.recursiveStream()));
    when(artifact.topLevelComponentsStream()).thenAnswer(inv -> Stream.of(child1, child2, child3));

    List<Pair<ComponentAst, List<ComponentAst>>> withHierarchy =
        MuleAstUtils.recursiveStreamWithHierarchy(artifact).collect(toList());

    assertThat(withHierarchy, hasSize(11));
    assertThat(withHierarchy.get(0).getFirst(), sameInstance(child1));
    assertThat(withHierarchy.get(0).getSecond(), empty());
    assertThat(withHierarchy.get(1).getFirst(), sameInstance(child10));
    assertThat(withHierarchy.get(1).getSecond(), hasItem(child1));
    assertThat(withHierarchy.get(2).getFirst(), sameInstance(child11));
    assertThat(withHierarchy.get(2).getSecond(), hasItem(child1));
    assertThat(withHierarchy.get(3).getFirst(), sameInstance(child110));
    assertThat(withHierarchy.get(3).getSecond(), hasItem(child11));
    assertThat(withHierarchy.get(4).getFirst(), sameInstance(child111));
    assertThat(withHierarchy.get(4).getSecond(), hasItem(child11));
    assertThat(withHierarchy.get(5).getFirst(), sameInstance(child1110));
    assertThat(withHierarchy.get(5).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(6).getFirst(), sameInstance(child1111));
    assertThat(withHierarchy.get(6).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(7).getFirst(), sameInstance(child1112));
    assertThat(withHierarchy.get(7).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(8).getFirst(), sameInstance(child2));
    assertThat(withHierarchy.get(8).getSecond(), empty());
    assertThat(withHierarchy.get(9).getFirst(), sameInstance(child22));
    assertThat(withHierarchy.get(9).getSecond(), hasItem(child2));
    assertThat(withHierarchy.get(10).getFirst(), sameInstance(child3));
    assertThat(withHierarchy.get(10).getSecond(), empty());

    withHierarchy = MuleAstUtils.recursiveStreamWithHierarchy(artifact, TOP_DOWN).collect(toList());

    assertThat(withHierarchy, hasSize(11));
    assertThat(withHierarchy.get(0).getFirst(), sameInstance(child1));
    assertThat(withHierarchy.get(0).getSecond(), empty());
    assertThat(withHierarchy.get(1).getFirst(), sameInstance(child10));
    assertThat(withHierarchy.get(1).getSecond(), hasItem(child1));
    assertThat(withHierarchy.get(2).getFirst(), sameInstance(child11));
    assertThat(withHierarchy.get(2).getSecond(), hasItem(child1));
    assertThat(withHierarchy.get(3).getFirst(), sameInstance(child110));
    assertThat(withHierarchy.get(3).getSecond(), hasItem(child11));
    assertThat(withHierarchy.get(4).getFirst(), sameInstance(child111));
    assertThat(withHierarchy.get(4).getSecond(), hasItem(child11));
    assertThat(withHierarchy.get(5).getFirst(), sameInstance(child1110));
    assertThat(withHierarchy.get(5).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(6).getFirst(), sameInstance(child1111));
    assertThat(withHierarchy.get(6).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(7).getFirst(), sameInstance(child1112));
    assertThat(withHierarchy.get(7).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(8).getFirst(), sameInstance(child2));
    assertThat(withHierarchy.get(8).getSecond(), empty());
    assertThat(withHierarchy.get(9).getFirst(), sameInstance(child22));
    assertThat(withHierarchy.get(9).getSecond(), hasItem(child2));
    assertThat(withHierarchy.get(10).getFirst(), sameInstance(child3));
    assertThat(withHierarchy.get(10).getSecond(), empty());

    withHierarchy = MuleAstUtils.recursiveStreamWithHierarchy(artifact, BOTTOM_UP).collect(toList());
    assertThat(withHierarchy, hasSize(11));
    assertThat(withHierarchy.get(0).getFirst(), sameInstance(child10));
    assertThat(withHierarchy.get(0).getSecond(), hasItem(child1));
    assertThat(withHierarchy.get(1).getFirst(), sameInstance(child110));
    assertThat(withHierarchy.get(1).getSecond(), hasItem(child11));
    assertThat(withHierarchy.get(2).getFirst(), sameInstance(child1110));
    assertThat(withHierarchy.get(2).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(3).getFirst(), sameInstance(child1111));
    assertThat(withHierarchy.get(3).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(4).getFirst(), sameInstance(child1112));
    assertThat(withHierarchy.get(4).getSecond(), hasItem(child111));
    assertThat(withHierarchy.get(5).getFirst(), sameInstance(child111));
    assertThat(withHierarchy.get(5).getSecond(), hasItem(child11));
    assertThat(withHierarchy.get(6).getFirst(), sameInstance(child11));
    assertThat(withHierarchy.get(6).getSecond(), hasItem(child1));
    assertThat(withHierarchy.get(7).getFirst(), sameInstance(child1));
    assertThat(withHierarchy.get(7).getSecond(), empty());
    assertThat(withHierarchy.get(8).getFirst(), sameInstance(child22));
    assertThat(withHierarchy.get(8).getSecond(), hasItem(child2));
    assertThat(withHierarchy.get(9).getFirst(), sameInstance(child2));
    assertThat(withHierarchy.get(9).getSecond(), empty());
    assertThat(withHierarchy.get(10).getFirst(), sameInstance(child3));
    assertThat(withHierarchy.get(10).getSecond(), empty());
  }

  @Test
  public void recursiveStreamWithHierarchyWithOrphans() {
    final ArtifactAst artifact = mock(BaseArtifactAst.class);

    final ComponentAst processor = createComponentAst();

    final ComponentAst topLevelComponent = createComponentAst(processor);

    final ComponentAst orphanProcessor = createComponentAst();

    when(artifact.topLevelComponentsStream()).thenAnswer(inv -> Stream.of(topLevelComponent));
    when(artifact.recursiveStream()).thenAnswer(inv -> Stream.of(topLevelComponent, processor, orphanProcessor));
    when(artifact.filteredComponents(any())).thenCallRealMethod();

    List<Pair<ComponentAst, List<ComponentAst>>> withHierarchy =
        MuleAstUtils.recursiveStreamWithHierarchy(artifact).collect(toList());

    assertThat(withHierarchy, hasSize(3));
    assertThat(withHierarchy.get(0).getFirst(), sameInstance(topLevelComponent));
    assertThat(withHierarchy.get(0).getSecond(), empty());
    assertThat(withHierarchy.get(1).getFirst(), sameInstance(processor));
    assertThat(withHierarchy.get(1).getSecond(), hasItem(topLevelComponent));
    assertThat(withHierarchy.get(2).getFirst(), sameInstance(orphanProcessor));
    assertThat(withHierarchy.get(2).getSecond(), not(empty()));

    withHierarchy = MuleAstUtils.recursiveStreamWithHierarchy(artifact, TOP_DOWN).collect(toList());

    assertThat(withHierarchy, hasSize(3));
    assertThat(withHierarchy.get(0).getFirst(), sameInstance(topLevelComponent));
    assertThat(withHierarchy.get(0).getSecond(), empty());
    assertThat(withHierarchy.get(1).getFirst(), sameInstance(processor));
    assertThat(withHierarchy.get(1).getSecond(), hasItem(topLevelComponent));
    assertThat(withHierarchy.get(2).getFirst(), sameInstance(orphanProcessor));
    assertThat(withHierarchy.get(2).getSecond(), not(empty()));

    withHierarchy = MuleAstUtils.recursiveStreamWithHierarchy(artifact, BOTTOM_UP).collect(toList());

    assertThat(withHierarchy, hasSize(3));
    assertThat(withHierarchy.get(0).getFirst(), sameInstance(processor));
    assertThat(withHierarchy.get(0).getSecond(), hasItem(topLevelComponent));
    assertThat(withHierarchy.get(1).getFirst(), sameInstance(topLevelComponent));
    assertThat(withHierarchy.get(1).getSecond(), empty());
    assertThat(withHierarchy.get(2).getFirst(), sameInstance(orphanProcessor));
    assertThat(withHierarchy.get(2).getSecond(), not(empty()));
  }

  @Features({@Feature(OPERATION_EXECUTION_FEATURE), @Feature(DIET_TOOLING)})
  @Stories({@Story(EXECUTE_OPERATION), @Story(TEST_CONNECTIVITY), @Story(METADATA_TYPES_RESOLUTION), @Story(VALUE_PROVIDER),
      @Story(SAMPLE_DATA)})
  @Test
  public void createAValidComponentParametrizationFromAComponentAst() {
    String fixedValueGroupName = "FIXED_VALUE";
    String expressionGroupName = "EXPRESSION";
    ParameterGroupModel fixedValueParameterGroupModel = mock(ParameterGroupModel.class);
    ParameterGroupModel expressionParameterGroupModel = mock(ParameterGroupModel.class);
    when(fixedValueParameterGroupModel.getName()).thenReturn(fixedValueGroupName);
    when(expressionParameterGroupModel.getName()).thenReturn(expressionGroupName);

    // Create nested component AST
    ComponentIdentifier nestedComponentIdentifier = mock(ComponentIdentifier.class);
    ParameterizedModel nestedComponentParameterizedModel = mock(ParameterizedModel.class);
    ComponentParameterAst nestedParameterAst =
        createFixedValueComponentParameterAst(fixedValueParameterGroupModel, "nestedParam1", "nestedParamValue");
    ComponentParameterAst nestedExpressionParameterAst =
        createExpressionMockedComponentParameterAst(expressionParameterGroupModel, "nestedParam2",
                                                    "nested ++ Expression ++ Value");
    ComponentAst nestedComponentAst = createMockedComponentAst(nestedComponentIdentifier, nestedComponentParameterizedModel,
                                                               asList(nestedParameterAst, nestedExpressionParameterAst));

    // Create father component AST
    ComponentIdentifier componentIdentifier = mock(ComponentIdentifier.class);
    ParameterizedModel componentParameterizedModel = mock(ParameterizedModel.class);

    ComponentParameterAst parameterAst = createFixedValueComponentParameterAst(fixedValueParameterGroupModel, "param1", "value");
    ComponentParameterAst anotherParameterAst =
        createFixedValueComponentParameterAst(fixedValueParameterGroupModel, "param2", "anotherValue");
    ComponentParameterAst expressionParameterAst =
        createExpressionMockedComponentParameterAst(expressionParameterGroupModel, "param3",
                                                    "expression ++ Value");
    ComponentParameterAst nestedComponentAstParameterAst =
        createFixedValueComponentParameterAst(fixedValueParameterGroupModel, "nestedComponentAst", nestedComponentAst);

    ComponentAst componentAst = createMockedComponentAst(componentIdentifier, componentParameterizedModel,
                                                         asList(parameterAst, anotherParameterAst, expressionParameterAst,
                                                                nestedComponentAstParameterAst));

    ComponentParameterization<ParameterizedModel> componentParameterization =
        createComponentParameterizationFromComponentAst(componentAst);

    assertThat(componentParameterization.getModel(), is(componentParameterizedModel));
    assertThat(componentParameterization.getParameters(), aMapWithSize(4));
    assertThat(componentParameterization.getParameter(fixedValueGroupName, "param1"), is("value"));
    assertThat(componentParameterization.getParameter(fixedValueGroupName, "param2"), is("anotherValue"));
    assertThat(componentParameterization.getParameter(expressionGroupName, "param3"), is("#[expression ++ Value]"));
    assertThat(componentParameterization.getParameter(fixedValueGroupName, "nestedComponentAst"),
               instanceOf(ComponentParameterization.class));
    assertThat(componentParameterization.getComponentIdentifier().isPresent(), is(true));
    assertThat(componentParameterization.getComponentIdentifier().get(), is(componentIdentifier));

    ComponentParameterization<ParameterizedModel> nestedComponentParameterization =
        (ComponentParameterization<ParameterizedModel>) componentParameterization.getParameter(fixedValueGroupName,
                                                                                               "nestedComponentAst");
    assertThat(nestedComponentParameterization.getModel(), is(nestedComponentParameterizedModel));
    assertThat(nestedComponentParameterization.getParameters(), aMapWithSize(2));
    assertThat(nestedComponentParameterization.getParameter(fixedValueGroupName, "nestedParam1"), is("nestedParamValue"));
    assertThat(nestedComponentParameterization.getParameter(expressionGroupName, "nestedParam2"),
               is("#[nested ++ Expression ++ Value]"));
    assertThat(nestedComponentParameterization.getComponentIdentifier().isPresent(), is(true));
    assertThat(nestedComponentParameterization.getComponentIdentifier().get(), is(nestedComponentIdentifier));
  }

  @Features({@Feature(OPERATION_EXECUTION_FEATURE), @Feature(DIET_TOOLING)})
  @Stories({
      @Story(EXECUTE_OPERATION),
      @Story(TEST_CONNECTIVITY),
      @Story(METADATA_TYPES_RESOLUTION),
      @Story(VALUE_PROVIDER),
      @Story(SAMPLE_DATA)
  })
  @Test
  @Issue("W-18030082")
  public void createAValidComponentParametrizationFromAComponentAstWithSimpleArrayParameter() {
    String arrayValueGroupName = "ARRAY_VALUE";
    ParameterGroupModel arrayValueGroupModel = mock(ParameterGroupModel.class);
    when(arrayValueGroupModel.getName()).thenReturn(arrayValueGroupName);
    ComponentParameterAst arrayParameterAst = createSimpleArrayComponentParameterAst(arrayValueGroupModel, "arrayParam");

    ComponentParameterization<ParameterizedModel> componentParameterization =
        createAValidComponentParametrizationFromAComponentAstWithArrayParameter(arrayValueGroupName, arrayParameterAst,
                                                                                "v1", "v2", false);

    List<String> arrayParam =
        (List<String>) componentParameterization.getParameter(arrayValueGroupName, "arrayParam");

    assertThat(arrayParam, hasSize(2));
    assertThat(arrayParam.get(0), is("v1"));
    assertThat(arrayParam.get(1), is("v2"));
  }

  @Features({@Feature(OPERATION_EXECUTION_FEATURE), @Feature(DIET_TOOLING)})
  @Stories({
      @Story(EXECUTE_OPERATION),
      @Story(TEST_CONNECTIVITY),
      @Story(METADATA_TYPES_RESOLUTION),
      @Story(VALUE_PROVIDER),
      @Story(SAMPLE_DATA)
  })
  @Test
  @Issue("W-18030082")
  public void createAValidComponentParametrizationFromAComponentAstWithSimpleExpressionArrayParameter() {
    String arrayValueGroupName = "ARRAY_VALUE";
    ParameterGroupModel arrayValueGroupModel = mock(ParameterGroupModel.class);
    when(arrayValueGroupModel.getName()).thenReturn(arrayValueGroupName);
    ComponentParameterAst arrayParameterAst = createSimpleArrayComponentParameterAst(arrayValueGroupModel, "arrayParam");

    ComponentParameterization<ParameterizedModel> componentParameterization =
        createAValidComponentParametrizationFromAComponentAstWithArrayParameter(arrayValueGroupName, arrayParameterAst,
                                                                                "v1", "v2", true);

    List<String> arrayParam =
        (List<String>) componentParameterization.getParameter(arrayValueGroupName, "arrayParam");

    assertThat(arrayParam, hasSize(2));
    assertThat(arrayParam.get(0), is("#[v1]"));
    assertThat(arrayParam.get(1), is("#[v2]"));
  }

  @Features({@Feature(OPERATION_EXECUTION_FEATURE), @Feature(DIET_TOOLING)})
  @Stories({
      @Story(EXECUTE_OPERATION),
      @Story(TEST_CONNECTIVITY),
      @Story(METADATA_TYPES_RESOLUTION),
      @Story(VALUE_PROVIDER),
      @Story(SAMPLE_DATA)
  })
  @Test
  public void createAValidComponentParametrizationFromAComponentAstWithComplexArrayParameter() {
    String arrayValueGroupName = "ARRAY_VALUE";
    ParameterGroupModel arrayValueGroupModel = mock(ParameterGroupModel.class);
    when(arrayValueGroupModel.getName()).thenReturn(arrayValueGroupName);
    ComponentParameterAst arrayParameterAst = createComplexArrayComponentParameterAst(arrayValueGroupModel, "arrayParam");

    ComponentParameterization<ParameterizedModel> componentParameterization =
        createAValidComponentParametrizationFromAComponentAstWithArrayParameter(arrayValueGroupName, arrayParameterAst,
                                                                                "v1", "v2", false);

    List<ComponentParameterization> arrayParam =
        (List<ComponentParameterization>) componentParameterization.getParameter(arrayValueGroupName, "arrayParam");

    assertThat(arrayParam, hasSize(2));
    assertThat(arrayParam.get(0).getParameter(DEFAULT_GROUP_NAME, VALUE_ATTRIBUTE_NAME),
               is("v1"));
    assertThat(arrayParam.get(1).getParameter(DEFAULT_GROUP_NAME, VALUE_ATTRIBUTE_NAME),
               is("v2"));
  }

  private ComponentParameterization<ParameterizedModel> createAValidComponentParametrizationFromAComponentAstWithArrayParameter(String arrayValueGroupName,
                                                                                                                                ComponentParameterAst arrayParameterAst,
                                                                                                                                String value1,
                                                                                                                                String value2,
                                                                                                                                boolean expressions) {
    ParameterGroupModel defaultGroupModel = mock(ParameterGroupModel.class);
    when(defaultGroupModel.getName()).thenReturn(DEFAULT_GROUP_NAME);
    ComponentIdentifier arrayValueComponentIdentifier = mock(ComponentIdentifier.class);
    ParameterizedModel arrayValueParameterizedModel = mock(ParameterizedModel.class);
    // 1st value
    ComponentAst componentAst1 = createMockedComponentAst(arrayValueComponentIdentifier, arrayValueParameterizedModel,
                                                          asList(expressions
                                                              ? createExpressionMockedComponentParameterAst(defaultGroupModel,
                                                                                                            VALUE_ATTRIBUTE_NAME,
                                                                                                            value1)
                                                              : createFixedValueComponentParameterAst(defaultGroupModel,
                                                                                                      VALUE_ATTRIBUTE_NAME,
                                                                                                      value1)));
    // 2nd value
    ComponentAst componentAst2 = createMockedComponentAst(arrayValueComponentIdentifier, arrayValueParameterizedModel,
                                                          asList(expressions
                                                              ? createExpressionMockedComponentParameterAst(defaultGroupModel,
                                                                                                            VALUE_ATTRIBUTE_NAME,
                                                                                                            value2)
                                                              : createFixedValueComponentParameterAst(defaultGroupModel,
                                                                                                      VALUE_ATTRIBUTE_NAME,
                                                                                                      value2)));

    // Create component AST
    ComponentIdentifier componentIdentifier = mock(ComponentIdentifier.class);
    ParameterizedModel componentParameterizedModel = mock(ParameterizedModel.class);
    when(arrayParameterAst.getValue()).thenReturn(right(asList(componentAst1, componentAst2)));
    ComponentAst componentAst = createMockedComponentAst(componentIdentifier, componentParameterizedModel,
                                                         asList(arrayParameterAst));

    ComponentParameterization<ParameterizedModel> componentParameterization =
        createComponentParameterizationFromComponentAst(componentAst);

    assertThat(componentParameterization.getModel(), is(componentParameterizedModel));
    assertThat(componentParameterization.getParameters(), aMapWithSize(1));
    assertThat(componentParameterization.getParameter(arrayValueGroupName, "arrayParam"), instanceOf(List.class));
    assertThat(componentParameterization.getComponentIdentifier().isPresent(), is(true));
    assertThat(componentParameterization.getComponentIdentifier().get(), is(componentIdentifier));
    return componentParameterization;
  }

  @Features({@Feature(OPERATION_EXECUTION_FEATURE), @Feature(DIET_TOOLING)})
  @Stories({
      @Story(EXECUTE_OPERATION),
      @Story(TEST_CONNECTIVITY),
      @Story(METADATA_TYPES_RESOLUTION),
      @Story(VALUE_PROVIDER),
      @Story(SAMPLE_DATA)
  })
  @Test
  public void createAValidComponentParametrizationFromAComponentAstWithSimpleMapParameter() {
    String mapValueGroupName = "MAP_VALUE";
    ParameterGroupModel mapValueGroupModel = mock(ParameterGroupModel.class);
    when(mapValueGroupModel.getName()).thenReturn(mapValueGroupName);
    ComponentParameterAst mapParameterAst = createSimpleMapComponentParameterAst(mapValueGroupModel, "mapParam");

    ComponentParameterization<ParameterizedModel> componentParameterization =
        createAValidComponentParametrizationFromAComponentAstWithMapParameter(mapValueGroupName, mapParameterAst, "v1", "v2");

    Map<String, String> mapParam = (Map<String, String>) componentParameterization.getParameter(mapValueGroupName, "mapParam");

    assertThat(mapParam, hasKey("e1"));
    assertThat(mapParam, hasKey("e2"));

    assertThat(mapParam.get("e1"), is("v1"));
    assertThat(mapParam.get("e2"), is("v2"));
  }

  @Features({@Feature(OPERATION_EXECUTION_FEATURE), @Feature(DIET_TOOLING)})
  @Stories({
      @Story(EXECUTE_OPERATION),
      @Story(TEST_CONNECTIVITY),
      @Story(METADATA_TYPES_RESOLUTION),
      @Story(VALUE_PROVIDER),
      @Story(SAMPLE_DATA)
  })
  @Test
  @Issue("W-18030082")
  public void createAValidComponentParametrizationFromAComponentAstWithComplexMapParameter() {
    String mapValueGroupName = "MAP_VALUE";
    ParameterGroupModel mapValueGroupModel = mock(ParameterGroupModel.class);
    when(mapValueGroupModel.getName()).thenReturn(mapValueGroupName);
    ComponentParameterAst mapParameterAst = createComplexMapComponentParameterAst(mapValueGroupModel, "mapParam");

    ComponentParameterization<ParameterizedModel> componentParameterization =
        createAValidComponentParametrizationFromAComponentAstWithMapParameter(mapValueGroupName, mapParameterAst, "v1", "v2");

    Map<String, ComponentParameterization> mapParam =
        (Map<String, ComponentParameterization>) componentParameterization.getParameter(mapValueGroupName, "mapParam");

    assertThat(mapParam, hasKey("e1"));
    assertThat(mapParam, hasKey("e2"));

    assertThat(mapParam.get("e1").getParameter(DEFAULT_GROUP_NAME, VALUE_ATTRIBUTE_NAME),
               is("v1"));
    assertThat(mapParam.get("e2").getParameter(DEFAULT_GROUP_NAME, VALUE_ATTRIBUTE_NAME),
               is("v2"));
  }

  private ComponentParameterization<ParameterizedModel> createAValidComponentParametrizationFromAComponentAstWithMapParameter(String mapValueGroupName,
                                                                                                                              ComponentParameterAst mapParameterAst,
                                                                                                                              String value1,
                                                                                                                              String value2) {
    ParameterGroupModel defaultGroupModel = mock(ParameterGroupModel.class);
    when(defaultGroupModel.getName()).thenReturn(DEFAULT_GROUP_NAME);
    ComponentIdentifier mapEntryComponentIdentifier = mock(ComponentIdentifier.class);
    ParameterizedModel mapEntryParameterizedModel = mock(ParameterizedModel.class);
    // 1st value
    ComponentAst componentAst1 = createMockedComponentAst(mapEntryComponentIdentifier, mapEntryParameterizedModel,
                                                          asList(createFixedValueComponentParameterAst(defaultGroupModel,
                                                                                                       KEY_ATTRIBUTE_NAME,
                                                                                                       "e1"),
                                                                 createFixedValueComponentParameterAst(defaultGroupModel,
                                                                                                       VALUE_ATTRIBUTE_NAME,
                                                                                                       value1)));
    // 2nd value
    ComponentAst componentAst2 = createMockedComponentAst(mapEntryComponentIdentifier, mapEntryParameterizedModel,
                                                          asList(createFixedValueComponentParameterAst(defaultGroupModel,
                                                                                                       KEY_ATTRIBUTE_NAME,
                                                                                                       "e2"),
                                                                 createFixedValueComponentParameterAst(defaultGroupModel,
                                                                                                       VALUE_ATTRIBUTE_NAME,
                                                                                                       value2)));

    // Create component AST
    ComponentIdentifier componentIdentifier = mock(ComponentIdentifier.class);
    ParameterizedModel componentParameterizedModel = mock(ParameterizedModel.class);
    when(mapParameterAst.getValue()).thenReturn(right(asList(componentAst1, componentAst2)));
    ComponentAst componentAst = createMockedComponentAst(componentIdentifier, componentParameterizedModel,
                                                         asList(mapParameterAst));

    ComponentParameterization<ParameterizedModel> componentParameterization =
        createComponentParameterizationFromComponentAst(componentAst);

    assertThat(componentParameterization.getModel(), is(componentParameterizedModel));
    assertThat(componentParameterization.getParameters(), aMapWithSize(1));
    assertThat(componentParameterization.getParameter(mapValueGroupName, "mapParam"), instanceOf(Map.class));
    assertThat(componentParameterization.getComponentIdentifier().isPresent(), is(true));
    assertThat(componentParameterization.getComponentIdentifier().get(), is(componentIdentifier));
    return componentParameterization;
  }

  @Features({@Feature(OPERATION_EXECUTION_FEATURE), @Feature(DIET_TOOLING)})
  @Stories({@Story(EXECUTE_OPERATION), @Story(TEST_CONNECTIVITY), @Story(METADATA_TYPES_RESOLUTION), @Story(VALUE_PROVIDER),
      @Story(SAMPLE_DATA)})
  @Test(expected = IllegalArgumentException.class)
  public void componentAstWithEmptyModelMustFail() {
    ComponentAst componentAst = mock(ComponentAst.class);
    when(componentAst.getModel(any())).thenReturn(Optional.empty());
    createComponentParameterizationFromComponentAst(componentAst);
  }

  private ComponentAst createMockedComponentAst(ComponentIdentifier identifier, ParameterizedModel parameterizedModel,
                                                List<ComponentParameterAst> componentParameterAstList) {
    ComponentAst componentAst = mock(ComponentAst.class);
    when(componentAst.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    List<ParameterGroupModel> parameterGroupModels =
        componentParameterAstList.stream().map(ComponentParameterAst::getGroupModel).collect(toList());
    when(parameterizedModel.getParameterGroupModels()).thenReturn(parameterGroupModels);
    when(componentAst.getParameters()).thenReturn(componentParameterAstList);
    when(componentAst.getIdentifier()).thenReturn(identifier);

    for (ComponentParameterAst componentParameter : componentParameterAstList) {
      when(componentAst.getParameter(componentParameter.getGroupModel().getName(), componentParameter.getModel().getName()))
          .thenReturn(componentParameter);
    }

    return componentAst;
  }

  private ComponentParameterAst createFixedValueComponentParameterAst(ParameterGroupModel parameterGroupModel,
                                                                      String parameterModelName,
                                                                      Object value) {
    ComponentParameterAst componentParameterAst = createMockedComponentParameterAst(parameterGroupModel, parameterModelName);
    when(componentParameterAst.getValue()).thenReturn(right(value));

    return componentParameterAst;
  }

  private ComponentParameterAst createExpressionMockedComponentParameterAst(ParameterGroupModel parameterGroupModel,
                                                                            String parameterModelName, String value) {
    ComponentParameterAst componentParameterAst = createMockedComponentParameterAst(parameterGroupModel, parameterModelName);
    when(componentParameterAst.getValue()).thenReturn(left(value));

    return componentParameterAst;
  }

  private ComponentParameterAst createMockedComponentParameterAst(ParameterGroupModel parameterGroupModel,
                                                                  String parameterModelName) {
    ComponentParameterAst componentParameterAst = mock(ComponentParameterAst.class);
    ParameterModel parameterModel = mock(ParameterModel.class);

    when(componentParameterAst.getGroupModel()).thenReturn(parameterGroupModel);
    when(parameterGroupModel.getParameter(parameterModelName)).thenReturn(of(parameterModel));
    when(componentParameterAst.getModel()).thenReturn(parameterModel);
    when(parameterModel.getName()).thenReturn(parameterModelName);

    return componentParameterAst;
  }

  private ComponentParameterAst createSimpleArrayComponentParameterAst(ParameterGroupModel parameterGroupModel,
                                                                       String parameterModelName) {
    return createArrayComponentParameterAst(parameterGroupModel, parameterModelName, this::createSimpleType);
  }

  private ComponentParameterAst createComplexArrayComponentParameterAst(ParameterGroupModel parameterGroupModel,
                                                                        String parameterModelName) {
    return createArrayComponentParameterAst(parameterGroupModel, parameterModelName, this::createComplexType);
  }

  private ComponentParameterAst createArrayComponentParameterAst(ParameterGroupModel parameterGroupModel,
                                                                 String parameterModelName, final Supplier<MetadataType> type) {
    ComponentParameterAst componentParameterAst = mock(ComponentParameterAst.class);
    ParameterModel parameterModel = mock(ParameterModel.class);

    when(componentParameterAst.getGroupModel()).thenReturn(parameterGroupModel);
    when(parameterGroupModel.getParameter(parameterModelName)).thenReturn(of(parameterModel));
    when(componentParameterAst.getModel()).thenReturn(parameterModel);
    when(parameterModel.getName()).thenReturn(parameterModelName);
    MetadataType arrayType =
        new DefaultArrayType(type, MetadataFormat.JAVA, emptyMap());
    when(parameterModel.getType()).thenReturn(arrayType);

    return componentParameterAst;
  }

  private ComponentParameterAst createSimpleMapComponentParameterAst(ParameterGroupModel parameterGroupModel,
                                                                     String parameterModelName) {
    return createMapComponentParameterAst(parameterGroupModel, parameterModelName, createSimpleType());
  }

  private ComponentParameterAst createComplexMapComponentParameterAst(ParameterGroupModel parameterGroupModel,
                                                                      String parameterModelName) {
    return createMapComponentParameterAst(parameterGroupModel, parameterModelName, createComplexType());
  }

  private ComponentParameterAst createMapComponentParameterAst(ParameterGroupModel parameterGroupModel, String parameterModelName,
                                                               final MetadataType type) {
    ComponentParameterAst componentParameterAst = mock(ComponentParameterAst.class);
    ParameterModel parameterModel = mock(ParameterModel.class);

    when(componentParameterAst.getGroupModel()).thenReturn(parameterGroupModel);
    when(parameterGroupModel.getParameter(parameterModelName)).thenReturn(of(parameterModel));
    when(componentParameterAst.getModel()).thenReturn(parameterModel);
    when(parameterModel.getName()).thenReturn(parameterModelName);
    MetadataType mapType = new ObjectTypeBuilder(MetadataFormat.JAVA)
        .with(new ClassInformationAnnotation(Map.class))
        .openWith(type)
        .build();
    when(parameterModel.getType()).thenReturn(mapType);

    return componentParameterAst;
  }

  private MetadataType createSimpleType() {
    return new DefaultStringType(MetadataFormat.JAVA, emptyMap());
  }

  private MetadataType createComplexType() {
    return new ObjectTypeBuilder(MetadataFormat.JAVA)
        .with(new ClassInformationAnnotation(Object.class))
        .build();
  }
}
